/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.chat;

import java.util.List;


/**
 * Chat Service interface.
 * This service interface defines a way to interact with Chat Service such as
 * Jabber, ICQ, Yahoo Messenger...
 * Providers must exposes three service properties:
 * <ul>
 * <li><tt>org.ow2.chameleon.chat.user</tt>: the connected user</li>
 * <li><tt>org.ow2.chameleon.chat.service</tt>: the connected service</li>
 * <li><tt>org.ow2.chameleon.chat.connected</tt>: the connection status (boolean)</li>
 * </ul>
 * @author <a href="mailto:chameleon-dev@ow2.org">Chameleon Project Team</a>
 * @version 1.0.0
 */
public interface ChatService {

    /**
     * Property exposed by provider indicating the connected user account.
     */
    public static final String USER_PROPERTY = "org.ow2.chameleon.chat.user";

    /**
     * Property exposed by provider indicating the used service.
     */
    public static final String SERVICE_PROPERTY = "org.ow2.chameleon.chat.service";

    /**
     * Property exposed by provider indicating the connection status.
     */
    public static final String CONNECTED_PROPERTY = "org.ow2.chameleon.chat.connected";

    /**
     * Presence values.
     */
    enum Presence {
        ONLINE,
        OFFLINE,
        AWAY
    }

    /**
     * Sets the presence of the currently connected user.
     * Implementation may choose different way to map the presence
     * on their specific values. Moreover, message status may not be
     * supported by some protocols.
     * @param presence the Presence value
     * @param status the status message
     */
    public void setPresence(Presence presence, String status);

    /**
     * Gets the list of online contacts.
     * The detection of online user is implementation
     * dependent.
     * @return the list of online users.
     */
    public List<String> getOnlineContactList();

    /**
     * Gets the list of contacts (offline and online).
     * Modifying this list does not add / remove contacts.
     * @return the list of contacts.
     */
    public List<String> getContactList();

    /**
     * Adds a contact listener.
     * Contact listeners are notified when contacts status change.
     * @param listener the lsitener to add
     */
    public void addContactListener(ContactListener listener);

    /**
     * Removes a contact listener.
     * @param listener the listener to remove
     */
    public void removeContactListener(ContactListener listener);

    /**
     * Opens a new or gets the current discussion with the
     * given contact.
     * @param contact the contact
     * @return the discussion
     */
    public Discussion getDiscussion(String contact);

    /**
     * Adds a discussion listener.
     * Discussion Listener are notified when a new conversation
     * is created.
     * @param listener the listener
     */
    public void addDiscussionListener(DiscussionListener listener);

    /**
     * Removes a discussion listener
     * @param listener the listener
     */
    public void removeDiscussionListener(DiscussionListener listener);

}
