/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.messaging.jabber.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Property;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.jivesoftware.smack.Chat;
import org.jivesoftware.smack.ChatManagerListener;
import org.jivesoftware.smack.ConnectionConfiguration;
import org.jivesoftware.smack.ConnectionListener;
import org.jivesoftware.smack.MessageListener;
import org.jivesoftware.smack.Roster;
import org.jivesoftware.smack.RosterEntry;
import org.jivesoftware.smack.RosterListener;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.XMPPException;
import org.jivesoftware.smack.packet.Message;
import org.jivesoftware.smack.packet.Presence.Mode;
import org.jivesoftware.smack.packet.Presence.Type;
import org.ow2.chameleon.chat.ChatService;
import org.ow2.chameleon.chat.ContactListener;
import org.ow2.chameleon.chat.Discussion;
import org.ow2.chameleon.chat.DiscussionListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Jabber Connector.
 * Provides a chat service implementation based on Jabber for a specific
 * user.
 * To configure the instance, you must give:
 * <ul>
 * <li>jabber.host: the host</li>
 * <li>jabber.port: the port (optional, default:5222)</li>
 * <li>jabber.user: the jabber user</li>
 * <li>jabber.password: the password</li>
 * <li>jabber.service: the jabber service name</li>
 * <li>jabber.invitation: the invitation policy among manual, accept_all, reject_all.
 * (optional, default:manual)</li>
 * </ul>
 *
 * The exposed service published the connected user (<tt>org.ow2.chameleon.chat.user</tt>), the
 * service used (<tt>org.ow2.chameleon.chat.service</tt>) and the status (<tt>org.ow2.chameleon.chat.connected</tt>). This last
 * property is important to check if the connection is valid of not. In case of reconnection
 * this property is set to <code>false</code> until the connection is restored.
 *
 * The factory name is <code>org.ow2.chameleon.chat.jabber</code>.
 *
 * @author <a href="mailto:chameleon-dev@ow2.org">Chameleon Project Team</a>
 * @version 1.0.0
 */
@Component(name="org.ow2.chameleon.chat.jabber")
@Provides(specifications={ChatService.class})
public class JabberConnector
    implements ConnectionListener, ChatService, MessageListener, ChatManagerListener, RosterListener {

   // private static final String WKSTA = "%WKSTA%";

    /**
     * Default retry period.
     */
    private static final int RETRY_LOGIN_TIMEOUT = 60 * 1000;

    /**
     * The connection object.
     */
    private XMPPConnection m_connection;

    /**
     * The logger.
     */
    private Logger m_logger;

    /**
     * Service property tracking the connection state.
     */
    @ServiceProperty(name=ChatService.CONNECTED_PROPERTY, value="false")
    private volatile boolean m_loggedIn;

    /**
     * Status variable indicating a connecting process.
     */
    private volatile boolean m_loggingIn;

    /**
     * The connection thread.
     */
    private Thread m_connectionThread;

    // Configuration.

    /**
     * The jabber server.
     */
    @Property(name="jabber.host", mandatory=true)
    private String m_host;
    /**
     * The jabber port (default 5222)
     */
    @Property(name="jabber.port", value="5222")
    private int m_port;

    /**
     * The jabber user.
     */
    @ServiceProperty(name=ChatService.USER_PROPERTY)
    private String m_user;

    /**
     * The jabber user password.
     */
    @Property(name="jabber.password", mandatory=true)
    private String m_password;

    /**
     * The jabber service name.
     */
    @ServiceProperty(name=ChatService.SERVICE_PROPERTY)
    private String m_serviceName;

    /**
     * The invitation policy (property).
     */
    @Property(name="jabber.invitation")
    private String m_invitationPolicy;


    /**
     * The invitation policy.
     */
    private Roster.SubscriptionMode m_invitation;

    /**
     * Reconnection flag.
     */
    private boolean m_doReconnect;

    /**
     * The active discussions.
     */
    private HashMap<String, DiscussionImpl> m_activeDiscussions = new HashMap<String, DiscussionImpl>();
    /**
     * The list of contact listeners.
     */
    private List<ContactListener> m_contactListeners = new ArrayList<ContactListener>();
    /**
     * The list of discussion listeners.
     */
    private List<DiscussionListener> m_discussionListeners = new ArrayList<DiscussionListener>();


    /**
     * Creates a JabberConnector.
     */
    public JabberConnector() {
        m_logger = LoggerFactory.getLogger( getClass().getName() );
        m_logger.info( "created " + getClass().getName() );

        if (m_invitationPolicy  == null) {
            m_invitation = Roster.SubscriptionMode.manual;
        } else {
            if (m_invitationPolicy.equalsIgnoreCase("accept_all")) {
                m_invitation = Roster.SubscriptionMode.accept_all;
            } else {
                m_invitation = Roster.SubscriptionMode.reject_all;
            }
        }
    }

    /**
     * Creates a jabber connector.
     * @param server the server
     * @param port the port
     * @param service the service
     * @param un the user name
     * @param pwd the password
     * @param invitation the invitation policy
     */
    public JabberConnector(String server, int port, String service, String un, String pwd, String invitation) {
        this();
        m_host = server;
        m_port = port;
        m_user = un;
        m_password = pwd;
        m_serviceName = service;

        if (invitation  == null) {
            m_invitation = Roster.SubscriptionMode.manual;
        } else {
            if (invitation.equalsIgnoreCase("accept_all")) {
                m_invitation = Roster.SubscriptionMode.accept_all;
            } else {
                m_invitation = Roster.SubscriptionMode.reject_all;
            }
        }
    }

    /**
     * Login to the jabber server.
     */
    public void login() {

        if( isLoggedIn() ) {
            m_logger.error( "Already logged in : " + m_user );
            return;
        }

        if( isLogginInProgress() ) {
            m_logger.error( "Already connecting ... " );
            return;
        }

        // Create the connection job
        Runnable runnable = new Runnable() {

            public void run() {
                try {
                    while( !isLoggedIn() && m_loggingIn ) {
                        try {
                            _login();
                            m_connection.getRoster().setSubscriptionMode(m_invitation);
                            m_connection.getRoster().addRosterListener(JabberConnector.this);

                            // if no exception it is considered as logged in
                            m_loggedIn = true;
                            m_logger.info( "logged in on '" + m_host
                                          + "' on port " + m_port + " with username '"
                                          + m_user + "'"
                            );

                        }
                        catch( Exception e ) {
                            e.printStackTrace();
                            if( m_connection != null ) {
                                m_connection.disconnect();
                                m_connection = null;
                            }
                            m_logger.error( "Can't login, error code: " + e.getMessage());
                            waitBeforeRetry();
                        }
                    }
                }
                finally {
                    m_loggingIn = false;
                }
            }



        };
        m_loggingIn = true;
        m_connectionThread = new Thread( runnable );
        m_connectionThread.start();
    }

    /**
     * Logout from jabber server.
     */
    public void logout() {
        // check if we are trying to login
        while( m_connectionThread != null && m_loggingIn ) {
            // stop the login thread
            m_loggingIn = false;
            try {
                m_connectionThread.join( 10000 );
            } catch( InterruptedException e ) {
                m_logger.error("Tried to stop login thread, but got interrupted", e);
            }
        }
        _logout();
    }

    /**
     * @return <code>true</code> if the service has logged in on the server
     */
    public boolean isLoggedIn() {
        return m_loggedIn;
    }

    /**
     * @return <code>true</code> if login is in progress
     */
    public boolean isLogginInProgress() {
        return m_loggingIn;
    }

    /**
     * Process a message.
     * @param chat the chat
     * @param message the message
     * @see org.jivesoftware.smack.MessageListener#processMessage(org.jivesoftware.smack.Chat, org.jivesoftware.smack.packet.Message)
     */
    public synchronized void processMessage( Chat chat, Message message ) {
        String body = message.getBody();
        if (body == null  || message.getType() != Message.Type.chat) {
            return;
        }
        Collection<String> names = message.getPropertyNames();

        Map<String, Object> properties = new HashMap<String, Object>();
        for (String k : names) {
            properties.put(k, message.getProperty(k));
        }

        String from = message.getFrom();

        m_logger.info( this + "-> got message '" + body + "' from " + from + " (" + message.getPacketID() + ")");
        DiscussionImpl disc = m_activeDiscussions.get( chat.getParticipant() );
        if( disc == null ) {
            DiscussionImpl newDiscussion = new DiscussionImpl(this, m_connection, from, chat);
            m_activeDiscussions.put( from, newDiscussion );
            m_logger.info( "Created new discussion for " + from );
        } else {
            // Inject the new chat
            disc.setChat(chat);
        }

        disc.dispatchMessage(body, properties);
    }

    /**
     * the connection was closed.
     * @see org.jivesoftware.smack.ConnectionListener#connectionClosed()
     */
    public void connectionClosed() {
        m_logger.info( "connectionClosed" );
        logout();

        if( m_doReconnect ) {
            // try to login again
            waitBeforeRetry();
            login();
        }
    }

    /**
     * The connection was closed after an error.
     * @param exception the error
     * @see org.jivesoftware.smack.ConnectionListener#connectionClosedOnError(java.lang.Exception)
     */
    public void connectionClosedOnError( Exception exception ) {
        m_logger.error( "connectionClosedOnError" + exception );
        logout();

        if( m_doReconnect ) {
            // try to login again
            waitBeforeRetry();
            login();
        }
    }

    /**
     * Simply wait before trying to reconnect.
     */
    private void waitBeforeRetry()  {
        m_logger.info( "waitBeforeRetry " + RETRY_LOGIN_TIMEOUT );
        try {
            Thread.sleep( RETRY_LOGIN_TIMEOUT );
        } catch( InterruptedException e1 ) { }
    }

    /**
     * Internal logout.
     */
    private synchronized void _logout() {
        m_logger.info( "logout" );
        if( !isLoggedIn() ) {
            m_logger.info( "tried to logout but was already logged out" );
            return;
        }

        Type type = Type.unsubscribed;
        org.jivesoftware.smack.packet.Presence presence = new org.jivesoftware.smack.packet.Presence( type );
        presence.setMode( Mode.away );
        if( m_connection.isConnected() ) {
            m_connection.sendPacket( presence );
            m_connection.removeConnectionListener( this );
            m_connection.disconnect();
        }
        m_activeDiscussions.clear();

        m_connection = null;
        m_loggedIn = false;
    }

    /**
     * Internal login.
     * @throws XMPPException the login failed.
     */
    protected synchronized void _login()
        throws XMPPException {
        // this method is called from with in a thread

        if( m_loggedIn ) {
            m_logger.error( "Tried to login but was already logged in" );
            return;
        }
        m_logger.info("Try to connect to " + m_host + ":" + m_port + " - " + m_serviceName + " (" + m_user + ")");
        ConnectionConfiguration config = new ConnectionConfiguration( m_host,
                                                                      m_port, m_serviceName
        );
        config.setSelfSignedCertificateEnabled( true );
        config.setSASLAuthenticationEnabled( true );

        m_connection = new XMPPConnection( config );
        m_connection.connect();

        // this method will block
        m_connection.login( m_user, m_password );
        m_connection.getChatManager().addChatListener( this );

        org.jivesoftware.smack.packet.Presence presence = new org.jivesoftware.smack.packet.Presence(
                org.jivesoftware.smack.packet.Presence.Type.available );

        m_connection.sendPacket( presence );

        m_connection.addConnectionListener( this );
    }

    /**
     * Stops the connector.
     */
    @Invalidate
    public void stop() {
        m_logger.info("Shutdown Jabber Connection");
        m_doReconnect = false;
        if (m_connection != null) {
            logout();
        }
    }

    /**
     * Starts the connector.
     */
    @Validate
    public void start() {
        m_doReconnect = true;
        login();
    }

    /**
     * Reconnection...
     * @param arg0 an integer
     * @see org.jivesoftware.smack.ConnectionListener#reconnectingIn(int)
     */
    public void reconnectingIn( int arg0 ) {
        m_logger.info("Start Reconnecting");
    }

    /**
     * Reconnection failed.
     * @param arg0 the error
     * @see org.jivesoftware.smack.ConnectionListener#reconnectionFailed(java.lang.Exception)
     */
    public void reconnectionFailed( Exception arg0 ) {
        m_logger.error("Reconnection failed", arg0);
    }

    /**
     * Reconnected.
     * @see org.jivesoftware.smack.ConnectionListener#reconnectionSuccessful()
     */
    public void reconnectionSuccessful() {
        m_logger.info("Reconnected");
    }

    /**
     * A chat was created.
     * @param chat the created chat
     * @param arg1 local or remote
     * @see org.jivesoftware.smack.ChatManagerListener#chatCreated(org.jivesoftware.smack.Chat, boolean)
     */
    public synchronized void chatCreated( Chat chat, boolean arg1 ) {
        m_logger.info(this + " - chat created: " + chat.getParticipant() );

        if ( !m_activeDiscussions.containsKey(chat.getParticipant())) {
            DiscussionImpl newDiscussion = new DiscussionImpl(this, m_connection, chat.getParticipant(), chat);
            m_activeDiscussions.put( chat.getParticipant(), newDiscussion );
            m_logger.info(this + " - Notify connection listener " + m_discussionListeners.size());
            for (DiscussionListener listener : m_discussionListeners) {
                List<String> participants = new ArrayList<String>();
                participants.add(chat.getParticipant());
                listener.onNewDiscussion(newDiscussion,  participants);
            }
        }

        if (chat.getListeners() != null  && ! chat.getListeners().contains(this)) {
                chat.addMessageListener(this);
        }


    }

    /**
     * Adds a contact listeners.
     * @param listener the listener
     * @see org.ow2.chameleon.chat.ChatService#addContactListener(org.ow2.chameleon.chat.ContactListener)
     */
    public synchronized void addContactListener(ContactListener listener) {
        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }
        m_contactListeners.add(listener);
    }

    /**
     * Gets the contact list.
     * @return the contact list
     * @see org.ow2.chameleon.chat.ChatService#getContactList()
     */
    public List<String> getContactList() {
        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }

        List<String> contacts = new ArrayList<String>();
        Collection<RosterEntry> entries = m_connection.getRoster().getEntries();
        if (entries != null) {
            for (RosterEntry contact : entries) {
                contacts.add(contact.getUser());
            }
        }
        return contacts;
    }

    /**
     * Gets a discussion with the given user.
     * @param contact the contact
     * @return the discussion
     * @see org.ow2.chameleon.chat.ChatService#getDiscussion(java.lang.String)
     */
    public Discussion getDiscussion(String contact) {
        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }

        if (m_activeDiscussions.containsKey(contact)) {
            return m_activeDiscussions.get(contact);
        } else {
            DiscussionImpl newDiscussion = new DiscussionImpl(this, m_connection, contact);
            m_activeDiscussions.put( contact, newDiscussion );
            m_logger.info("Create conversation for " + contact);
            return newDiscussion;
        }
    }

    /**
     * Gets the list of online users.
     * @return the online users list.
     * @see org.ow2.chameleon.chat.ChatService#getOnlineContactList()
     */
    public List<String> getOnlineContactList() {
        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }

        List<String> list = getContactList();
        List<String> online = new ArrayList<String>();
        for (String user : list) {
            if (m_connection.getRoster().getPresence(user).getType() == Type.available) {
                online.add(user);
            }
        }
        return online;
    }

    /**
     * Removes a contact listener.
     * @param listener the listener
     * @see org.ow2.chameleon.chat.ChatService#removeContactListener(org.ow2.chameleon.chat.ContactListener)
     */
    public synchronized void removeContactListener(ContactListener listener) {
        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }

        m_contactListeners.remove(listener);
    }

    /**
     * Sets the presence of the connected user.
     * @param newPresence the new presence
     * @param status the status message
     * @see org.ow2.chameleon.chat.ChatService#setPresence(org.ow2.chameleon.chat.ChatService.Presence, java.lang.String)
     */
    public void setPresence(
            ChatService.Presence newPresence,
            String status) {

        if (m_connection == null) {
            throw new IllegalStateException("Not connected");
        }

        Mode mode = org.jivesoftware.smack.packet.Presence.Mode.available;
        Type type = Type.available;
        switch (newPresence) {
            case AWAY:
                mode = org.jivesoftware.smack.packet.Presence.Mode.away;
                type = Type.available;
                break;
            case OFFLINE:
                mode = org.jivesoftware.smack.packet.Presence.Mode.xa;
                type = Type.unavailable;
            case ONLINE:
                mode = org.jivesoftware.smack.packet.Presence.Mode.available;
                type = Type.available;
            default:
                break;
        }

        org.jivesoftware.smack.packet.Presence presence =
            new org.jivesoftware.smack.packet.Presence(type, status, 0, mode);
        m_connection.sendPacket( presence );
    }

    public void entriesAdded(Collection<String> entries) {
        // Do nothing.
    }

    public void entriesDeleted(Collection<String> arg0) {
        // Do nothing.
    }

    public void entriesUpdated(Collection<String> arg0) {
        // Do nothing.
    }

    /**
     * The presence of a contact changed.
     * @param presence the new presence
     * @see org.jivesoftware.smack.RosterListener#presenceChanged(org.jivesoftware.smack.packet.Presence)
     */
    public void presenceChanged(org.jivesoftware.smack.packet.Presence presence) {
        List<ContactListener> listeners = null;
        synchronized (this) {
            listeners = new ArrayList<ContactListener>(m_contactListeners);
        }
        String user = presence.getFrom();
        org.jivesoftware.smack.packet.Presence bestPresence = m_connection.getRoster().getPresence(user);
        Mode mode = bestPresence.getMode();
        Presence pres = Presence.OFFLINE;
        switch (bestPresence.getType()) {
            case available:
                pres = Presence.ONLINE;
                if (mode != null) {
                    switch (mode) {
                        case available: pres = Presence.ONLINE; break;
                        case away: pres = Presence.AWAY; break;
                        case chat: pres = Presence.ONLINE; break;
                        case dnd: pres = Presence.AWAY; break;
                        case xa: pres = Presence.AWAY; break;
                    }
                }
                break;
            case unavailable:
                pres = Presence.OFFLINE;
                break;
            default:
                break;
        }

        for (ContactListener listener : listeners) {
            listener.onContactStatusChanged(user, pres, bestPresence.getStatus());
        }

    }

    /**
     * Adds a discussion listener.
     * @param listener the discussion listener
     * @see org.ow2.chameleon.chat.ChatService#addDiscussionListener(org.ow2.chameleon.chat.DiscussionListener)
     */
    public void addDiscussionListener(DiscussionListener listener) {
        List<Discussion> list = new ArrayList<Discussion>();
        synchronized (this) {
            m_logger.info(this + " - New connection listener added");
            m_discussionListeners.add(listener);
            list = new ArrayList<Discussion>(m_activeDiscussions.values());
        }

        for (Discussion conv : list) {
            listener.onNewDiscussion(conv, conv.getParticipants());
        }
    }

    /**
     * Removes a discussion listener.
     * @param listener the listener
     * @see org.ow2.chameleon.chat.ChatService#removeDiscussionListener(org.ow2.chameleon.chat.DiscussionListener)
     */
    public synchronized void removeDiscussionListener(DiscussionListener listener) {
        m_discussionListeners.remove(listener);
    }

    /**
     * Closes a discussion.
     * @param contact the contact
     * @param conv the conversation
     */
    public synchronized void closeDiscussion(String contact,
            DiscussionImpl conv) {
        m_activeDiscussions.remove(contact);
    }

    @Property(name="jabber.service", mandatory=true)
    public void setService(String service) {
        m_serviceName = service;
    }

    @Property(name="jabber.user", mandatory=true)
    public void setUSer(String user) {
        m_user = user;
    }

}
