/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.messaging.jabber.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.jivesoftware.smack.Chat;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.packet.Message;
import org.ow2.chameleon.chat.Discussion;
import org.ow2.chameleon.chat.MessageReceiver;

/**
 * Discussion Implementation for Jabber.
 * These objects map Jabber chat and allows to receive and
 * send messages.
 * This implementation does not support multi-contact rooms,
 * and supports only one Message Receiver.
 * @author <a href="mailto:chameleon-dev@ow2.org">Chameleon Project Team</a>
 * @version 1.0.0
 */
public class DiscussionImpl implements Discussion {

    /**
     * The Jabber connector.
     */
    private JabberConnector m_connector;
    /**
     * The XMPP Connection.
     */
    private XMPPConnection m_connection;

    /**
     * The contact.
     */
    private String m_contact;
    /**
     * The Jabber chat.
     */
    private Chat m_chat;

    /**
     * The Message Receiver.
     */
    private MessageReceiver m_receiver;

    /**
     * Creates a Discussion. This methods
     * creates the chat.
     * @param jabberConnector the connector
     * @param connection the connection
     * @param contact the contact
     */
    public DiscussionImpl(JabberConnector jabberConnector,
            XMPPConnection connection, String contact) {
        m_connector = jabberConnector;
        m_connection = connection;
        m_contact = contact;

        m_chat = m_connection.getChatManager().createChat(m_contact, m_connector);
    }

    /**
     * Creates a Discussion but also injects the Chat.
     * @param jabberConnector the connector
     * @param connection the connection
     * @param contact the contact
     * @param chat the jabber chat
     */
    public DiscussionImpl(JabberConnector jabberConnector,
            XMPPConnection connection, String contact, Chat chat) {
        m_connector = jabberConnector;
        m_connection = connection;
        m_contact = contact;

        m_chat = chat;
    }

    /**
     * Closes the discussion.
     * @see org.ow2.chameleon.chat.Discussion#close()
     */
    public synchronized void close() {
        m_receiver = null;
        m_connector.closeDiscussion(m_contact, this);
    }

    /**
     * Gets the participant of the chat without the current user.
     * This returns only the other participants as multi-contacts
     * chat are not supported.
     * @see org.ow2.chameleon.chat.Discussion#getParticipants()
     */
    public synchronized List<String> getParticipants() {
        List<String> list = new ArrayList<String>();
        list.add(m_chat.getParticipant());
        return list;
    }

    /**
     * Registers the message receiver.
     * @see org.ow2.chameleon.chat.Discussion#registerMessageReceiver(org.ow2.chameleon.chat.MessageReceiver)
     */
    public synchronized void registerMessageReceiver(MessageReceiver receiver) {
        m_receiver = receiver;
    }

    /**
     * Sends a message to the chat.
     * @param message the message
     * @throws Exception if the method cannot be sent
     * @see org.ow2.chameleon.chat.Discussion#sendMessage(java.lang.String)
     */
    public synchronized void sendMessage(String message) throws Exception {
        if (m_chat != null) {
            m_chat.sendMessage(message);
        }
    }

    /**
     * Sends a message to the chat.
     * @param body the message
     * @param props the message properties
     * @throws Exception if the method cannot be sent
     * @see org.ow2.chameleon.chat.Discussion#sendMessage(java.lang.String, java.util.Map)
     */
    public synchronized void sendMessage(String body, Map<String, ?> props) throws Exception {
        Message message = new Message();
        message.setBody(body);
        if (props != null) {
            for (String k : props.keySet()) {
                message.setProperty(k, props.get(k));
            }
        }
        if (m_chat != null) {
            m_chat.sendMessage(message);
        }
    }

    /**
     * Unregisters the message receiver.
     * @param the message receiver
     * @see org.ow2.chameleon.chat.Discussion#unregisterMessageReceiver(org.ow2.chameleon.chat.MessageReceiver)
     */
    public  synchronized void unregisterMessageReceiver(MessageReceiver receiver) {
        m_receiver = null;
    }

    /**
     * Dispatches a message to the message listener.
     * @param body the message
     * @param properties the properties
     */
    protected synchronized void dispatchMessage(String body, Map<String, Object> properties) {
        if (m_receiver != null) {
            m_receiver.onReceivedMessage(this, m_contact, body, properties);
        }
    }

    /**
     * Sets the current wrapped chat.
     * @param chat the chat
     */
    public synchronized void setChat(Chat chat) {
        m_chat = chat;
    }

}
