/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.messaging.jabber.test;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.Assert;

import org.junit.Test;
import org.ow2.chameleon.chat.Discussion;
import org.ow2.chameleon.chat.DiscussionListener;
import org.ow2.chameleon.chat.MessageReceiver;
import org.ow2.chameleon.messaging.jabber.impl.JabberConnector;



public class ConnectionTest {

    @Test
    public void testConnection() throws Exception {
        JabberConnector connector = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test@googlemail.com", "chameleon", null);
        connector.start();

        waitForLoggedIn(connector);

        Assert.assertTrue(connector.isLoggedIn());

        connector.stop();
    }

    @Test
    public void testContactLists() throws Exception {
        JabberConnector connector = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test@googlemail.com", "chameleon", null);
        connector.start();
        waitForLoggedIn(connector);

        Assert.assertTrue(connector.isLoggedIn());


        List<String> contacts = connector.getContactList();
        boolean found = false;
        for (String c : contacts) {
            if (c.contains("ow2.chameleon.test.2")) {
                found = true;
            }
        }
        Assert.assertTrue(found);


        contacts = connector.getOnlineContactList();
        found = false;
        for (String c : contacts) {
            if (c.contains("ow2.chameleon.test.2")) {
                found = true;
            }
        }
        Assert.assertFalse(found);

        JabberConnector connector2 = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test.2@googlemail.com", "chameleon", null);
        connector2.start();
        waitForLoggedIn(connector2);

        waitALittleBit();


        contacts = connector.getOnlineContactList();
        found = false;
        for (String c : contacts) {
            if (c.contains("ow2.chameleon.test.2")) {
                found = true;
            }
        }
        Assert.assertTrue(found);

        connector.stop();
        connector2.stop();
    }

//    @Test
//    public void testConversationWithOfflineUser() throws Exception {
//        JabberConnector connector = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test@googlemail.com", "chameleon", null);
//        connector.start();
//
//        waitForLoggedIn(connector);
//
//        Conversation conv = connector.getConversation("ow2.chameleon.test.2@googlemail.com");
//        conv.registerMessageReceiver(new MessageReceiver() {
//
//            public void onReceivedMessage(Conversation conversation, String from,
//                    String message, Map<String, Object> properties) {
//                Assert.fail("Unexpected message : " + message);
//            }
//        });
//
//        conv.sendMessage("Hello");
//        conv.close();
//        waitALittleBit();
//
//        connector.stop();
//    }

    @Test
    public void testConversationWithOnlineUser() throws Exception {

        JabberConnector connector = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test@googlemail.com", "chameleon", null);
        connector.start();

        JabberConnector connector2 = new JabberConnector("talk.google.com", 5222, "gmail.com", "ow2.chameleon.test.2@googlemail.com", "chameleon", null);
        connector2.start();

        waitForLoggedIn(connector);
        waitForLoggedIn(connector2);

        waitALittleBit();

        final List<String> receivedBy1 = new ArrayList<String>();
        final List<String> receivedBy2 = new ArrayList<String>();

        DiscussionListener discListener = new DiscussionListener() {

            public void onNewDiscussion(Discussion newConversation,
                    List<String> participants) {
                System.out.println("On new discussion...");
                newConversation.registerMessageReceiver(new MessageReceiver() {

                    public void onReceivedMessage(Discussion conversation, String from,
                            String message, Map<String, Object> properties) {
                        receivedBy2.add(message);
                        try {
                            System.out.println("Has received " + message);
                            // Do an echo
                            conversation.sendMessage(message);
                            // Send a second message
                            conversation.sendMessage("ping");
                        } catch (Exception e) {
                            Assert.fail(e.getMessage());
                        }
                    }
                });
        }};

        connector2.addDiscussionListener(discListener);

        Discussion conv = connector.getDiscussion("ow2.chameleon.test.2@googlemail.com");
        MessageReceiver theReceiver = new MessageReceiver() {

            public void onReceivedMessage(Discussion conversation, String from,
                    String message, Map<String, Object> properties) {
                receivedBy1.add(message);
            }
        };
        conv.registerMessageReceiver(theReceiver);


        receivedBy1.clear();
        receivedBy2.clear();

        System.out.println("Send hello");
        conv.sendMessage("Hello");

        waitALittleBit();

        Assert.assertEquals(1,conv.getParticipants().size());
        conv.unregisterMessageReceiver(theReceiver);
        conv.sendMessage("Hello", new HashMap<String, Object>());

        waitALittleBit();


        conv.close();

        // Check
        System.out.println("received by 1 " + receivedBy1);
        System.out.println("received by 2 " + receivedBy2);
        Assert.assertEquals(2, receivedBy1.size()); // The Echo of Hello + another message others are ignored.
        Assert.assertEquals(2, receivedBy2.size());

        connector.stop();
        connector2.stop();
    }

    private void waitForLoggedIn(JabberConnector connector) throws InterruptedException {
        int count = 0;
        while (! connector.isLoggedIn()  && count < 50) {
            count ++;
            Thread.sleep(1000);
        }
        if (count == 10) {
            Assert.fail("Cannot connect");
        }
        System.out.println("Connected ! ");
    }

    private void waitALittleBit() {
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

}
