/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.derby;

import java.util.Properties;

import javax.sql.DataSource;

import org.apache.derby.jdbc.EmbeddedDataSource;
import org.osgi.service.jdbc.DataSourceFactory;


/**
 * Derby Data Source Factory is a component creating a DataSource for an
 * embeddedDerby database. Is is an implementation of the OSGi R4.2 JDBC
 * service.
 * 
 * @author <a href="mailto:chameleon-dev@ow2.org">Chameleon Project Team</a>
 */
public class DerbyDataSourceFactory implements DataSourceFactory
{
    /**
     * Configures and returns a Derby DataSource.
     * 
     * @param props
     *            the DataSource properties
     * @return the configured DataSource
     * @see org.osgi.service.jdbc.DataSourceFactory#
     *      createDataSource(java.util.Properties)
     */
    public DataSource createDataSource(Properties props) {
        String url = (String) props.get(JDBC_URL);
        String user = (String) props.get(JDBC_USER);
        String pwd = (String) props.get(JDBC_PASSWORD);

        if (url == null || url.trim().equals("")) {
            throw new IllegalArgumentException("url was null or blank");
        }

        /*
         * Derby's JDBC URL syntax is:
         * jdbc:derby:[subsubprotocol:][databaseName][;attribute=value]*
         */
        String dataBaseName = url;

        // "jdbc:derby" prefix is constant, we remove that
        dataBaseName = dataBaseName.replace("jdbc:derby:", "");

        // if existant, "subsubprotocol" is kept as prefix, attributes as
        // postfix for databaseName
        EmbeddedDataSource dataSource = new EmbeddedDataSource();
        dataSource.setDatabaseName(dataBaseName);
        dataSource.setUser(user);
        dataSource.setPassword(pwd);

        return dataSource;
    }
}
