/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.i18n.extension;

import java.util.Locale;

import junit.framework.Assert;

import org.junit.Test;
import org.ow2.chameleon.i18n.I18nService;
import org.ow2.chameleon.i18n.I18nServiceSelector;


public class I18NServiceSelectorImplTest {


	public I18nServiceSelector createSelector() {
		return new I18NServiceSelectorImpl(
				new I18nService[] {
						I18NServiceFactory.createMyApp_En(),
						I18NServiceFactory.createMyApp_fr(),
						I18NServiceFactory.createMyApp2_de(),
				}
		);
	}

	public I18nServiceSelector createSelectorWithDegression() {
		return new I18NServiceSelectorImpl(
				new I18nService[] {
						I18NServiceFactory.createMyApp_fr(),
						I18NServiceFactory.createMyApp_fr_FR(),
						I18NServiceFactory.createMyApp_fr_CA(),
						I18NServiceFactory.createMyApp_fr_CA_QE(),
						I18NServiceFactory.createMyApp2_de(),
				}
		);
	}

	@Test
	public void getAllI18nServicesForAGivenResource() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices("myapp");
		Assert.assertEquals(2, svc.length);
	}

	@Test
	public void getAllI18nServicesForAMissingResource() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices("this-does-not-exist");
		Assert.assertEquals(0, svc.length);
	}

	@Test
	public void getAllI18nServicesForAGivenResourceAndLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices("myapp", new Locale("fr"));
		Assert.assertEquals(1, svc.length);
	}

	@Test
	public void getAllI18nServicesForAGivenResourceAndMissingLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices("myapp", new Locale("hu"));
		Assert.assertEquals(0, svc.length);
	}

	@Test
	public void getAllI18nServicesForAMissingResourceAndMissingLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices("this-does-not-exist", new Locale("hu"));
		Assert.assertEquals(0, svc.length);
	}

	@Test
	public void getAllI18nServicesForAGivenResourceAndDefaultLocale() {
		Locale def = Locale.getDefault();
		Locale.setDefault(new Locale("de"));
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServicesForCurrentLocale("myapp");
		Assert.assertEquals(1, svc.length);
		Locale.setDefault(def); // Restore the locale.
	}

	@Test
	public void getAllI18nServicesForAMissingResourceAndDefaultLocale() {
		Locale def = Locale.getDefault();
		Locale.setDefault(new Locale("de"));
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServicesForCurrentLocale("do-not-exit");
		Assert.assertEquals(0, svc.length);
		Locale.setDefault(def); // Restore the locale.
	}

	@Test
	public void getAllI18NServicesForAGivenLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices(new Locale("fr"));
		Assert.assertEquals(1, svc.length);
	}

	@Test
	public void getAllI18NServicesForAMissingLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices(new Locale("hu"));
		Assert.assertEquals(0, svc.length);
	}

	@Test
	public void getAllI18NServicesForTheCurrentLocale() {
		Locale def = Locale.getDefault();
		Locale.setDefault(new Locale("de"));
		I18nServiceSelector selector = createSelector();
		I18nService[] svc = selector.getI18nServices(new Locale("fr"));
		Assert.assertEquals(1, svc.length);
		Locale.setDefault(def); // Restore the locale.
	}

	@Test
	public void getI18nServiceForAGivenResource() {
		I18nServiceSelector selector = createSelector();
		I18nService svc = selector.getI18nService("myapp", new Locale("de"));
		Assert.assertNotNull(svc);
	}

	@Test
	public void getI18nServiceForAGivenResourceAndCurrentLocale() {
		Locale def = Locale.getDefault();
		Locale.setDefault(new Locale("de"));
		I18nServiceSelector selector = createSelector();
		I18nService svc = selector.getI18nServiceForCurrentLocale("myapp");
		Assert.assertNotNull(svc);
		Locale.setDefault(def); // Restore the locale.
	}

	@Test
	public void getI18nServiceForAMissingResource() {
		I18nServiceSelector selector = createSelector();
		I18nService svc = selector.getI18nService("do_not_exist", new Locale("de"));
		Assert.assertNull(svc);
	}

	@Test
	public void getI18nServiceForAGivenResourceButMissingLocale() {
		I18nServiceSelector selector = createSelector();
		I18nService svc = selector.getI18nService("myapp", new Locale("hu"));
		Assert.assertNull(svc);
	}

	@Test
	public void getAllI18nServicesForAGivenResourceIncludingParents() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService[] svc = selector.getI18nServices("myapp", new Locale("fr", "CA"), true);
		Assert.assertEquals(2, svc.length); // fr, fr_CA
	}

	@Test
	public void getAllI18nServicesForAGivenResourceIncludingParentsAndVariant() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService[] svc = selector.getI18nServices("myapp", new Locale("fr", "CA", "QE"), true);
		Assert.assertEquals(3, svc.length); // fr, fr_CA, fr_CA_QE
	}

	@Test
	public void getAllI18nServicesForAGivenResourceIncludingParentsAndMissingVariant() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService[] svc = selector.getI18nServices("myapp", new Locale("fr", "CA", "QB"), true);
		Assert.assertEquals(2, svc.length); // fr, fr_CA
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParents() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nService("myapp", new Locale("fr", "CA"), true);
		Assert.assertNotNull(svc);
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParentsAndVariant() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nService("myapp", new Locale("fr", "CA", "QE"), true);
		Assert.assertNotNull(svc); // fr, fr_CA, fr_CA,QE
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParentsAndVariantForDefaultLocale() {
		Locale def = Locale.getDefault();
		Locale.setDefault(new Locale("fr", "CA", "QE"));
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nServiceForCurrentLocale("myapp", true);
		Assert.assertNotNull(svc); // fr, fr_CA, fr_CA,QE
		Locale.setDefault(def);
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParentsAndVariantWithDegression() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nService("myapp", new Locale("fr", "FR", "toto"), true);
		Assert.assertNotNull(svc); // fr_FR
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParentsAndMissingVariant() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nService("myapp", new Locale("fr", "CA", "QB"), true);
		Assert.assertNotNull(svc); // fr_CA and fr match
	}

	@Test
	public void getI18nServiceForAGivenResourceIncludingParentsAndMissingVariantWithNoMatch() {
		I18nServiceSelector selector = createSelectorWithDegression();
		I18nService svc = selector.getI18nService("myapp", new Locale("en", "UK", "QB"), true);
		Assert.assertNull(svc);
	}

}
