package org.ow2.chameleon.i18n.extension.integration;

import org.junit.*;
import org.junit.runner.RunWith;
import org.ops4j.io.StreamUtils;
import org.ops4j.pax.exam.CoreOptions;
import org.ops4j.pax.exam.Inject;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.OptionUtils;
import org.ops4j.pax.exam.container.def.PaxRunnerOptions;
import org.ops4j.pax.exam.junit.Configuration;
import org.ops4j.pax.exam.junit.JUnit4TestRunner;
import org.ops4j.pax.swissbox.tinybundles.core.TinyBundles;
import org.osgi.framework.*;
import org.ow2.chameleon.i18n.I18nService;
import org.ow2.chameleon.i18n.I18nServiceSelector;
import org.ow2.chameleon.testing.helpers.OSGiHelper;

import java.io.*;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

@RunWith(JUnit4TestRunner.class)
public class I18nExtenderTest {

    /**
     * the name of the system property providing the bundle file to be installed and tested
     */
    protected static final String BUNDLE_JAR_SYS_PROP = "project.bundle.file";

    @Inject
    private BundleContext context;

    private OSGiHelper osgi;

    @Before
    public void setUp() {
        osgi = new OSGiHelper(context);
    }

    @After
    public void tearDown() {
        osgi.dispose();
    }

    @Configuration
    public static Option[] configure() throws IOException {

        Option[] platform = CoreOptions.options(
                CoreOptions.felix());

        URL english = createEnglishExtension();
        URL german = createGermanExtension();
        URL germanXML = createGermanXMLExtension();
        URL germanUTF8 = createGermanUTF8Extension();

        URL french = createFrenchExtension();
        URL french2 = createAnotherFrenchExtension();

        URL english_in_subfolder = createEnglishExtensionInSubFolder();
        URL english_in_subsubfolder = createEnglishExtensionInSubSubFolder();

        URL invalid = createBundleWithInvalidResources();

        // The bundle
        final String bundleFileName = System.getProperty(BUNDLE_JAR_SYS_PROP);
        final File bundleFile = new File(bundleFileName);

        Option[] bundles = CoreOptions.options(
                CoreOptions.provision(
                        CoreOptions.mavenBundle().groupId("org.apache.felix").artifactId("org.apache.felix.ipojo").version("1.8.0"),
                        CoreOptions.mavenBundle().groupId("org.ow2.chameleon.testing").artifactId("osgi-helpers").version("0.4.0"),
                        CoreOptions.bundle(bundleFile.toURI().toString())
                        ),
                CoreOptions.systemProperty("ENGLISH").value(english.toExternalForm()),
                CoreOptions.systemProperty("GERMAN").value(german.toExternalForm()),
                CoreOptions.systemProperty("GERMAN_XML").value(germanXML.toExternalForm()),
                CoreOptions.systemProperty("GERMAN_UTF8").value(germanUTF8.toExternalForm()),
                CoreOptions.systemProperty("FRENCH").value(french.toExternalForm()),
                CoreOptions.systemProperty("FRENCH_2").value(french2.toExternalForm()),
                CoreOptions.systemProperty("ENGLISH_SUBFOLDER").value(english_in_subfolder.toExternalForm()),
                CoreOptions.systemProperty("ENGLISH_SUBSUBFOLDER").value(english_in_subsubfolder.toExternalForm()),
                CoreOptions.systemProperty("INVALID").value(invalid.toExternalForm()),

                CoreOptions.bootClasspathLibraries(new String[]{
                        "mvn:org.slf4j/slf4j-api/1.6.0",
                        "mvn:ch.qos.logback/logback-classic/0.9.21",
                        "mvn:ch.qos.logback/logback-core/0.9.21"
                }
                ),
                CoreOptions.systemPackage("org.slf4j; version=1.6.0"),

                PaxRunnerOptions.repository("http://maven.ow2.org/maven2")); // Add the ow2 repository to download chameleon-commons


        Option[] r = OptionUtils.combine(platform, bundles);

        return r;
    }

    @Test
    public void checkExtensionWithOneResource() throws InterruptedException {
        // Install the English bundle
        osgi.installAndStart(System.getProperty("ENGLISH"));

        System.out.println("===");
        for (Bundle b : context.getBundles()) {
            System.out.println(b.getSymbolicName() + " - " + b.getState());
        }
        System.out.println("===");

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(), null);
        Assert.assertEquals(new Locale("en"), i18n.getLocale());

        String hello = i18n.getString("HELLO");
        Assert.assertEquals("hello", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);
    }

    @Test
    public void checkExtensionWithTwoResources() throws InterruptedException {
        // Install the French bundle
        osgi.installAndStart(System.getProperty("FRENCH"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.LOCALE_PROPERTY + "=fr)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("fr"), i18n.getLocale());

        String hello = i18n.getString("HELLO");
        Assert.assertEquals("Bonjour", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);

        i18n = waitFori18nService(new Locale("fr", "FR"), 5000);
        Assert.assertEquals(new Locale("fr", "FR"), i18n.getLocale());

        hello = i18n.getString("HELLO");
        Assert.assertEquals("Bonjour!", hello);

        resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);
    }

    public I18nService waitFori18nService(Locale locale, long timeout) {
        String itf = I18nService.class.getName();

        if (timeout == 0) {
            timeout = 10000; // Default 10 secondes.
        }
        ServiceReference[] refs = osgi.getServiceReferences(itf, null);
        long begin = System.currentTimeMillis();
        if (refs.length != 0) {
            for (ServiceReference ref : refs) {
                Locale loc = (Locale) ref.getProperty(I18nService.LOCALE_PROPERTY);
                System.out.println("try to match " + locale + " with " + loc);
                if (loc.equals(locale)) {
                    return (I18nService) osgi.getServiceObject(ref);
                }
            }
        }

        refs = new ServiceReference[0];

        while (refs.length == 0) {
            try {
                Thread.sleep(5);
            } catch (InterruptedException e) {
                // Interrupted
            }
            long now = System.currentTimeMillis();

            if ((now - begin) > timeout) {
                Assert.fail("Timeout ... no services matching with the request after " + timeout + "ms");
            }
            refs = osgi.getServiceReferences(itf, null);
            if (refs.length != 0) {
                for (ServiceReference ref : refs) {
                    Locale loc = (Locale) ref.getProperty(I18nService.LOCALE_PROPERTY);
                    System.out.println("try to match " + locale + " with " + loc);
                    if (loc.equals(locale)) {
                        return (I18nService) osgi.getServiceObject(ref);
                    }
                }
            }
            refs = new ServiceReference[0];
        }

        Assert.fail("Service not found");
        return null;
    }

    public I18nService[] getI18NServices(Locale locale) {
        ServiceReference[] refs = osgi.getServiceReferences(I18nService.class.getName(), null);
        if (refs.length == 0) {
            return new I18nService[0];
        } else {
            List<I18nService> l = new ArrayList<I18nService>();
            for (ServiceReference ref : refs) {
                Locale loc = (Locale) ref.getProperty(I18nService.LOCALE_PROPERTY);
                if (locale.equals(loc)) {
                    l.add((I18nService) osgi.getServiceObject(ref));
                }
            }
            return (I18nService[]) l.toArray(new I18nService[l.size()]);
        }
    }

    @Test
    public void checkExtensionWithTwoDifferentResources() throws InterruptedException {
        // Install the German bundle
        osgi.installAndStart(System.getProperty("GERMAN"));

        osgi.waitForService(I18nService.class.getName(), "(" + I18nService.RESOURCE_PROPERTY + "=myapp)", 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.RESOURCE_PROPERTY + "=myapp)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("de"), i18n.getLocale());

        String hello = i18n.getString("HELLO");
        Assert.assertEquals("Hallo", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);

        i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.RESOURCE_PROPERTY + "=myapp2)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("de"), i18n.getLocale());

        hello = i18n.getString("YES");
        Assert.assertEquals("Ja", hello);

        resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp2", resources[0]);
    }

    @Test
    public void checkTwiceTheSameResource() throws InterruptedException {
        // Install the German bundle
        osgi.installAndStart(System.getProperty("FRENCH"));
        osgi.installAndStart(System.getProperty("FRENCH_2"));

        waitFori18nService(new Locale("fr"), 5000);
        waitFori18nService(new Locale("fr", "FR"), 5000);

        Thread.sleep(1000);

        Object[] i18n = getI18NServices(new Locale("fr"));
        Assert.assertNotNull(i18n);
        Assert.assertEquals(2, i18n.length);
    }

    @Test
    @Ignore
    public void checkXMLExtension
            () throws InterruptedException {
        // Install the German bundle
        osgi.installAndStart(System.getProperty("GERMAN_XML"));

        osgi.waitForService(I18nService.class.getName(), "(" + I18nService.RESOURCE_PROPERTY + "=myapp)", 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.RESOURCE_PROPERTY + "=myapp)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("de"), i18n.getLocale());

        String hello = i18n.getString("WELCOME");
        Assert.assertEquals("Gr��e", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);

    }

    @Test
    @Ignore
    public void checkUTF8Extension
            () throws InterruptedException {
        // Install the German bundle
        osgi.installAndStart(System.getProperty("GERMAN_UTF8"));

        osgi.waitForService(I18nService.class.getName(), "(" + I18nService.RESOURCE_PROPERTY + "=myapp)", 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.RESOURCE_PROPERTY + "=myapp)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("de"), i18n.getLocale());

        String hello = i18n.getString("WELCOME");
        Assert.assertEquals("Gr��e", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("myapp", resources[0]);

    }


    @Test
    public void checkExtensionWithOneResourceInSubfolder
            () throws InterruptedException {
        // Install the English bundle
        osgi.installAndStart(System.getProperty("ENGLISH_SUBFOLDER"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(), null);
        Assert.assertEquals(new Locale("en"), i18n.getLocale());

        String hello = i18n.getString("HELLO");
        Assert.assertEquals("hello", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("org.Myapp", resources[0]);
    }

    @Test
    public void checkExtensionWithOneResourceInSubSubfolder
            () throws InterruptedException {
        // Install the English bundle
        osgi.installAndStart(System.getProperty("ENGLISH_SUBSUBFOLDER"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(), null);
        Assert.assertEquals(new Locale("en"), i18n.getLocale());

        String hello = i18n.getString("HELLO");
        Assert.assertEquals("hello", hello);

        String[] resources = i18n.getResources();
        Assert.assertTrue(resources.length == 1);
        Assert.assertEquals("org.ow2.i18n.Myapp", resources[0]);
    }

    @Test
    public void checkExtensionWithInvalidResources
            () throws InterruptedException {
        // Install the English bundle
        osgi.installAndStart(System.getProperty("INVALID"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        Object[] objects = osgi.getServiceObjects(I18nService.class.getName(), null);

        Assert.assertEquals("Number of service (" + objects.length + ")", 1, objects.length);
    }

    @Test
    public void testDeploymentOFOneExtension
            () throws InterruptedException, BundleException {

        ServiceReference ref = osgi.getServiceReference(I18nService.class.getName());
        Assert.assertNull(ref);

        osgi.installAndStart(System.getProperty("ENGLISH"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(), null);
        Assert.assertEquals(new Locale("en"), i18n.getLocale());

        Bundle bundle = osgi.getBundle("i18n.test.en");
        bundle.uninstall();

        Thread.sleep(1000);
        ref = osgi.getServiceReference(I18nService.class.getName());
        Assert.assertNull(ref);
    }

    @Test
    public void testDeploymentOfTwoExtensions
            () throws InterruptedException, BundleException {

        ServiceReference ref = osgi.getServiceReference(I18nService.class.getName());
        Assert.assertNull(ref);

        osgi.installAndStart(System.getProperty("ENGLISH"));

        osgi.waitForService(I18nService.class.getName(), null, 5000);
        I18nService i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(), null);
        Assert.assertEquals(new Locale("en"), i18n.getLocale());

        osgi.installAndStart(System.getProperty("FRENCH"));

        osgi.waitForService(I18nService.class.getName(), "(" + I18nService.LOCALE_PROPERTY + "=fr)", 5000);
        i18n = (I18nService) osgi.getServiceObject(I18nService.class.getName(),
                "(" + I18nService.LOCALE_PROPERTY + "=fr)");
        Assert.assertNotNull(i18n);
        Assert.assertEquals(new Locale("fr"), i18n.getLocale());

        Bundle bundle = osgi.getBundle("i18n.test.en");
        bundle.uninstall();

        Thread.sleep(2000); // Wait propagation.
        ref = osgi.getServiceReference(I18nService.class.getName(), "(" + I18nService.LOCALE_PROPERTY + "=en)");
        Assert.assertNull(ref);

        bundle = osgi.getBundle("i18n.test.fr");
        bundle.uninstall();

        Thread.sleep(1000);
        ref = osgi.getServiceReference(I18nService.class.getName());
        Assert.assertNull(ref);
    }

    @Test
    public void testManagementOfPreviouslyInstalledExtensions
            () throws BundleException, InterruptedException {
        Bundle bundle = osgi.getBundle("org.ow2.chameleon.i18n.resource-bundle-extender");
        bundle.stop();

        osgi.installAndStart(System.getProperty("ENGLISH"));
        osgi.installAndStart(System.getProperty("FRENCH"));

        bundle.start();

        waitFori18nService(new Locale("fr"), 5000);
        waitFori18nService(new Locale("en"), 5000);
        waitFori18nService(new Locale("fr", "FR"), 5000);

        ServiceReference[] refs = osgi.getServiceReferences(I18nService.class.getName(), null);
        Assert.assertEquals(3, refs.length);

        bundle = osgi.getBundle("i18n.test.en");
        bundle.uninstall();

        bundle = osgi.getBundle("i18n.test.fr");
        bundle.uninstall();

        Thread.sleep(1000);
        ServiceReference ref = osgi.getServiceReference(I18nService.class.getName());
        Assert.assertNull(ref);
    }

    @Test
    public void testSelectorAvailability
            () {
        osgi.waitForService(I18nServiceSelector.class.getName(), null, 5000);
    }


    private static URL createEnglishExtension() throws IOException {
        File file = new File("target/i18n/en.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_en.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_en.properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.en")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createEnglishExtensionInSubFolder() throws IOException {
        File file = new File("target/i18n/en_sub.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/org/Myapp_en.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_en.properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.en")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createEnglishExtensionInSubSubFolder() throws IOException {
        File file = new File("target/i18n/en_subsub.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/org/ow2/i18n/Myapp_en.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_en.properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.en")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createGermanExtension() throws IOException {
        File file = new File("target/i18n/de.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_de.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_de.properties")))
                .add("i18n/myapp2_de.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp2_de.properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.de")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createGermanXMLExtension() throws IOException {
        File file = new File("target/i18n/de_xml.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_de.xml", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_de.xml")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.de-xml")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createGermanUTF8Extension() throws IOException {
        File file = new File("target/i18n/de_utf8.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_de.utf8-properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_de.utf8-properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.de-utf8")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createFrenchExtension() throws IOException {
        File file = new File("target/i18n/fr.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_fr.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_fr.properties")))
                .add("i18n/myapp_fr_FR.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_fr_FR.properties")))

                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.fr")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createAnotherFrenchExtension() throws IOException {
        File file = new File("target/i18n/fr2.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/myapp_fr.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_fr.properties")))

                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.fr2")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }

    private static URL createBundleWithInvalidResources() throws IOException {
        File file = new File("target/i18n/invalid.jar");
        file.getParentFile().mkdirs();

        InputStream is = TinyBundles.newBundle()
                .add("i18n/xml.xml", new FileInputStream(new File("src/test/resources/i18n-resources/xml.xml")))
                .add("i18n/xml_fr.xml", new FileInputStream(new File("src/test/resources/i18n-resources/xml.xml")))
                .add("i18n/myapp_fr.properties", new FileInputStream(new File("src/test/resources/i18n-resources/myapp_fr.properties")))
                .set(Constants.BUNDLE_SYMBOLICNAME, "i18n.test.invalid")
                .set(Constants.IMPORT_PACKAGE, "*")
                .set(Constants.EXPORT_PACKAGE, "!*")
                .build(TinyBundles.withBnd());

        StreamUtils.copyStream(is, new FileOutputStream(file), true);

        return file.toURI().toURL();
    }
}
