/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.sharedprefs.xml;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.Map.Entry;

import org.ow2.chameleon.sharedprefs.SharedPreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xmlpull.v1.XmlPullParserException;

public class SharedPreferencesImpl implements SharedPreferences {

    private Logger m_logger = LoggerFactory.getLogger(SharedPreferencesServiceImpl.class);

    
    private final File m_file;
    private final File m_backupFile;

    private Map<String, Object> m_map;

    private final Object m_content = new Object();
    private WeakHashMap<OnSharedPreferenceChangeListener, Object> m_listeners;
    
    private long m_timestamp;

    SharedPreferencesImpl(
            File file, Map<String, Object> initialContents) {
        m_file = file;
        m_backupFile = makeBackupFile(file);
        m_map = initialContents != null ? initialContents : new HashMap<String, Object>();

        m_listeners =
                new WeakHashMap<OnSharedPreferenceChangeListener, Object>();
        
        m_timestamp = file.lastModified();
    }

    public void replace(Map<String, Object> newContents) {
        if (newContents != null) {
            synchronized (this) {
                m_map = newContents;
            }
        }
    }

    public void registerOnSharedPreferenceChangeListener(
            OnSharedPreferenceChangeListener listener) {
        synchronized (this) {
            m_listeners.put(listener, m_content);
        }
    }

    public void unregisterOnSharedPreferenceChangeListener(
            OnSharedPreferenceChangeListener listener) {
        synchronized (this) {
            m_listeners.remove(listener);
        }
    }

    public Map<String, ?> getAll() {
        synchronized (this) {
            // noinspection unchecked
            return new HashMap<String, Object>(m_map);
        }
    }

    public String getString(String key, String defValue) {
        synchronized (this) {
            String v = (String) m_map.get(key);
            return v != null ? v : defValue;
        }
    }

    public int getInt(String key, int defValue) {
        synchronized (this) {
            Integer v = (Integer) m_map.get(key);
            return v != null ? v : defValue;
        }
    }

    public long getLong(String key, long defValue) {
        synchronized (this) {
            Long v = (Long) m_map.get(key);
            return v != null ? v : defValue;
        }
    }

    public float getFloat(String key, float defValue) {
        synchronized (this) {
            Float v = (Float) m_map.get(key);
            return v != null ? v : defValue;
        }
    }

    public boolean getBoolean(String key, boolean defValue) {
        synchronized (this) {
            Boolean v = (Boolean) m_map.get(key);
            return v != null ? v : defValue;
        }
    }

    public boolean contains(String key) {
        synchronized (this) {
            return m_map.containsKey(key);
        }
    }

    public final class EditorImpl implements Editor
    {
        private final Map<String, Object> m_modified =
                new HashMap<String, Object>();
        private boolean m_clear = false;

        public Editor putString(String key, String value) {
            synchronized (this) {
                m_modified.put(key, value);
                return this;
            }
        }

        public Editor putInt(String key, int value) {
            synchronized (this) {
                m_modified.put(key, value);
                return this;
            }
        }

        public Editor putLong(String key, long value) {
            synchronized (this) {
                m_modified.put(key, value);
                return this;
            }
        }

        public Editor putFloat(String key, float value) {
            synchronized (this) {
                m_modified.put(key, value);
                return this;
            }
        }

        public Editor putBoolean(String key, boolean value) {
            synchronized (this) {
                m_modified.put(key, value);
                return this;
            }
        }

        public Editor remove(String key) {
            synchronized (this) {
                m_modified.put(key, this);
                return this;
            }
        }

        public Editor clear() {
            synchronized (this) {
                m_clear = true;
                return this;
            }
        }

        public boolean commit() {
            boolean returnValue;

            boolean hasListeners;
            List<String> keysModified = null;
            Set<OnSharedPreferenceChangeListener> listeners = null;

            synchronized (SharedPreferencesImpl.this) {
                hasListeners = m_listeners.size() > 0;
                if (hasListeners) {
                    keysModified = new ArrayList<String>();
                    listeners =
                            new HashSet<OnSharedPreferenceChangeListener>(
                                    m_listeners.keySet());
                }

                synchronized (this) {
                    if (m_clear) {
                        m_map.clear();
                        m_clear = false;
                    }

                    for (Entry<String, Object> e : m_modified.entrySet()) {
                        String k = e.getKey();
                        Object v = e.getValue();
                        if (v == this) {
                            m_map.remove(k);
                        }
                        else {
                            m_map.put(k, v);
                        }

                        if (hasListeners) {
                            keysModified.add(k);
                        }
                    }

                    m_modified.clear();
                }

                returnValue = writeFileLocked();
            }

            if (hasListeners) {
                for (int i = keysModified.size() - 1; i >= 0; i--) {
                    final String key = keysModified.get(i);
                    for (OnSharedPreferenceChangeListener listener : listeners) {
                        if (listener != null) {
                            listener.onSharedPreferenceChanged(
                                    SharedPreferencesImpl.this, key);
                        }
                    }
                }
            }

            return returnValue;
        }
    }

    public Editor edit() {
        return new EditorImpl();
    }

    private FileOutputStream createFileOutputStream(File file) {
        FileOutputStream str = null;
        try {
            str = new FileOutputStream(file);
        }
        catch (FileNotFoundException e) {
            File parent = file.getParentFile();
            if (!parent.mkdir()) {
                m_logger.error("Couldn't create directory for " +
                		"SharedPreferences file " + file);
                return null;
            }

            try {
                str = new FileOutputStream(file);
            }
            catch (FileNotFoundException e2) {
                m_logger.error("Couldn't create SharedPreferences file " + file, e2);
            }
        }
        return str;
    }

    private boolean writeFileLocked() {
        // Rename the current file so it may be used as a backup during the next
        // read
        if (m_file.exists()) {
            if (!m_backupFile.exists()) {
                if (!m_file.renameTo(m_backupFile)) {
                    m_logger.error("Couldn't rename file " + m_file 
                            + " to backup file " + m_backupFile);
                    return false;
                }
            }
            else {
                m_file.delete();
            }
        }

        // Attempt to write the file, delete the backup and return true as
        // atomically as
        // possible. If any exception occurs, delete the new file; next time we
        // will restore
        // from the backup.
        try {
            FileOutputStream str = createFileOutputStream(m_file);
            if (str == null) {
                return false;
            }
            XMLUtils.writeMapXml(m_map, str);
            str.close();
            m_timestamp = m_file.lastModified();

            // Writing was successful, delete the backup file if there is one.
            m_backupFile.delete();
            return true;
        }
        catch (XmlPullParserException e) {
            m_logger.warn("writeFileLocked: Got exception:", e);
        }
        catch (IOException e) {
            m_logger.warn("writeFileLocked: Got exception:", e);
        }
        // Clean up an unsuccessfully written file
        if (m_file.exists()) {
            if (!m_file.delete()) {
                m_logger.error("Couldn't clean up partially-written file " + m_file);
            }
        }
        return false;
    }
    
    public static File makeBackupFile(File prefsFile) {
        return new File(prefsFile.getPath() + ".bak");
    }

    public boolean hasFileChanged() {
        return m_timestamp != m_file.lastModified();
    }
}
