/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.sharedprefs.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Property;
import org.apache.felix.ipojo.annotations.Provides;
import org.ow2.chameleon.sharedprefs.SharedPreferences;
import org.ow2.chameleon.sharedprefs.SharedPreferencesService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


@Component(name="org.ow2.chameleon.sharedprefs.XmlSharedPreferences")
@Provides
public class SharedPreferencesServiceImpl implements SharedPreferencesService {

    private Logger m_logger = LoggerFactory.getLogger(SharedPreferencesServiceImpl.class);
    
    private File m_storage;

    private final HashMap<File, SharedPreferencesImpl> m_prefs =
                     new HashMap<File, SharedPreferencesImpl>();

    @Property(name = "location")
    public void setLocation(String location) throws IOException {
        String loc =
                SubstVars.substVars(location, "location", null, System
                        .getProperties());

        m_storage = new File(loc);

        if (!m_storage.exists()) {
            m_storage.mkdirs();
        }
        else {
            // check that it's a directory
            if (!m_storage.isDirectory()) {
                throw new IOException(m_storage.getAbsolutePath()
                        + " is not a directory");
            }
        }

        // Check that we can write
        if (!m_storage.canWrite()) {
            throw new IOException(m_storage.getAbsolutePath()
                    + " is not writable");
        }

    }

    public SharedPreferences getSharedPreferences(String name) {
        return getPreferences(name);
    }

    public File getSharedPrefsFile(String name) {
        return new File(m_storage, name + ".xml");
    }

    private SharedPreferences getPreferences(String name) {
        SharedPreferencesImpl sp;
        File f = getSharedPrefsFile(name);
        synchronized (m_prefs) {
            sp = m_prefs.get(f);
            if (sp != null && !sp.hasFileChanged()) {
                m_logger.info("Returning existing prefs " + name + ": " + sp);
                return sp;
            }
        }

        FileInputStream str = null;
        File backup = SharedPreferencesImpl.makeBackupFile(f);
        if (backup.exists()) {
            f.delete();
            backup.renameTo(f);
        }

        // Debugging
        if (f.exists() && !f.canRead()) {
            m_logger.warn("Attempt to read preferences file " + f 
                    + " without permission");
        }

        Map<String, Object> map = null;
        if (f.exists() && f.canRead()) {
            try {
                str = new FileInputStream(f);
                map = XMLUtils.readMapXml(str);
                str.close();
            }
            catch (org.xmlpull.v1.XmlPullParserException e) {
                m_logger.warn("getSharedPreferences", e);
            }
            catch (FileNotFoundException e) {
                m_logger.warn("getSharedPreferences", e);
            }
            catch (IOException e) {
                m_logger.warn("getSharedPreferences", e);
            }
        }

        synchronized (m_prefs) {
            if (sp != null) {
                m_logger.info("Updating existing prefs " + name + " " + sp + ": " + map);
                sp.replace(map);
            }
            else {
                sp = m_prefs.get(f);
                if (sp == null) {
                    sp = new SharedPreferencesImpl(f, map);
                    m_prefs.put(f, sp);
                }
            }
            return sp;
        }
    }

}
