/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.shrten;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.ow2.chameleon.url.shortener.service.URLShortenerService;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * URL Shortener Service implementation used Shrten.
 * @author Clement Escoffier clement.escoffier@akquinet.de
 */
public class ShrtenURLShortener implements URLShortenerService {

    /**
     * Service Endpoint.
     */
    String m_endpoint; // Injected

    /**
     * Query parameter.
     */
    String m_parameter; // Injected


    /**
     * Gets a short URL from the TinyURL service.
     * Shrten has the specificity to work every time (regardless the input).
     * However the returned URL returns a blank page or connection error.
     * @param url the url to reduce
     * @return the reduced url
     * @throws Exception if something wrong happened (null or empty entry,
     *  Internet connectivity disruption)
     * @see org.ow2.chameleon.url.shortener.
     *  service.URLShortenerService#getShortURL(java.lang.String)
     */
    public String getShortURL(String url) throws Exception {
        if (url == null  || url.length() == 0) {
            throw new NullPointerException("The given URL cannot be null"
                    + " or empty");
        }

        URL query = new URL(m_endpoint + "?" + m_parameter + "=" + url);

        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        // Use the factory to create a builder
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document doc = builder.parse(query.openStream());

        /*
         * <response>
         * <success>true</success>
         * <when>1262875985</when>
         * <shorturl>http://shrten.com/bwn</shorturl>
         * </response>
         */

        boolean success = false;
        String response = null;
        Node node = doc.getDocumentElement(); // <Response>
        NodeList nodes = node.getChildNodes();
        for (int i = 0; i < nodes.getLength(); i++) {
            Node n = nodes.item(i);
            if (n.getNodeName().equals("success")) {
                success = n.getChildNodes().item(0)
                    .getNodeValue().equals("true");
            } else if (n.getNodeName().equals("shorturl")) {
                response = n.getChildNodes().item(0).getNodeValue();
            }
        }

        if (success) {
            return response;
        } else {
            throw new Exception("Cannot compute the short url");
        }
    }

    /**
     * Reads an input stream and returns the result
     * as a String.
     * @param in the input stream
     * @return the read String
     * @throws IOException if the stream cannot be read.
     */
    String read(InputStream in) throws IOException {
        StringBuffer out = new StringBuffer();
        byte[] b = new byte[4096];
        for (int n; (n = in.read(b)) != -1;) {
            out.append(new String(b, 0, n));
        }
        return out.toString();
    }

}
