/*
 * Copyright 2009 OW2 Chameleon
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.chameleon.tinyurl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.ow2.chameleon.url.shortener.service.URLShortenerService;

/**
 * URL Shortener Service implementation used TinyURL.
 * @author Clement Escoffier clement.escoffier@akquinet.de
 */
public class TinyURLShortener implements URLShortenerService {

    /**
     * Service Endpoint.
     */
    String m_endpoint; // Injected

    /**
     * Query parameter.
     */
    String m_parameter; // Injected


    /**
     * Gets a short URL from the TinyURL service.
     * TinyURL has the specificity to work every time (regardless the input).
     * However the returned URL returns a blank page.
     * @param url the url to reduce
     * @return the reduced url
     * @throws Exception if something wrong happened (null or empty entry,
     *  Internet connectivity disruption)
     * @see org.ow2.chameleon.url.shortener.
     *  service.URLShortenerService#getShortURL(java.lang.String)
     */
    public String getShortURL(String url) throws Exception {
        if (url == null  || url.length() == 0) {
            throw new NullPointerException("The given URL cannot be null"
                    + " or empty");
        }

        URL query = new URL(m_endpoint + "?" + m_parameter + "=" + url);
        InputStream is = query.openStream();
        return read(is);
    }

    /**
     * Reads an input stream and returns the result
     * as a String.
     * @param in the input stream
     * @return the read String
     * @throws IOException if the stream cannot be read.
     */
    String read(InputStream in) throws IOException {
        StringBuffer out = new StringBuffer();
        byte[] b = new byte[4096];
        for (int n; (n = in.read(b)) != -1;) {
            out.append(new String(b, 0, n));
        }
        return out.toString();
    }

}
