package org.ow2.clif.util;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;

/**
 * Generates representatives from fractal definitions.
 *
 * @author vbodnart
 */
public class RepresentativesGenerator {


	/**
	 * @param args
	 */
	public static void main(String[] args) {

		if (Arrays.asList(0,1).contains(args.length)) {
			System.out.println("Usage: RepresentativesGenerator <src> <dst>");
			System.out.println("src: src path that contains the .fractal files");
			System.out.println("dst: dst path that will contain the generated files");
			return;
		}

		final File sourceDirFile = new File(args[0]);
		final File outputDirFile = new File(args[1]);

		final String[] arr = System.getProperty("java.class.path").split(File.pathSeparator);
		String proActiveJarPath = null;
		for (int i = 0; i<arr.length; i++){
			if (arr[i].contains("ProActive.jar")) {
				proActiveJarPath = arr[i];
			}
		}

		if (proActiveJarPath == null) {
			throw new RuntimeException("Unable to find to ProActive.jar required to generate representatives");
		}

		final URL[] urls = new URL[2];
		try {
			urls[0] = new URL("file:///"+outputDirFile+"/");
			urls[1] = new URL("file:///"+proActiveJarPath);
		} catch (MalformedURLException e1) {
			e1.printStackTrace();
		}

		long s = System.currentTimeMillis();

	    ExecutorService es = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors(),
			new ThreadFactory() {
			@Override
			public Thread newThread(final Runnable r) {
				Thread t = new Thread(r);
				try {
					ClassLoader cl = new URLClassLoader(urls, null);
				    t.setContextClassLoader(cl);
				} catch (Exception e) {
					e.printStackTrace();
				}
				return t;
			}
		});

	    // Get all fractal files
		final ArrayList<File> files = new ArrayList<File>();
		sourceDirFile.listFiles(new FileFilter(){
			@Override
			public boolean accept(final File pathname) {
				if (pathname.isDirectory()) {
					pathname.listFiles(this);
				}
				if ( pathname.getName().endsWith(".fractal") ){
					files.add(pathname);
				}
				return false;
			}
		});


		HashSet<String> names = new HashSet<String>();

		// get Reader connected to XML input from somewhere..
		try {
			final XMLInputFactory factory = XMLInputFactory.newInstance();
			for (File f : files) {
				Reader reader = new FileReader(f);
				XMLStreamReader streamReader = factory.createXMLStreamReader(reader);
				while (streamReader.hasNext()) {
					int code = streamReader.next();
					if (code == XMLStreamReader.START_ELEMENT) {
						String elementName = streamReader.getLocalName();
						if ("definition".equals(elementName)) {
							continue;
						}
						if ("interface".equals(elementName)) {
							String name = streamReader.getAttributeValue(null, "name");
							String signature = streamReader.getAttributeValue(null, "signature");

							String mocrClassname = org.objectweb.proactive.core.component.gen.Utils.getMetaObjectComponentRepresentativeClassName(name, signature);
							names.add(mocrClassname);
						}
					}
				}
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

		// Submit all generators
		for (final String classname : names) {
			es.submit(new Generator(classname, outputDirFile));
		}

		// Await termination
	    try {
			es.shutdown();
			es.awaitTermination(5, TimeUnit.MINUTES);
	    } catch (Exception e) {
		e.printStackTrace();
	    }

	    System.out.println("Generated " + names.size() + " representatives in " + (System.currentTimeMillis() - s) + " ms");
	}

	/**
	 * Parses a .fractal definition file then dumps into a .class file the generated bytecode of a representative
	 * @author vbodnart
	 */
	static final class Generator implements Runnable {
		final String classname;
		final File outputDir;

		public Generator(String classname, File outputDir) {
			this.classname = classname;
			this.outputDir = outputDir;
		}

		@Override
		public void run() {
			final ClassLoader cl = Thread.currentThread().getContextClassLoader();
			// get Reader connected to XML input from somewhere..
			try {
				Class<?> clas = cl.loadClass("org.objectweb.proactive.core.component.gen.Utils");

				Method m2 = clas.getMethod("getClassData", String.class);
				byte[] classBytes = (byte[]) m2.invoke(null, classname);

				// byte[] classBytes =
				// org.objectweb.proactive.core.component.gen.Utils.getClassData(mocrClassname);

				try {

					File outputFile = new File(this.outputDir, classname + ".class");
					FileOutputStream fos = new FileOutputStream(outputFile);
					fos.write(classBytes);
					fos.close();

					System.out.println("Wrote " + outputFile);

				} catch (FileNotFoundException ex) {
					System.out.println("FileNotFoundException : " + ex);
				} catch (IOException ioe) {
					System.out.println("IOException : " + ioe);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}
}
