/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * WSDLManagerServiceImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.wsdl;

import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.activation.DataHandler;
import javax.jws.WebService;
import javax.mail.util.ByteArrayDataSource;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import org.ow2.dragon.api.service.wsdl.WSDLManager;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.service.DragonFault;
import org.ow2.dragon.service.util.HashMapAdapter;
import org.ow2.dragon.util.XMLUtil;
import org.w3c.dom.Document;

@WebService(endpointInterface = "org.ow2.dragon.service.wsdl.WSDLManagerService")
public class WSDLManagerServiceImpl implements WSDLManagerService {

    private WSDLManager manager;

    public WSDLManager getManager() {
        return this.manager;
    }

    public String[] importServiceDefFile(final String wsdlURI) throws DragonFault {
        String[] result = null;
        try {
            URI uri = URI.create(wsdlURI);
            result = this.manager.importServiceDefFile(uri);
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return result;
    }

    public String[] importServiceDefFileWithRelativeImports(final String wsdlURI,
            @XmlJavaTypeAdapter(HashMapAdapter.class) HashMap<String, String> imports)
            throws DragonFault {
        String[] result = null;
        try {
            URI uri = URI.create(wsdlURI);
            Map<String, URI> importsURIs = new HashMap<String, URI>();
            for (String key : imports.keySet()) {
                importsURIs.put(key, URI.create(imports.get(key)));
            }
            result = this.manager.importServiceDefFile(uri, importsURIs, null);
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return result;
    }

    public void setManager(final WSDLManager manager) {
        this.manager = manager;
    }

    public String[] importServiceDefFile(AttachedDescription attachedDescription)
            throws DragonFault {
        String[] result = null;
        try {
            DataHandler desc = attachedDescription.getDescription();
            Document documentDesc = XMLUtil.loadDocument(desc.getInputStream());
            documentDesc.setDocumentURI(".");
            result = this.manager.importServiceDefFile(documentDesc);
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return result;
    }

    public List<String> loadServiceDescsAsString(String serviceId) throws DragonFault {
        List<String> result = new ArrayList<String>();
        try {
            List<ServiceSpecificationTO> specs = this.manager.getServSpecsForTechServ(serviceId);
            if (specs != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : specs) {
                    result.add(this.manager.getWsdlDescAsString(serviceSpecificationTO.getId()));
                }

            }
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return result;
    }

    public List<AttachedDescription> loadServiceDescs(String serviceId) throws DragonFault {
        List<AttachedDescription> descriptions = new ArrayList<AttachedDescription>();
        try {
            List<ServiceSpecificationTO> specs = this.manager.getServSpecsForTechServ(serviceId);
            if (specs != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : specs) {
                    InputStream inputStream = this.manager.getWsdlDesc(serviceSpecificationTO
                            .getId());
                    AttachedDescription description = new AttachedDescription();
                    description.setDescription(new DataHandler(new ByteArrayDataSource(inputStream,
                            "application/octet-stream")));
                    descriptions.add(description);
                }
            }
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return descriptions;
    }

    public List<AttachedDescription> loadEndpointDescs(String endpointId) throws DragonFault {
        List<AttachedDescription> descriptions = new ArrayList<AttachedDescription>();
        try {
            List<ServiceSpecificationTO> specs = this.manager.getServSpecsForEndpoint(endpointId);
            if (specs != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : specs) {
                    InputStream inputStream = this.manager.getWsdlDesc(serviceSpecificationTO
                            .getId());
                    AttachedDescription description = new AttachedDescription();
                    description.setDescription(new DataHandler(new ByteArrayDataSource(inputStream,
                            "application/octet-stream")));
                    descriptions.add(description);
                }
            }
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return descriptions;
    }

    public List<String> loadEndpointDescsAsString(String endpointId) throws DragonFault {
        List<String> result = new ArrayList<String>();
        try {
            List<ServiceSpecificationTO> specs = this.manager.getServSpecsForEndpoint(endpointId);
            if (specs != null) {
                for (ServiceSpecificationTO serviceSpecificationTO : specs) {
                    result.add(this.manager.getWsdlDescAsString(serviceSpecificationTO.getId()));
                }

            }
        } catch (Throwable t) {
            throw new DragonFault(t.getMessage(), t);
        }
        return result;
    }
}
