/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PostManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.organization;

import java.util.List;

import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.springframework.transaction.annotation.Transactional;

/**
 * A manager of posts. Provides methods to create, retrieve, delete or update
 * posts
 * 
 * @author ambarthe, ofabre
 * 
 */
@Transactional
public interface PostManager {

    /**
     * Add a new post in registry. The post name must be specified.
     * 
     * @param postTO
     *            the {@link PostTO} handling data of the {@link Post} to add
     * @return the added {@link Post} id
     * @throws OrganizationException
     *             if you try to create a post with no name or a name that
     *             already exist
     */
    public String createPost(PostTO postTO) throws OrganizationException;

    /**
     * Return the list of all posts in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link PostTO}s in the registry. Not
     *         null, could be empty
     */
    @Transactional(readOnly = true)
    public List<PostTO> getAllPosts(RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the post matching the given id
     * 
     * @param postId
     *            a {@link Post} id
     * @return the {@link PostTO} handling data of the {@link Post} that match
     *         the given id
     * @throws OrganizationException
     *             if you try to retrieve a post that doesn't exist
     */
    @Transactional(readOnly = true)
    public PostTO getPost(String postId) throws OrganizationException;

    /**
     * Return the list of all posts in the registry that aren't linked to the
     * given organization
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link PostTO}s in the registry that
     *         aren't linked to the given organization
     * @throws OrganizationException
     *             if an error occurs during posts retrieval
     */
    public List<PostTO> getPostsNotLinkedToOrganization(String orgId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Remove the post matching the given id from the registry
     * 
     * @param postId
     *            a {@link Post} id
     */
    public void removePost(String postId);

    /**
     * Update the post in the registry. Post name and id must be specified
     * 
     * @param postTO
     *            the {@link PostTO} handling data of the {@link Post} to update
     * @return the updated post ID
     * @throws OrganizationException
     *             if you try to update a post with a name that already exist or
     *             with a null or empty name, or if you try to update a post
     *             that doesn't exist
     */
    public String updatePost(PostTO postTO) throws OrganizationException;

}
