/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleOfPartyManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.organization;

import java.util.List;

import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.OrgToEndpointTO;
import org.ow2.dragon.api.to.organization.OrgToTechServiceTO;
import org.ow2.dragon.api.to.organization.PersonToEndpointTO;
import org.ow2.dragon.api.to.organization.PersonToTechServiceTO;
import org.ow2.dragon.persistence.bo.common.Link;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Party;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
@Transactional
public interface RoleOfPartyManager {

    /**
     * An enumeration of all available types of links between registry entities
     */
    public enum LINK_TYPE {
        PERSON_TO_TECHSERVICE, ORG_TO_TECHSERVICE, PERSON_TO_ENDPOINT, ORG_TO_ENDPOINT
    }

    /**
     * Create a link between a party (person or organization) and an endpoint
     * 
     * @param partyId
     *            the {@link Party} id ( {@link Person} id or
     *            {@link OrganizationUnit} id)
     * @param endpointId
     *            the {@link Endpoint} id
     * @param type
     *            the role of the {@link Party} towards the {@link Endpoint}
     * @return the created role id
     * @throws OrganizationException
     *             if an error occurs during role creation
     */
    public String createEndpointRole(String partyId, String endpointId, String type)
            throws OrganizationException;

    /**
     * Create a link between a party (person or organization) and a tech service
     * 
     * @param partyId
     *            the {@link Party} id ( {@link Person} id or
     *            {@link OrganizationUnit} id)
     * @param technicalServiceId
     *            the {@link TechnicalService} id
     * @param type
     *            the role of the {@link Party} towards the
     *            {@link TechnicalService}
     * @return the created role id
     * @throws OrganizationException
     *             if an error occurs during role creation
     */
    public String createTechServiceRole(String partyId, String technicalServiceId, String type)
            throws OrganizationException;

    /**
     * Retrieve all links between the given endpoint and an organization
     * 
     * @param endpointId
     *            an {@link Endpoint} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link OrgToEndpointTO} roles between
     *         {@link OrganizationUnit}s and the given {@link Endpoint}
     * @throws OrganizationException
     *             if an error occurs during roles retrieval
     */
    @Transactional(readOnly = true)
    public List<OrgToEndpointTO> getOrgRolesByEndpoint(String endpointId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Retrieve all links between the given tech service and an organization
     * 
     * @param techServiceId
     *            a {@link TechnicalService} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link OrgToTechServiceTO} roles between
     *         {@link OrganizationUnit}s and the given {@link TechnicalService}
     * @throws OrganizationException
     *             if an error occurs during roles retrieval
     */
    @Transactional(readOnly = true)
    public List<OrgToTechServiceTO> getOrgRolesByTechService(String techServiceId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Retrieve all links between the given endpoint and a person
     * 
     * @param endpointId
     *            an {@link Endpoint} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link PersonToEndpointTO} roles between
     *         {@link Person}s and the given {@link Endpoint}
     * @throws OrganizationException
     *             if an error occurs during roles retrieval
     */
    @Transactional(readOnly = true)
    public List<PersonToEndpointTO> getPersonRolesByEndpoint(String endpointId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Retrieve all links between the given tech service and a person
     * 
     * @param techServiceId
     *            a {@link TechnicalService} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link PersonToTechServiceTO} roles between
     *         {@link Person}s and the given {@link TechnicalService}
     * @throws OrganizationException
     *             if an error occurs during roles retrieval
     */
    @Transactional(readOnly = true)
    public List<PersonToTechServiceTO> getPersonRolesByTechService(String techServiceId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Get all role types for a particular link type. For exemple, the list of
     * role type for a link between an organization and an endpoint is:
     * "Analyst", "Consumer", "Owner", "Provider".
     * 
     * @param linkType
     *            a {@link LINK_TYPE}
     * @param iOptions
     *            sort, pagination and case sensitive parameters, could be null
     * @return all role types as {@link String} array for a particular
     *         {@link LINK_TYPE}
     * @throws OrganizationException
     *             if an error occurs during role types retrieval
     */
    @Transactional(readOnly = true)
    public String[] getRolesTypes(LINK_TYPE linkType, RequestOptionsTO iOptions)
            throws OrganizationException;

    /**
     * Remove a role from the registry
     * 
     * @param roleId
     *            a role ( {@link Link}) id
     * @throws OrganizationException
     *             if an error occurs during role removal
     */
    public void removeRole(String roleId) throws OrganizationException;

}
