/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechnologyManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.technology;

import java.util.List;

import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.technology.EnvironmentFederationTO;
import org.ow2.dragon.api.to.technology.ExecEnvManagerSearchProperties;
import org.ow2.dragon.api.to.technology.ExecEnvSearchProperties;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentManagerTO;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentTO;
import org.ow2.dragon.api.to.technology.FederationSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorTO;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author ofabre - eBM Websourcing
 * 
 */
@Transactional
public interface TechnologyManager {

    /**
     * Add a new runtime manager to the registry. Uses the provided address to
     * connect to the manager interface and retrieve manager properties
     * 
     * @param address
     *            a manager service interface address like
     *            'http://com.foo/ManagerService'. Must be a valid URL
     * @return the added manager ID
     * @throws TechnologyException
     *             if an error occurs during runtime manager creation
     */
    public String addRuntimeManager(String address) throws TechnologyException;

    /**
     * Retrieve the runtime manager matching the given ID
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @return the {@link ExecutionEnvironmentTO} (aka runtime manager) matching
     *         the given ID
     */
    @Transactional(readOnly = true)
    public ExecutionEnvironmentManagerTO getRuntimeManager(String runtimeManagerId);

    /**
     * Retrieve the environment federation matching the given ID
     * 
     * @param envFedId
     *            an environment federation ID
     * @return the {@link EnvironmentFederationTO} matching the given ID
     */
    @Transactional(readOnly = true)
    public EnvironmentFederationTO getEnvironmentFederation(String envFedId);

    /**
     * Retrieve the list of execution environments managed by the given runtime
     * manager (aka execution environment manager)
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s managed by
     *         the given runtime manager (aka
     *         {@link ExecutionEnvironmentManagerTO})
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getManagedExecEnv(String runtimeManagerId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of execution environments that are parts of the given
     * environment federation
     * 
     * @param envFedId
     *            an environment federation ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s that are
     *         parts of the given {@link EnvironmentFederationTO}
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getExecEnvsInFederation(String envFedId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Remove the runtime manager matching the given ID from the registry
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     */
    public void removeRuntimeManager(String runtimeManagerId);

    /**
     * Remove the environment federation matching the given ID from the registry
     * 
     * @param envFedId
     *            an environment federation ID
     */
    public void removeEnvironmentFederation(String envFedId);

    /**
     * Synchronize all execution environments managed by this execution runtime
     * manager. All linked endpoints, environment federations and processors
     * will also be synchronized. Synchronization consist in adding elements to
     * the registry
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @throws TechnologyException
     *             if an error occurs during synchronization
     */
    public void synchronizeManagedExecEnv(String runtimeManagerId) throws TechnologyException;

    /**
     * Retrieve the execution environment matching the given ID
     * 
     * @param execEnvId
     *            an execution environment ID
     * @return the {@link ExecutionEnvironmentTO} matching the given ID
     */
    @Transactional(readOnly = true)
    public ExecutionEnvironmentTO getExecutionEnvironment(String execEnvId);

    /**
     * Retrieve the processor matching the given ID
     * 
     * @param processorId
     *            a processor ID
     * @return the {@link ProcessorTO} matching the given ID
     */
    @Transactional(readOnly = true)
    public ProcessorTO getProcessor(String processorId);

    /**
     * Remove the execution environment matching the given ID from the registry
     * 
     * @param execEnvId
     *            an execution environment ID
     */
    public void removeExecutionEnvironment(String execEnvId);

    /**
     * Remove the processor matching the given ID from the registry
     * 
     * @param processorId
     *            a processor ID
     */
    public void removeProcessor(String processorId);

    /**
     * Retrieve the list of endpoints that are hosted by the given execution
     * environment
     * 
     * @param execEnvId
     *            an execution environment ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of {@link EndpointTO}s that are hosted by the
     *         given {@link ExecutionEnvironmentTO}
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEpsHostedOnExecEnv(String execEnvId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of endpoints that are hosted by the given processor. It
     * means all endpoints hosted by execution environment hosted on the given
     * processor
     * 
     * @param processorId
     *            a processor ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of {@link EndpointTO}s that are hosted by the
     *         given {@link ProcessorTO}
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEpsHostedOnProcessor(String processorId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of execution environments that are hosted by the given
     * processor.
     * 
     * @param processorId
     *            a processor ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s that are
     *         hosted by the given {@link ProcessorTO}
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getExecEnvsHostedOnProcessor(String processorId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Return the list of all runtime managers (aka Execution Environment
     * Managers) in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return {@link List} of all runtime managers (aka
     *         {@link ExecutionEnvironmentManagerTO}) in the registry
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentManagerTO> getAllExecutionEnvironmentManagers(
            RequestOptionsTO requestOptionsTO);

    /**
     * Return the list of all environment federations in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link EnvironmentFederationTO}s in the
     *         registry
     */
    @Transactional(readOnly = true)
    public List<EnvironmentFederationTO> getAllEnvironmentFederations(
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all runtime managers (aka execution environment
     * managers) matching the given criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces)
     * @param searchedProperties
     *            the searched properties
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all runtime managers (aka
     *         {@link ExecutionEnvironmentManagerTO}s) matching the given
     *         criteria for the given properties
     * @throws TechnologyException
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentManagerTO> searchExecutionEnvironmentManagers(
            String searchCriteria, List<ExecEnvManagerSearchProperties> searchedProperties,
            RequestOptionsTO requestOptionsTO) throws TechnologyException;

    /**
     * Retrieve the list of all environment federations matching the given
     * criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces)
     * @param searchedProperties
     *            the searched properties
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link EnvironmentFederationTO}s matching
     *         the given criteria for the given properties
     * @throws TechnologyException
     */
    @Transactional(readOnly = true)
    public List<EnvironmentFederationTO> searchEnvironmentFederations(String searchCriteria,
            List<FederationSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException;

    /**
     * Return the list of all execution environments in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link ExecutionEnvironmentTO}s in the
     *         registry
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getAllExecutionEnvironments(
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all execution environments matching the given
     * criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces)
     * @param searchedProperties
     *            the searched properties
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link ExecutionEnvironmentTO}s matching
     *         the given criteria for the given properties
     * @throws TechnologyException
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> searchExecutionEnvironments(String searchCriteria,
            List<ExecEnvSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException;

    /**
     * Return the list of all processors in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link ProcessorTO}s in the registry
     */
    @Transactional(readOnly = true)
    public List<ProcessorTO> getAllProcessors(RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all processors matching the given criteria for the
     * given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces)
     * @param searchedProperties
     *            the searched properties
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return the {@link List} of all {@link ProcessorTO}s matching the given
     *         criteria for the given properties
     * @throws TechnologyException
     */
    @Transactional(readOnly = true)
    public List<ProcessorTO> searchProcessors(String searchCriteria,
            List<ProcessorSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException;

}
