/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * SimpleDAOImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.persistence.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.hibernate.NonUniqueResultException;
import org.ow2.dragon.persistence.util.HibernateQueryHelper;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;

import com.trg.search.IMutableSearch;
import com.trg.search.ISearch;
import com.trg.search.Search;
import com.trg.search.SearchResult;
import com.trg.search.hibernate.HibernateSearchProcessor;

/**
 * This class serves as the Base class for all other DAOs - namely to hold
 * common CRUD methods that they might all use. You should only need to extend
 * this class when your require custom CRUD logic.
 * 
 * <p>
 * To register this class in your Spring context file, use the following XML.
 * 
 * <pre>
 *      &lt;bean id=&quot;fooDao&quot; class=&quot;org.ow2.dragon.persistence.dao.GenericDaoHibernate&quot;&gt;
 *          &lt;property name=&quot;type&quot; value=&quot;org.ow2.dragon.persistence.bo.Foo&quot;/&gt;
 *          &lt;property name=&quot;sessionFactory&quot; ref=&quot;sessionFactory&quot;/&gt;
 *      &lt;/bean&gt;
 * </pre>
 * 
 * @author <a href="mailto:bwnoll@gmail.com">Bryan Noll</a>, ofabre - eBM
 *         WebSourcing
 * @param <T>
 *            a type variable
 * @param <PK>
 *            the primary key for that type
 * 
 */
public class GenericHibernateDAOImpl<T, PK extends Serializable> extends HibernateDaoSupport
        implements GenericORMDAO<T, PK> {

    private Logger log = Logger.getLogger(this.getClass());

    public Class<T> type;

    public void setType(Class<T> type) {
        this.type = type;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAll(final RequestOptions requestOptions) {
        /*
         * List result = new ArrayList<T>(); final DetachedCriteria
         * detachedCriteria = DetachedCriteria.forClass(this.type); result =
         * HibernateQueryHelper
         * .sortAndPaginateByCriteria(this.getHibernateTemplate(),
         * requestOptions, detachedCriteria); return result;
         */

        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        ISearch search = HibernateQueryHelper.createSearchContext(this.type, null, null,
                requestOptions, null);

        return searchProcessor.search(getSession(), search);
    }

    @SuppressWarnings("unchecked")
    private List<T> search(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions, final String predicate) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        ISearch search = HibernateQueryHelper.createSearchContext(this.type, criteria, properties,
                requestOptions, predicate);

        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    public List<T> searchEquals(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions) {
        return this.search(criteria, properties, requestOptions,
                HibernateQueryHelper.PREDICATE_EQUALS);
    }

    /**
     * {@inheritDoc}
     */
    public List<T> searchLike(final String[] criteria, final String[] properties,
            final RequestOptions requestOptions) {
        return this.search(criteria, properties, requestOptions,
                HibernateQueryHelper.PREDICATE_LIKE);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAll() {
        return super.getHibernateTemplate().loadAll(this.type);
    }

    public List<T> getAll(List<PK> ids) {
        return this.getAll(ids, null);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAll(List<PK> ids, RequestOptions requestOptions) {
        /*
         * List<T> result = new ArrayList<T>(); if (ids != null &&
         * !ids.isEmpty()) { // StringBuffer queryBuffer = new StringBuffer("");
         * String simpleTypeName = this.type.getSimpleName(); String
         * fullTypeName = this.type.getName();
         * 
         * HibernateQueryHelper queryBuilder = new HibernateQueryHelper();
         * 
         * queryBuilder.appendSelectStatement(simpleTypeName, fullTypeName);
         * 
         * queryBuilder.WHERE().pad();
         * queryBuilder.appendInList(simpleTypeName.toLowerCase() + ".id", ids);
         * 
         * queryBuilder.appendSortOption(requestOptions,
         * simpleTypeName.toLowerCase()); Query query =
         * queryBuilder.buildHibernatePaginatedQuery(this.getSession(),
         * requestOptions);
         * 
         * log.debug("Hibernate Query: " + query); result = query.list(); }
         * return result;
         */
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());

        Search search = (Search) HibernateQueryHelper.createSearchContext(this.type, null, null,
                requestOptions, null);
        search.addFilterIn("id", ids);

        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> getAllDistinct() {
        Collection result = new LinkedHashSet(getAll());
        return new ArrayList(result);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public T get(PK id) {
        T entity = (T) super.getHibernateTemplate().get(this.type, id);

        // if (entity == null) {
        // log.warn("Uh oh, '" + this.type + "' object with id '" + id +
        // "' not found...");
        // throw new ObjectRetrievalFailureException(this.type, id);
        // }

        return entity;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public boolean exists(PK id) {
        T entity = (T) super.getHibernateTemplate().get(this.type, id);
        return entity != null;
    }

    /**
     * {@inheritDoc}
     */
    public T save(T object) {
        super.getHibernateTemplate().saveOrUpdate(object);
        return object;
    }

    /**
     * {@inheritDoc}
     */
    public void remove(PK id) {
        T object = this.get(id);
        // if object is null, skip removal
        if (object != null) {
            super.getHibernateTemplate().delete(object);
        }
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> findByNamedQuery(String queryName, Map<String, Object> queryParams) {
        String[] params = new String[queryParams.size()];
        Object[] values = new Object[queryParams.size()];
        int index = 0;
        Iterator<String> i = queryParams.keySet().iterator();
        while (i.hasNext()) {
            String key = i.next();
            params[index] = key;
            values[index++] = queryParams.get(key);
        }
        return getHibernateTemplate().findByNamedQueryAndNamedParam(queryName, params, values);
    }

    /**
     * {@inheritDoc}
     */
    public Class<T> getManipulatedType() {
        return this.type;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public List<T> search(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.search(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    public int count(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.count(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public SearchResult<T> searchAndCount(IMutableSearch search) {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return searchProcessor.searchAndCount(getSession(), search);
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public T searchUnique(IMutableSearch search) throws NonUniqueResultException {
        HibernateSearchProcessor searchProcessor = HibernateSearchProcessor
                .getInstanceForSessionFactory(getSessionFactory());
        search.setSearchClass(this.type);
        return (T) searchProcessor.searchUnique(getSession(), search);
    }

}
