/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * SimpleDAO.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.persistence.dao;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import org.hibernate.NonUniqueResultException;
import org.springframework.transaction.annotation.Transactional;

import com.trg.search.IMutableSearch;
import com.trg.search.ISearch;
import com.trg.search.SearchResult;

/**
 * Generic DAO (Data Access Object) with common methods to CRUD POJOs.
 * 
 * <p>
 * Extend this interface if you want typesafe (no casting necessary) DAO's for
 * your domain objects.
 * 
 * @author <a href="mailto:bwnoll@gmail.com">Bryan Noll</a>, ofabre - eBM
 *         WebSourcing
 * @param <T>
 *            a type variable
 * @param <PK>
 *            the primary key for that type
 * 
 */
public interface GenericORMDAO<T, PK extends Serializable> {

    /**
     * Generic method used to get all objects of a particular type, sorted and
     * paginated according to the given request options. This is the same as
     * lookup up all rows in a table. The sort option works only on single
     * properties, cause colection properties need to be sorted. If you have to
     * sort on collection properties, use search(IMutableSearch search) method
     * with sort and fetching options
     * 
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return List of populated objects
     */
    List<T> getAll(RequestOptions requestOptionsTO);

    /**
     * Generic method used to get all objects of a particular type. This is the
     * same as lookup up all rows in a table.
     * 
     * @return List of populated objects
     */
    List<T> getAll();

    /**
     *Return the persistent instance of an entity with the given identifier, or
     * <code>null</code> if not found.
     * 
     * @param id
     *            the identifier of the persistent instance
     * @return the persistent instance, or <code>null</code> if not found
     * @throws org.springframework.dao.DataAccessException
     *             in case of Hibernate errors
     * @see org.hibernate.Session#get(Class, java.io.Serializable)
     */
    T get(PK id);

    /**
     * Checks for existence of an object of type T using the id arg.
     * 
     * @param id
     *            the id of the entity
     * @return - true if it exists, false if it doesn't
     */
    boolean exists(PK id);

    /**
     * Generic method to save an object - handles both update and insert.
     * 
     * @param object
     *            the object to save
     * @return the persisted object
     */
    @Transactional
    T save(T object);

    /**
     * Generic method to delete an object based on class and id
     * 
     * @param id
     *            the identifier (primary key) of the object to remove
     */
    void remove(PK id);

    /**
     * Gets all records without duplicates.
     * <p>
     * Note that if you use this method, it is imperative that your model
     * classes correctly implement the hashcode/equals methods
     * </p>
     * 
     * @return List of populated objects
     */
    List<T> getAllDistinct();

    /**
     * Find a list of records by using a named query
     * 
     * @param queryName
     *            query name of the named query
     * @param queryParams
     *            a map of the query names and the values
     * @return a list of the records found
     */
    List<T> findByNamedQuery(String queryName, Map<String, Object> queryParams);

    /**
     * This method allows to search Objects on String properties, fitting search
     * criteria. Properties can be direct object fields or fields of included
     * objects (in this case, fields must be referenced by the doted notation
     * like "foo.bar"). Use a like statement for each criteria on each property.
     * Results are sorted and paginated in respect of the given request options.
     * 
     * @param criteria
     *            the search criteria
     * @param properties
     *            the searched properties
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return a list of object matching the different criteria sorted and
     *         paginated
     */
    List<T> searchLike(String[] criteria, String[] properties, RequestOptions requestOptionsTO);

    /**
     * This method allows to search Objects on String properties, fitting search
     * criteria. Properties can be direct object fields or fields of included
     * objects (in this case, fields must be referenced by the doted notation
     * like "foo.bar"). Use a equality statement for each criteria on each
     * property. Results are sorted and paginated in respect of the given
     * request options.
     * 
     * @param criteria
     *            the search criteria
     * @param properties
     *            the searched properties
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return a list of object matching the different criteria sorted and
     *         paginated
     */
    List<T> searchEquals(String[] criteria, String[] properties, RequestOptions requestOptionsTO);

    /**
     * Return the type of entity manipulated by this DAO
     * 
     * @return the {@link Class} of the manipulated entity
     */
    Class<T> getManipulatedType();

    /**
     * Retrieve a {@link List} of entities matching given ids
     * 
     * @param ids
     *            a {@link List} of ids
     * @return a {@link List} of entities matching ids, must be non null, could
     *         be empty
     */
    List<T> getAll(List<PK> ids);

    /**
     * Retrieve a {@link List} of entities matching given ids, sorted and
     * paginated according to the given request options
     * 
     * @param ids
     *            a {@link List} of ids
     * @param requestOptions
     *            include sort order and pagination information
     * @return a sorted/paginated {@link List} of entities matching ids, must be
     *         non null, could be empty
     */
    List<T> getAll(List<PK> ids, RequestOptions requestOptions);

    /**
     * Search for objects based on the search parameters in the specified
     * <code>ISearch</code> object.
     * 
     * @see ISearch
     */
    public List<T> search(IMutableSearch search);

    /**
     * Returns the total number of results that would be returned using the
     * given <code>ISearch</code> if there were no paging or maxResult limits.
     * 
     * @see ISearch
     */
    public int count(IMutableSearch search);

    /**
     * Returns a <code>SearchResult</code> object that includes the list of
     * results like <code>search()</code> and the total length like
     * <code>searchLength</code>.
     * 
     * @see ISearch
     */
    public SearchResult<T> searchAndCount(IMutableSearch search);

    /**
     * Search for a single result using the given parameters.
     */
    public T searchUnique(IMutableSearch search) throws NonUniqueResultException;

}
