/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TransfertObjectAssemblerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.namespace.QName;

import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.common.CategoryTO;
import org.ow2.dragon.api.to.common.CategoryValueTO;
import org.ow2.dragon.api.to.common.IdentifierTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.BindingOperationTO;
import org.ow2.dragon.api.to.deployment.BindingTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.ProtocolTO;
import org.ow2.dragon.api.to.deployment.RatingTO;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.deployment.TransportTO;
import org.ow2.dragon.api.to.metadata.SimpleFileTO;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.api.to.specification.ServiceInterfaceTO;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.api.to.specification.SpecifiedOperationTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Comment;
import org.ow2.dragon.persistence.bo.common.Description;
import org.ow2.dragon.persistence.bo.common.Identifier;
import org.ow2.dragon.persistence.bo.common.KRKeyNames;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.Rating;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.BindingOperation;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.Protocol;
import org.ow2.dragon.persistence.bo.deployment.Standard;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.deployment.Transport;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.organization.Address;
import org.ow2.dragon.persistence.bo.organization.AddressLine;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.Email;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.bo.organization.PhoneNumber;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.bo.specification.SpecifiedOperation;
import org.ow2.dragon.persistence.bo.technology.ExecutionEnvironment;
import org.ow2.dragon.persistence.bo.technology.Node;
import org.ow2.dragon.persistence.bo.technology.Processor;
import org.ow2.dragon.persistence.dao.RequestOptions;
import org.ow2.dragon.persistence.dao.organization.OrganizationUnitDAO;
import org.ow2.dragon.persistence.dao.organization.PostDAO;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class TransfertObjectAssemblerImpl implements TransfertObjectAssembler {

    private OrganizationUnitDAO organizationUnitDAO;

    private PostDAO postDAO;

    public static final String STREET_NUMBER_KEY = "streetNumber";

    public static final String COMPLEMENT_KEY = "complement";

    public static final String STREET_KEY = "street";

    public static final String COUNTRY_KEY = "country";

    public static final String STATE_KEY = "state";

    public static final String ZIPCODE_KEY = "zipcode";

    public static final String CITY_KEY = "city";

    public void setOrganizationUnitDAO(final OrganizationUnitDAO organizationUnitDAO) {
        this.organizationUnitDAO = organizationUnitDAO;
    }

    public void setPostDAO(final PostDAO postDAO) {
        this.postDAO = postDAO;
    }

    public ServiceSpecificationTO toServiceSpecTO(final ServiceSpecification serviceSpecification) {
        ServiceSpecificationTO serviceSpecificationTO = null;
        if (serviceSpecification != null) {
            serviceSpecificationTO = new ServiceSpecificationTO();
            serviceSpecificationTO.setId(serviceSpecification.getId());

            serviceSpecificationTO.setName(serviceSpecification.getName().getName());
            // TODO For the moment there's only one file attached to a service
            // spec
            serviceSpecificationTO.setFileName(serviceSpecification.getOverviewDocs().get(0)
                    .getFileName());
        }
        return serviceSpecificationTO;
    }

    public BindingOperationTO toBindingOpTO(final BindingOperation bindingOperation) {
        BindingOperationTO bindingOperationTO = null;
        if (bindingOperation != null) {
            bindingOperationTO = new BindingOperationTO();
            bindingOperationTO.setId(bindingOperation.getId());
            bindingOperationTO.setSignature(bindingOperation.getSignature());
            bindingOperationTO.setStyle(bindingOperation.getStyle().toString());
            bindingOperationTO.setPattern(bindingOperation.getSpecifiedOp().getTransmission()
                    .toString());
        }
        return bindingOperationTO;
    }

    public BindingTO toBindingTO(final Binding binding) {
        BindingTO bindingTO = null;
        if (binding != null) {
            bindingTO = new BindingTO();
            bindingTO.setId(binding.getId());
            bindingTO.setName(binding.getFullName());
            final Set<Protocol> protocols = binding.getProtocols();
            if (protocols != null) {
                final Set<ProtocolTO> protocolTOs = new HashSet<ProtocolTO>();
                for (Protocol protocol : protocols) {
                    protocolTOs.add(this.toProtocolTO(protocol));
                }
                bindingTO.setProtocols(protocolTOs);
            }

            final Set<Transport> transports = binding.getTransports();
            if (transports != null) {
                final Set<TransportTO> transportTOs = new HashSet<TransportTO>();
                for (Transport transport : transports) {
                    transportTOs.add(this.toTransportTO(transport));
                }
                bindingTO.setTransports(transportTOs);
            }
            bindingTO.setServiceInterface(this.toServiceInterfaceTO(binding.getServiceInterface()));
            final Set<BindingOperation> bindingOperations = binding.getBindingOps();
            if (bindingOperations != null) {
                final List<BindingOperationTO> bindingOperationsTO = new ArrayList<BindingOperationTO>();
                for (final BindingOperation bindingOperation : bindingOperations) {
                    bindingOperationsTO.add(this.toBindingOpTO(bindingOperation));
                }
                bindingTO.setBindingOps(bindingOperationsTO);
            }
        }
        return bindingTO;
    }

    public EndpointTO toEndpointTO(final Endpoint endpoint, final String user) {
        EndpointTO endpointTO = null;
        if (endpoint != null) {
            endpointTO = new EndpointTO();
            endpointTO.setBinding(this.toBindingTO(endpoint.getBinding()));
            endpointTO.setId(endpoint.getId());
            endpointTO.setName(endpoint.getName());
            endpointTO.setNetworkAddress(endpoint.getNetworkAddress());
            if (endpoint.getBinding() != null) {
                final Set<Protocol> protocols = endpoint.getBinding().getProtocols();
                if (protocols != null) {
                    final Set<ProtocolTO> protocolTOs = new HashSet<ProtocolTO>();
                    for (Protocol protocol : protocols) {
                        protocolTOs.add(this.toProtocolTO(protocol));
                    }
                    endpointTO.setProtocols(protocolTOs);
                }

                final Set<Transport> transports = endpoint.getBinding().getTransports();
                if (transports != null) {
                    final Set<TransportTO> transportTOs = new HashSet<TransportTO>();
                    for (Transport transport : transports) {
                        transportTOs.add(this.toTransportTO(transport));
                    }
                    endpointTO.setTransports(transportTOs);
                }
            }
            Node hostNode = endpoint.getHostNode();
            if (hostNode != null) {
                if (hostNode instanceof Processor) {
                    endpointTO.setHostProcessorId(hostNode.getId());
                } else {
                    ExecutionEnvironment execEnv = (ExecutionEnvironment) hostNode;
                    endpointTO.setHostExecEnvId(execEnv.getId());
                    if (execEnv.getHostProcessor() != null) {
                        endpointTO.setHostProcessorId(execEnv.getHostProcessor().getId());
                    }
                }
            }
            endpointTO.setTechService(this.toTechServiceTO(endpoint.getTechnicalService(), null));

            if (user != null) {
                Set<Rating> ratings = endpoint.getRatings();
                Rating userRating = null;
                float globalRating = 0f;
                Iterator<Rating> iter = ratings.iterator();
                while (iter.hasNext()) {
                    Rating ratingTemp = iter.next();
                    globalRating = globalRating + ratingTemp.getMyRating();
                    if (ratingTemp.getUserName().compareTo(user) == 0) {
                        userRating = ratingTemp;
                    }
                }
                // if the user never rated or opened this service
                if (userRating == null) {
                    userRating = new Rating(user);
                }
                if (ratings.size() != 0) {
                    endpointTO.setGlobalRating(globalRating / ratings.size());
                } else {
                    endpointTO.setGlobalRating(0);
                }
                endpointTO.setRatingTO(this.toRatingTO(userRating));
            }

            // add categories
            endpointTO.setCategories(this.toCategoriesTO(endpoint.getCategoryBag()));

            // add the comments
            endpointTO.setComments(this.toCommentsTO(endpoint.getComments()));
            // add the tags
            endpointTO.setTags(new HashSet<String>(endpoint.getTags()));

        }
        return endpointTO;
    }

    /**
     * Create a BO copy of the given TO. Can be used for create or update
     * purposes
     * 
     * @param organizationUnitTO
     *            must not be null
     * @param organizationUnitBO
     *            must not be null
     * @throws OrganizationException
     * @throws NullPointerException
     *             if bo or to is null
     */
    public void toOrganizationUnitBO(final OrganizationUnitTO organizationUnitTO,
            final OrganizationUnit organizationUnitBO) throws OrganizationException {

        setAddresses(organizationUnitTO, organizationUnitBO);

        // only one url of type "homepage" for the moment
        if (organizationUnitTO.getDiscoveryURL() != null) {
            // Cause there's only one url in ui for the moment
            // Remove old urls before adding it
            organizationUnitBO.getDiscoveryUrls().clear();
            DiscoveryUrl discoveryUrl = new DiscoveryUrl();
            discoveryUrl.setUrl(organizationUnitTO.getDiscoveryURL());
            discoveryUrl.setUseType("homepage");
            organizationUnitBO.addDiscoveryUrl(discoveryUrl);
        }
        organizationUnitBO.setId(organizationUnitTO.getId());
        organizationUnitBO.setLegalEntity(organizationUnitTO.getIsLegalEntity());
        if ((organizationUnitTO.getMotherOrganization() != null)
                && (organizationUnitTO.getMotherOrganization().getId() != null)) {
            final OrganizationUnit motherOrganization = this.organizationUnitDAO
                    .get(organizationUnitTO.getMotherOrganization().getId());
            if (motherOrganization != null) {
                organizationUnitBO.setParentOrganization(motherOrganization);
            } else {
                throw new OrganizationException(
                        "Your are trying to connect an organization to an other non existing organization");
            }
        } else {
            organizationUnitBO.setParentOrganization(null);
        }

        // TODO Only one email for the moment
        // Cause there's only one email in ui for the moment
        // Remove old emails before adding it
        if (organizationUnitTO.getEmailAddress() != null) {
            organizationUnitBO.getEmailAddresses().clear();
            Email email = new Email();
            email.setEmail(organizationUnitTO.getEmailAddress());
            organizationUnitBO.addEmail(email);
        }

        // TODO Set only one name for the moment with langCode set to en
        // Cause there's only one name in ui for the moment
        // Remove old names before adding it
        organizationUnitBO.getNames().clear();
        Name name = new Name();
        name.setLangCode("en");
        name.setName(organizationUnitTO.getName());
        organizationUnitBO.addName(name);

        organizationUnitBO.setType(organizationUnitTO.getType());

    }

    private void setAddresses(OrganizationUnitTO organizationUnitTO,
            OrganizationUnit organizationUnitBO) {
        // TODO Set only one address for the moment
        // Cause there's only one address in ui for the moment
        // Remove old addresses before adding it
        organizationUnitBO.getAddresses().clear();

        Address address = new Address();

        addAddressLine(address, STREET_NUMBER_KEY, null, organizationUnitTO.getStreetNumber());
        addAddressLine(address, STREET_KEY, null, organizationUnitTO.getStreet());
        addAddressLine(address, COMPLEMENT_KEY, null, organizationUnitTO.getComplement());
        addAddressLine(address, COUNTRY_KEY, null, organizationUnitTO.getCountry());
        addAddressLine(address, STATE_KEY, null, organizationUnitTO.getState());
        addAddressLine(address, ZIPCODE_KEY, null, organizationUnitTO.getZipcode());
        addAddressLine(address, CITY_KEY, null, organizationUnitTO.getCity());

        organizationUnitBO.addAddress(address);
    }

    private void addAddressLine(Address address, String keyName, String keyValue, String line) {
        AddressLine addressLine = new AddressLine();
        addressLine.setAddressLine(line);
        addressLine.setKeyName(keyName);
        addressLine.setKeyValue(keyValue);
        address.addAddressLine(addressLine);
    }

    public OrganizationUnitTO toOrganizationUnitTO(final OrganizationUnit organizationUnitBO) {
        OrganizationUnitTO organizationUnitTO = null;
        if (organizationUnitBO != null) {
            organizationUnitTO = new OrganizationUnitTO();

            // Select the first "homepage" url for the moment
            List<DiscoveryUrl> discoveryUrls = organizationUnitBO.getDiscoveryUrls();
            if (discoveryUrls != null && !discoveryUrls.isEmpty()) {
                for (DiscoveryUrl discoveryUrl : discoveryUrls) {
                    if ("homepage".equals(discoveryUrl.getUseType())) {
                        organizationUnitTO.setDiscoveryURL(discoveryUrl.getUrl());
                        break;
                    }
                }
            }
            organizationUnitTO.setId(organizationUnitBO.getId());
            organizationUnitTO.setIsLegalEntity(organizationUnitBO.isLegalEntity());
            organizationUnitTO.setMotherOrganization(this.toOrganizationUnitTO(organizationUnitBO
                    .getParentOrganization()));

            // add categories
            organizationUnitTO.setCategories(this.toCategoriesTO(organizationUnitBO
                    .getCategoryBag()));

            // add identifiers
            organizationUnitTO.setIdentifiers(this.toIdentifiersTO(organizationUnitBO
                    .getIdentifierBag()));

            // TODO For the moment get the first and only one name
            organizationUnitTO.setName(organizationUnitBO.getNames().get(0).getName());

            // TODO Only one email for the moment
            List<Email> emailAddresses = organizationUnitBO.getEmailAddresses();
            if (emailAddresses != null && !emailAddresses.isEmpty()) {
                organizationUnitTO.setEmailAddress(emailAddresses.get(0).getEmail());
            }

            organizationUnitTO.setType(organizationUnitBO.getType());

            setAddressesTO(organizationUnitTO, organizationUnitBO);
        }
        return organizationUnitTO;
    }

    private void setAddressesTO(OrganizationUnitTO organizationUnitTO,
            OrganizationUnit organizationUnitBO) {
        // TODO Only one address for the moment
        Address address = organizationUnitBO.getAddresses().get(0);
        List<AddressLine> addressLines = address.getAddressLines();

        for (AddressLine addressLine : addressLines) {
            if (STREET_NUMBER_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setStreetNumber(addressLine.getAddressLine());
            } else if (STREET_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setStreet(addressLine.getAddressLine());
            } else if (COUNTRY_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setCountry(addressLine.getAddressLine());
            } else if (STATE_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setState(addressLine.getAddressLine());
            } else if (ZIPCODE_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setZipcode(addressLine.getAddressLine());
            } else if (CITY_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setCity(addressLine.getAddressLine());
            } else if (COMPLEMENT_KEY.equals(addressLine.getKeyName())) {
                organizationUnitTO.setComplement(addressLine.getAddressLine());
            }
        }
    }

    /**
     * Create a BO copy of the given TO. Can be used for create or update
     * purposes
     * 
     * @param personTO
     *            must not be null
     * @param personBO
     *            must not be null
     * @throws OrganizationException
     * @throws NullPointerException
     *             if bo or to is null
     */
    public void toPersonBO(final PersonTO personTO, final Person personBO)
            throws OrganizationException {

        // Only one email for the moment
        // Cause there's only one email in ui for the moment
        // Remove old emails before adding it
        personBO.getEmailAddresses().clear();
        Email email = new Email();
        email.setEmail(personTO.getEmail());
        personBO.addEmail(email);

        personBO.setContact(personTO.isContact());
        personBO.setFirstName(personTO.getFirstname());
        personBO.setLastName(personTO.getLastname());
        personBO.setLocation(personTO.getLocalization());
        personBO.setMiddleName(personTO.getMiddlename());

        // TODO Set only one name for the moment with langCode set to en
        // Cause there's only one name in ui for the moment
        // Remove old names before adding it
        personBO.getNames().clear();
        Name name = new Name();
        name.setLangCode("en");
        name.setName(personTO.getFirstname() + " " + personTO.getMiddlename() + " "
                + personTO.getLastname());
        personBO.addName(name);

        if ((personTO.getOrganization() != null) && (personTO.getOrganization().getId() != null)) {
            final OrganizationUnit motherOrganization = this.organizationUnitDAO.get(personTO
                    .getOrganization().getId());
            if (motherOrganization != null) {
                personBO.setParentOrganization(motherOrganization);
            } else {
                throw new OrganizationException(
                        "Your are trying to connect a person to a non existing organization");
            }
        } else {
            personBO.setParentOrganization(null);
        }
        personBO.setId(personTO.getId());

        // TODO Only one phone number for the moment
        // Cause there's only one phone in ui for the moment
        // Remove old phones before adding it
        personBO.getPhoneNumbers().clear();
        PhoneNumber phoneNumber = new PhoneNumber();
        phoneNumber.setPhoneNumber(personTO.getPhone());
        personBO.addPhoneNumber(phoneNumber);

        if (personTO.getPost() != null) {
            final Post post = this.postDAO.get(personTO.getPost().getId());
            if (post != null) {
                personBO.setPost(post);
            } else {
                throw new OrganizationException(
                        "Your are trying to connect a person to a non existing post");
            }
        }
        personBO.setTitle(personTO.getTitle());
    }

    public PersonTO toPersonTO(final Person person) {
        PersonTO personTO = null;
        if (person != null) {
            personTO = new PersonTO();

            // Only one email for the moment
            List<Email> emailAddresses = person.getEmailAddresses();
            if (emailAddresses != null && !emailAddresses.isEmpty()) {
                personTO.setEmail(emailAddresses.get(0).getEmail());
            }

            personTO.setContact(person.isContact());
            personTO.setFirstname(person.getFirstName());
            personTO.setId(person.getId());
            personTO.setLastname(person.getLastName());
            personTO.setLocalization(person.getLocation());
            personTO.setMiddlename(person.getMiddleName());
            personTO.setOrganization(this.toOrganizationUnitTO(person.getParentOrganization()));

            // Only one phone number for the moment
            List<PhoneNumber> phoneNumbers = person.getPhoneNumbers();
            if (phoneNumbers != null && !phoneNumbers.isEmpty()) {
                personTO.setPhone(phoneNumbers.get(0).getPhoneNumber());
            }

            final Post post = person.getPost();
            if (post != null) {
                personTO.setPost(this.toPostTO(post));
            }
            personTO.setTitle(person.getTitle());
        }
        return personTO;
    }

    /**
     * Create a BO copy of the given TO. Can be used for create or update
     * purposes
     * 
     * @param postTO
     *            must not be null
     * @param postBO
     *            must not be null
     * @throws NullPointerException
     *             if bo or to is null
     */
    public void toPostBO(final PostTO postTO, final Post postBO) throws OrganizationException {
        postBO.setDescription(postTO.getDescription());
        postBO.setName(postTO.getName());
        postBO.setNature(postTO.getNature());
        postBO.setId(postTO.getId());
        postBO.setContact(postTO.isContact());
    }

    public PostTO toPostTO(final Post post) {
        PostTO postTO = null;
        if (post != null) {
            postTO = new PostTO();
            postTO.setDescription(post.getDescription());
            postTO.setId(post.getId());
            postTO.setName(post.getName());
            postTO.setNature(post.getNature());
            postTO.setContact(post.isContact());
        }
        return postTO;
    }

    public ProtocolTO toProtocolTO(final Set<Standard> standards) {
        ProtocolTO protocolTO = new ProtocolTO();
        if (standards != null) {
            for (Standard standard : standards) {

            }
        }
        return protocolTO;
    }

    public RequestOptions toPartyRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.PARTY_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("names.name");
                requestOptions.setFetchCriteria("names");
            }
        }
        return requestOptions;
    }

    public RequestOptions toPersonRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.PARTY_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("names.name");
                requestOptions.setFetchCriteria("names");
            } else if (SortCriteria.PERSON_LASTNAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("lastName");
            }
        }
        return requestOptions;
    }

    public RequestOptions toPostRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.POST_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toEndpointRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.ENDPOINT_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toServiceRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.SERVICE_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("fullName");
            }
        }
        return requestOptions;
    }

    public RequestOptions toAgreementRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.AGREEMENT_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toExecEnvManRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.EXEC_ENV_MAN_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toExecEnvRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.EXEC_ENV_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toFederationRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.FED_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toProcessorRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.PROCESSOR_NAME.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("name");
            }
        }
        return requestOptions;
    }

    public RequestOptions toLinkRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = toRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            if (SortCriteria.LINK_TYPE.equals(requestOptionsTO.getSortCriteria())) {
                requestOptions.setSortCriteria("type");
            }
        }
        return requestOptions;
    }

    private RequestOptions toRequestOptions(final RequestOptionsTO requestOptionsTO) {
        RequestOptions result = null;
        if (requestOptionsTO != null) {
            result = new RequestOptions();
            result.setFirstResult(requestOptionsTO.getFirstResult());
            result.setNbOfResults(requestOptionsTO.getNbOfResults());
            result.setSortAscendingly(requestOptionsTO.isSortAscendingly());
            // Doesn't transfer sortCriteria automatically, it's specific to
            // each type of entity
            // result.setSortCriteria(requestOptionsTO.getSortCriteria());
            result.setCaseSensitive(requestOptionsTO.isCaseSensitive());
        }
        return result;
    }

    public ServiceInterfaceTO toServiceInterfaceTO(final ServiceInterface serviceInterface) {
        ServiceInterfaceTO serviceInterfaceTO = null;
        if (serviceInterface != null) {
            serviceInterfaceTO = new ServiceInterfaceTO();
            serviceInterfaceTO.setId(serviceInterface.getId());
            serviceInterfaceTO.setName(serviceInterface.getFullName());
            final Set<SpecifiedOperation> specifiedOperations = serviceInterface.getSpecifiedOps();
            if (specifiedOperations != null) {
                final List<SpecifiedOperationTO> specifiedOperationsTO = new ArrayList<SpecifiedOperationTO>();
                for (final SpecifiedOperation specifiedOperation : specifiedOperations) {
                    specifiedOperationsTO.add(this.toSpecifiedOperationTO(specifiedOperation));
                }
                serviceInterfaceTO.setSpecifiedOps(specifiedOperationsTO);
            }
        }
        return serviceInterfaceTO;
    }

    public SpecifiedOperationTO toSpecifiedOperationTO(final SpecifiedOperation specifiedOperation) {
        SpecifiedOperationTO specifiedOperationTO = null;
        if (specifiedOperation != null) {
            specifiedOperationTO = new SpecifiedOperationTO();
            specifiedOperationTO.setId(specifiedOperation.getId());
            specifiedOperationTO.setIsAtomicTransaction(specifiedOperation.isAtomicTransaction());
            specifiedOperationTO.setIsCanParticipateInTransaction(specifiedOperation
                    .isCanParticipateInTransaction());
            specifiedOperationTO.setIsNoneUpdating(specifiedOperation.isNonUpdating());
            specifiedOperationTO.setIsTransactionCoordNeeded(specifiedOperation
                    .isTransactionCoordNeeded());
            specifiedOperationTO.setName(specifiedOperation.getName());
            specifiedOperationTO.setPurpose(specifiedOperation.getPurpose());
            specifiedOperationTO.setTransactionCoordIsBy(specifiedOperation
                    .getTransactionCoordIsBy());
            specifiedOperationTO.setTransmission(specifiedOperation.getTransmission().toString());
        }
        return specifiedOperationTO;
    }

    /**
     * Create a BO copy of the given TO. Can be used for create or update
     * purposes
     * 
     * @param technicalServiceTO
     *            must not be null
     * @param technicalServiceBO
     *            must not be null
     * @throws NullPointerException
     *             if bo or to is null
     */
    public void toTechServiceBO(final TechnicalServiceTO technicalServiceTO,
            final TechnicalService technicalServiceBO) {
        /*
         * setServiceCategory(technicalServiceBO, KRKeyNames.SERVICE_CATEGORY,
         * technicalServiceTO .getCategory());
         */
        setName(technicalServiceBO, technicalServiceTO.getName());
        setPurpose(technicalServiceBO, technicalServiceTO.getPurpose());

        // Added By TIAR Abderrahmane
        setRating(technicalServiceBO, technicalServiceTO.getRatingTO());
        setComments(technicalServiceBO, technicalServiceTO.getComments());
        technicalServiceBO.setTags(technicalServiceTO.getTags());

        // technicalServiceBO.setId(technicalServiceTO.getId());
    }

    private void setComments(TechnicalService technicalServiceBO, List<CommentTO> comments) {
        List<Comment> commentsBO = new ArrayList<Comment>();
        toCommentsBO(comments, commentsBO);
        technicalServiceBO.setComments(commentsBO);
    }

    private void setRating(TechnicalService technicalServiceBO, RatingTO rating) {
        Iterator<Rating> iter = technicalServiceBO.getRatings().iterator();
        boolean found = false;
        // If the user already voted, we change the value
        while (iter.hasNext() && !found) {
            Rating ratingTemp = iter.next();
            if (ratingTemp.getUserName().compareTo(rating.getUserName()) == 0) {
                ratingTemp.setMyRating(rating.getMyRating());
                found = true;
            }
        }
        // If it's his first vote, we check if he really voted, (vote > 0 )
        if (!found && rating.getMyRating() > 0) {
            Rating ratingBO = new Rating(rating.getUserName(), rating.getMyRating());
            technicalServiceBO.getRatings().add(ratingBO);
        }
    }

    private void setPurpose(TechnicalService technicalServiceBO, String purpose) {
        // Cause there's only one desc in ui for the moment
        // Remove old descs before adding it
        technicalServiceBO.getDescriptions().clear();

        Description description = new Description();
        description.setLangCode("en");
        description.setDescription(purpose);
        technicalServiceBO.addDescription(description);
    }

    private void setName(TechnicalService technicalServiceBO, String name) {
        technicalServiceBO.setFullName(name);

        // Cause there's only one name in ui for the moment
        // Remove old names before adding it
        technicalServiceBO.getNames().clear();

        QName servQname = QName.valueOf(name);
        Name nameObj = new Name();
        nameObj.setLangCode("en");
        nameObj.setName(servQname.getLocalPart());
        technicalServiceBO.addName(nameObj);

        // Update categories
        setServiceCategory(technicalServiceBO, KRKeyNames.SERVICE_LOCAL_NAME, servQname
                .getLocalPart());
        setServiceCategory(technicalServiceBO, KRKeyNames.SERVICE_NS, servQname.getNamespaceURI());
    }

    private void setServiceCategory(TechnicalService service, String categoryName,
            String categoryValue) {
        CategoryBag bag = service.getCategoryBag();
        if (bag == null) {
            bag = new CategoryBag();
            service.setCategoryBag(bag);
        }
        List<KeyedReference> keyedReferences = bag.getKeyedReferences();
        boolean serviceCategoryExist = false;
        if (keyedReferences != null) {
            for (KeyedReference keyedReference : keyedReferences) {
                if (categoryName.equals(keyedReference.getKeyName())) {
                    keyedReference.setKeyValue(categoryValue);
                    serviceCategoryExist = true;
                    break;
                }
            }
        }
        if (!serviceCategoryExist) {
            // TODO Adding a tModel for dragon categories
            addKeyedReference(bag, categoryName, categoryValue, null);
        }
    }

    private void addKeyedReference(CategoryBag bag, String name, String value, TModel tModel) {
        KeyedReference keyedReference = new KeyedReference();
        keyedReference.setKeyName(name);
        keyedReference.setKeyValue(value);
        keyedReference.setTmodel(tModel);
        bag.addKeyedReference(keyedReference);
    }

    public TechnicalServiceTO toTechServiceTO(final TechnicalService technicalService,
            final String user) {
        TechnicalServiceTO technicalServiceTO = null;
        if (technicalService != null) {
            technicalServiceTO = new TechnicalServiceTO();
            technicalServiceTO.setId(technicalService.getId());

            // Get the one called : "service category"
            /*
             * CategoryBag bag = technicalService.getCategoryBag(); if (bag !=
             * null) { List<KeyedReference> keyedReferences =
             * bag.getKeyedReferences(); if (keyedReferences != null) { for
             * (KeyedReference keyedReference : keyedReferences) { if
             * (KRKeyNames.SERVICE_CATEGORY.equals(keyedReference.getKeyName()))
             * { technicalServiceTO.setCategory(keyedReference.getKeyValue());
             * break; } } } }
             */

            technicalServiceTO.setName(technicalService.getFullName());

            // Only one for the moment
            List<Description> descriptions = technicalService.getDescriptions();
            if (descriptions != null && !descriptions.isEmpty()) {
                technicalServiceTO.setPurpose(descriptions.get(0).getDescription());
            }

            // Added By TIAR Abderrahmane
            // retreive the rating of this user and the globalRating
            if (user != null) {
                Set<Rating> ratings = technicalService.getRatings();
                Rating userRating = null;
                float globalRating = 0f;
                Iterator<Rating> iter = ratings.iterator();
                while (iter.hasNext()) {
                    Rating ratingTemp = iter.next();
                    globalRating = globalRating + ratingTemp.getMyRating();
                    if (ratingTemp.getUserName().compareTo(user) == 0) {
                        userRating = ratingTemp;
                    }
                }
                // if the user never rated or opened this service
                if (userRating == null) {
                    userRating = new Rating(user);
                }
                if (ratings.size() != 0) {
                    technicalServiceTO.setGlobalRating(globalRating / ratings.size());
                } else {
                    technicalServiceTO.setGlobalRating(0);
                }
                technicalServiceTO.setRatingTO(this.toRatingTO(userRating));
            }

            // add categories
            technicalServiceTO
                    .setCategories(this.toCategoriesTO(technicalService.getCategoryBag()));

            // add the comments
            technicalServiceTO.setComments(this.toCommentsTO(technicalService.getComments()));
            // add the tags
            technicalServiceTO.setTags(new HashSet<String>(technicalService.getTags()));
        }
        return technicalServiceTO;
    }

    public List<KeyedRefTO> toCategoriesTO(CategoryBag categoryBag) {
        List<KeyedRefTO> result = new ArrayList<KeyedRefTO>();
        if (categoryBag != null) {
            // TODO Use only direct keyed ref for the moment
            List<KeyedReference> keyedRefs = categoryBag.getKeyedReferences();
            if (keyedRefs != null) {
                for (KeyedReference keyedReference : keyedRefs) {
                    KeyedRefTO keyedRefTO = new KeyedRefTO();
                    if (keyedReference.getTmodel() != null) {
                        keyedRefTO.settModelName(keyedReference.getTmodel().getName().getName());
                    } else {
                        keyedRefTO.settModelName("Undefined");
                    }
                    keyedRefTO.setKefRefId(keyedReference.getId());
                    keyedRefTO.setName(keyedReference.getKeyName());
                    keyedRefTO.setValue(keyedReference.getKeyValue());
                    result.add(keyedRefTO);
                }
            }
        }
        return result;
    }

    public List<KeyedRefTO> toIdentifiersTO(List<KeyedReference> identifierBag) {
        List<KeyedRefTO> result = new ArrayList<KeyedRefTO>();
        if (identifierBag != null) {
            for (KeyedReference keyedReference : identifierBag) {
                KeyedRefTO keyedRefTO = new KeyedRefTO();
                if (keyedReference.getTmodel() != null) {
                    keyedRefTO.settModelName(keyedReference.getTmodel().getName().getName());
                } else {
                    keyedRefTO.settModelName("Undefined");
                }
                keyedRefTO.setKefRefId(keyedReference.getId());
                keyedRefTO.setName(keyedReference.getKeyName());
                keyedRefTO.setValue(keyedReference.getKeyValue());
                result.add(keyedRefTO);
            }
        }
        return result;
    }

    public SimpleFileTO toSimpleFileTO(final SimpleFile simpleFile) {
        SimpleFileTO simpleFileTO = null;
        if (simpleFile != null) {
            simpleFileTO = new SimpleFileTO();
            simpleFileTO.setAuthor(simpleFile.getAuthor());
            if (simpleFile.getFileType() != null) {
                simpleFileTO.setFileType(simpleFile.getFileType().toString());
            }
            simpleFileTO.setId(simpleFile.getId());
            simpleFileTO.setTitle(simpleFile.getTitle());
            simpleFileTO.setFileName(simpleFile.getFileName());
        }
        return simpleFileTO;
    }

    // Added By TIAR Abderrahmane
    public RatingTO toRatingTO(final Rating rating) {
        RatingTO ratingTO = null;
        if (rating != null) {
            ratingTO = new RatingTO();
            ratingTO.setId(rating.getId());
            ratingTO.setMyRating(rating.getMyRating());
            ratingTO.setUserName(rating.getUserName());
        }
        return ratingTO;
    }

    public void toRatingBO(final RatingTO ratingTO, final Rating ratingBO) {
        ratingBO.setId(ratingTO.getid());
        ratingBO.setMyRating(ratingTO.getMyRating());
        ratingBO.setUserName(ratingTO.getUserName());
    }

    public List<CommentTO> toCommentsTO(final List<Comment> comments) {
        List<CommentTO> commentsTO = null;
        if (comments != null) {
            commentsTO = new ArrayList<CommentTO>();
            Iterator<Comment> iter = comments.iterator();
            while (iter.hasNext()) {
                Comment commentTemp = iter.next();
                CommentTO commentTOTemp = new CommentTO();

                commentTOTemp.setContent(commentTemp.getContent());
                commentTOTemp.setId(commentTemp.getId());
                commentTOTemp.setUserName(commentTemp.getUserName());
                commentTOTemp.setDate(commentTemp.getDate());

                commentsTO.add(0, commentTOTemp);
            }
        }
        return commentsTO;
    }

    public void toCommentsBO(final List<CommentTO> commentsTO, final List<Comment> commentsBO) {
        Iterator<CommentTO> iter = commentsTO.iterator();
        commentsBO.clear();

        while (iter.hasNext()) {
            Comment commentTemp = new Comment();
            CommentTO commentTOTemp = iter.next();

            commentTemp.setContent(commentTOTemp.getContent());
            commentTemp.setDate(commentTOTemp.getDate());
            commentTemp.setUserName(commentTOTemp.getUserName());
            commentTemp.setId(commentTOTemp.getId());

            commentsBO.add(commentTemp);
        }
    }

    private void setComments(Endpoint endpointBO, List<CommentTO> comments) {
        List<Comment> commentsBO = new ArrayList<Comment>();
        toCommentsBO(comments, commentsBO);
        endpointBO.setComments(commentsBO);
    }

    private void setRating(Endpoint endpointBO, RatingTO rating) {
        Iterator<Rating> iter = endpointBO.getRatings().iterator();
        boolean found = false;
        // If the user already voted, we change the value
        while (iter.hasNext() && !found) {
            Rating ratingTemp = iter.next();
            if (ratingTemp.getUserName().compareTo(rating.getUserName()) == 0) {
                ratingTemp.setMyRating(rating.getMyRating());
                found = true;
            }
        }
        // If it's his first vote, we check if he really voted, (vote > 0 )
        if (!found && rating.getMyRating() > 0) {
            Rating ratingBO = new Rating(rating.getUserName(), rating.getMyRating());
            endpointBO.getRatings().add(ratingBO);
        }
    }

    public void toEndpointBO(final EndpointTO endpointTO, final Endpoint endpointBO) {
        // For the moment there just the ratings, comments and tags to set.
        setComments(endpointBO, endpointTO.getComments());
        setRating(endpointBO, endpointTO.getRatingTO());
        endpointBO.setTags(endpointTO.getTags());
    }

    public IdentifierTO toIdentifierTO(Identifier identifierBO) {
        IdentifierTO identifierTO = null;
        if (identifierBO != null) {
            identifierTO = new IdentifierTO(identifierBO.getId(), identifierBO.getName().getName(),
                    null);
            List<Description> descriptionsBO = identifierBO.getDescriptions();
            if (descriptionsBO != null && !descriptionsBO.isEmpty()) {
                // TODO get only the first description for the moment
                identifierTO.setDescription(descriptionsBO.get(0).getDescription());
            }
            // categoryTO.setValues(toCategoryValuesTO(categoryBO.getValues()));
        }
        return identifierTO;
    }

    public CategoryTO toCategoryTO(Category categoryBO) {
        CategoryTO categoryTO = null;
        if (categoryBO != null) {
            categoryTO = new CategoryTO(categoryBO.getId(), categoryBO.getName().getName(), null);
            List<Description> descriptionsBO = categoryBO.getDescriptions();
            if (descriptionsBO != null && !descriptionsBO.isEmpty()) {
                // TODO get only the first description for the moment
                categoryTO.setDescription(descriptionsBO.get(0).getDescription());
            }
            // categoryTO.setValues(toCategoryValuesTO(categoryBO.getValues()));
        }
        return categoryTO;
    }

    public List<CategoryValueTO> toCategoryValuesTO(List<CategoryValue> valuesBO) {
        List<CategoryValueTO> result = new ArrayList<CategoryValueTO>();
        if (valuesBO != null && !valuesBO.isEmpty()) {
            for (CategoryValue categoryValueBO : valuesBO) {
                result.add(toCategoryValueTO(categoryValueBO));
            }
        }
        return result;
    }

    private CategoryValueTO toCategoryValueTO(CategoryValue categoryValueBO) {
        CategoryValueTO categoryValueTO = null;
        if (categoryValueBO != null) {
            categoryValueTO = new CategoryValueTO(categoryValueBO.getId(), categoryValueBO
                    .getValue(), categoryValueBO.getDescription());
        }
        return categoryValueTO;
    }

    public ProtocolTO toProtocolTO(Protocol protocol) {
        ProtocolTO result = null;
        if (protocol != null) {
            result = new ProtocolTO();
            result.setName(protocol.getName().getName());
            result.setId(protocol.getId());
        }
        return result;
    }

    public TransportTO toTransportTO(Transport transport) {
        TransportTO result = null;
        if (transport != null) {
            result = new TransportTO();
            result.setName(transport.getName().getName());
            result.setId(transport.getId());
        }
        return result;
    }
}
