/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EntityLinkManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.common;

import java.util.List;

import org.apache.log4j.Logger;
import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.common.CommonException;
import org.ow2.dragon.api.service.common.EntityLinkManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.persistence.bo.common.Link;
import org.ow2.dragon.persistence.bo.common.LinkedEntity;
import org.ow2.dragon.persistence.dao.RequestOptions;
import org.ow2.dragon.persistence.dao.UniversalORMDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;

import com.trg.search.Filter;
import com.trg.search.Search;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class EntityLinkManagerImpl implements EntityLinkManager {

    private Logger logger = Logger.getLogger(this.getClass());

    private UniversalORMDAO universalORMDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    @CheckAllArgumentsNotNull
    public String createLink(final String fromId, final String toId, final String type)
            throws CommonException {

        // Check if the link already exist
        Link link = linkExist(fromId, toId, type);
        if (link != null) {
            logger.debug("Link already exist : " + link.toString() + ". Skip creation !");
        } else {

            // Retrieve from
            final LinkedEntity from = this.retrieveLinkableEntity(fromId);

            // Retrieve to
            final LinkedEntity to = this.retrieveLinkableEntity(toId);

            // Create and persist Role
            link = new Link();
            link.setType(type);
            from.addFromLink(link);
            to.addToLink(link);

            this.universalORMDAO.save(link);
        }

        return link.getId();
    }

    private Link linkExist(final String fromId, final String toId, final String type) {
        Search search = new Search();
        search.addFilter(Filter.equal("from.id", fromId));
        search.addFilter(Filter.equal("to.id", toId));
        search.addFilter(Filter.equal("type", type));
        return (Link) universalORMDAO.searchUnique(Link.class, search);
    }

    private LinkedEntity retrieveLinkableEntity(String linkableId) throws CommonException {
        final LinkedEntity linkable = (LinkedEntity) this.universalORMDAO.get(LinkedEntity.class,
                linkableId);
        if (linkable == null) {
            throw new CommonException("One of the link end doesn't exist. Id: " + linkableId);
        }
        return linkable;
    }

    @CheckAllArgumentsNotNull
    public void removeLink(final String linkId) {
        this.universalORMDAO.remove(Link.class, linkId);
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    public void setUniversalORMDAO(UniversalORMDAO universalORMDAO) {
        this.universalORMDAO = universalORMDAO;
    }

    @CheckArgumentsNotNull
    @SuppressWarnings("unchecked")
    public List<Link> getAllLinksForFrom(String fromId, RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = transfertObjectAssembler
                .toLinkRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            requestOptions.setCaseSensitive(true);
        }

        List<Link> links = this.universalORMDAO.searchEquals(Link.class, new String[] { fromId },
                new String[] { "from.id" }, requestOptions);

        return links;
    }

    @CheckArgumentsNotNull
    @SuppressWarnings("unchecked")
    public List<Link> getAllLinksForTo(String toId, RequestOptionsTO requestOptionsTO) {
        RequestOptions requestOptions = transfertObjectAssembler
                .toLinkRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            requestOptions.setCaseSensitive(true);
        }

        List<Link> links = this.universalORMDAO.searchEquals(Link.class, new String[] { toId },
                new String[] { "to.id" }, requestOptions);

        return links;
    }
}
