/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * DatasetReader.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.dataloader;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.Source;

import org.ow2.dragon.api.service.dataloader.DataLoaderException;
import org.ow2.dragon.dataset.Dataset;
import org.ow2.dragon.util.JAXBUtil;

/**
 * @author ofabre - eBM Websourcing
 * 
 */
public class DragonDatasetHandler {

    private static DragonDatasetHandler instance;

    /**
     * The JAXB unique unmarshaller.
     */
    private static Unmarshaller unmarshaller;

    private static Marshaller marshaller;

    // TODO Validate schema with Schema Factory
    // private static final String DRAGON_DATASET_XSD =
    // "classpath:dragon-dataset.xsd";

    @SuppressWarnings("unchecked")
    private static final List<Class> defaultObjectFactories = new ArrayList<Class>(Arrays
            .asList(new Class[] { org.ow2.dragon.dataset.ObjectFactory.class }));

    /*
     * Private object initializations
     */
    private DragonDatasetHandler() throws DataLoaderException {
        try {
            JAXBContext context = JAXBUtil.createJAXBContext(defaultObjectFactories);
            unmarshaller = context.createUnmarshaller();
            marshaller = context.createMarshaller();
        } catch (final JAXBException e) {
            throw new DataLoaderException(e);
        }
    }

    /**
     * Build the Java bindings from the Dragon dataset in XML form. The
     * validation against the dataset schema is processed during the
     * transformation.
     * 
     * @param datasetStream
     *            The Dragon dataset in XML form
     * @return The root object of the dataset
     * @throws DataLoaderException
     *             The exception raised during the unmarshaller creation or the
     *             exception raised during the build of the java bindings.
     */
    public Dataset convertStreamSource2DragonDataset(final Source datasetStream)
            throws DataLoaderException {

        try {
            // TODO : Check if it is a Thread safe method
            final JAXBElement<Dataset> wsdlBinding = unmarshaller.unmarshal(datasetStream,
                    Dataset.class);

            return wsdlBinding.getValue();

        } catch (final JAXBException e) {
            throw new DataLoaderException(
                    "Failed to build Java bindings from dataset XML document", e);
        }
    }

    public static DragonDatasetHandler getInstance() throws DataLoaderException {
        if (instance == null) {
            instance = new DragonDatasetHandler();
        }
        return instance;
    }

}
