/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechServiceManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.deployment;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.deployment.TechServiceManager;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.ServiceSearchProperties;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.metadata.SimpleFileTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Comment;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Rating;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.dao.deployment.BindingDAO;
import org.ow2.dragon.persistence.dao.deployment.CommentDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceInterfaceDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceSpecificationDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.StringHelper;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class TechServiceManagerImpl implements TechServiceManager {

    private GenericUnifiedDAO<TechnicalService, String> technicalServiceUnifiedDAO;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    private MetadataService metadataService;

    private ServiceSpecificationDAO serviceSpecificationDAO;

    private BindingDAO bindingDAO;

    private ServiceInterfaceDAO serviceInterfaceDAO;

    // Added By TIAR Abderrahmane
    private CommentDAO commentDAO;

    public void setCommentDAO(final CommentDAO commentDAO) {
        this.commentDAO = commentDAO;
    }

    // End

    private String[] createSearchProperties(List<ServiceSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(ServiceSearchProperties.NAME)) {
                propertiesList.add("fullName");
            }
            if (searchedProperties.contains(ServiceSearchProperties.CATEGORY)) {
                propertiesList.add("categoryBag.keyedReferences.keyValue");
                propertiesList.add("categoryBag.keyedReferences.keyName");
            }
            if (searchedProperties.contains(ServiceSearchProperties.PURPOSE)) {
                propertiesList.add("descriptions.description");
            }
            if (searchedProperties.contains(ServiceSearchProperties.OPERATIONS)) {
                propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.name");
                propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.purpose");
            }
            if (searchedProperties.contains(ServiceSearchProperties.PROTOCOL)) {
                propertiesList.add("endpoints.binding.protocols.name.name");
                propertiesList.add("endpoints.binding.transports.name.name");
            }
            if (searchedProperties.contains(ServiceSearchProperties.ORGANIZATION)) {
                propertiesList.add("toLinks.type");
                propertiesList.add("toLinks.from.names.name");
            }
            if (searchedProperties.contains(ServiceSearchProperties.RELATED_DOCS)) {
                propertiesList.add("relatedDocs.author");
                propertiesList.add("relatedDocs.title");
                propertiesList.add("relatedDocs.extractedContent");
            }
        } else {
            propertiesList.add("fullName");
            propertiesList.add("categoryBag.keyedReferences.keyValue");
            propertiesList.add("categoryBag.keyedReferences.keyName");
            propertiesList.add("descriptions.description");
            propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.name");
            propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.purpose");
            propertiesList.add("endpoints.binding.protocols.name.name");
            propertiesList.add("endpoints.binding.transports.name.name");
            propertiesList.add("toLinks.type");
            propertiesList.add("toLinks.from.names.name");
            propertiesList.add("relatedDocs.author");
            propertiesList.add("relatedDocs.title");
            propertiesList.add("relatedDocs.extractedContent");
        }
        // TODO add search on person
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * createTechService
     * (org.ow2.dragon.ui.model.to.deployment.TechnicalServiceTO)
     */
    public String createTechService(final TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        throw new RuntimeException("not implemented method");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * getAllTechServices(org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<TechnicalServiceTO> getAllTechServices(final RequestOptionsTO requestOptionsTO)
            throws DeploymentException {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();
        final List<TechnicalService> techServices = this.technicalServiceUnifiedDAO
                .getAll(this.transfertObjectAssembler.toServiceRequestOptions(requestOptionsTO));
        toTechServicesTO(result, techServices);
        return result;
    }

    private void toTechServicesTO(List<TechnicalServiceTO> result,
            final Collection<TechnicalService> techServices) {
        if ((techServices != null) && !techServices.isEmpty()) {
            for (final TechnicalService techServ : techServices) {
                TechnicalServiceTO techTO = toTechServiceTO(techServ, null);
                techTO.setGlobalRating(updateGlobalRating(techTO.getId()));
                result.add(techTO);
            }
        }
    }

    private TechnicalServiceTO toTechServiceTO(final TechnicalService techServ, String user) {
        return this.transfertObjectAssembler.toTechServiceTO(techServ, user);
    }

    public EndpointTO getEndpoint(final String techServiceName, final String endpointName)
            throws DeploymentException {
        EndpointTO res = null;
        final List<EndpointTO> results = new ArrayList<EndpointTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(techServiceName);

        // Search for bo
        final List<TechnicalService> techServicesBO = this.technicalServiceUnifiedDAO.searchLike(
                criteria, null, null);

        for (final TechnicalService service : techServicesBO) {
            // Create result array
            final Set<Endpoint> endpointsBO = service.getEndpoints();
            if ((endpointsBO != null) && !endpointsBO.isEmpty()) {
                for (final Endpoint endpoint : endpointsBO) {
                    if (endpoint.getName().equals(endpointName)) {
                        results.add(toEndpointTO(endpoint));
                    }
                }
            }
        }

        if (results.size() == 1) {
            res = results.get(0);
        }

        return res;
    }

    private EndpointTO toEndpointTO(final Endpoint endpoint) {
        return this.transfertObjectAssembler.toEndpointTO(endpoint, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * getTechService(java.lang.String)
     */
    public TechnicalServiceTO getTechService(final String techServiceId, String user)
            throws DeploymentException {
        // retrieve tech serv bo
        final TechnicalService technicalServiceBO = this.technicalServiceUnifiedDAO
                .get(techServiceId);
        // create tech serv to from bo
        final TechnicalServiceTO techServiceTO = toTechServiceTO(technicalServiceBO, user);
        return techServiceTO;
    }

    public List<TechnicalServiceTO> loadServicesFromWSDL(final String[] servicesId,
            final RequestOptionsTO requestOptionsTO) throws DeploymentException {
        // Doesn't use requestOption for the moment
        final List<TechnicalServiceTO> techServs = new ArrayList<TechnicalServiceTO>();
        if (servicesId != null) {
            for (final String servId : servicesId) {
                final TechnicalServiceTO serviceTO = this.getTechService(servId, null);
                if (serviceTO != null) {
                    techServs.add(serviceTO);
                }
            }
        }
        return techServs;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * removeTechService(java.lang.String)
     */
    public void removeTechService(final String techServiceId) throws DeploymentException {
        final TechnicalService technicalService = this.technicalServiceUnifiedDAO
                .get(techServiceId);
        if (technicalService != null) {
            final Set<ServiceSpecification> serviceSpecifications = technicalService
                    .getServiceSpecifications();
            if (serviceSpecifications != null) {
                for (ServiceSpecification serviceSpecification : serviceSpecifications) {

                    // Delete Tech Services
                    deleteTechServices(serviceSpecification);

                    // Delete Bindings
                    deleteBindings(serviceSpecification);

                    // Delete Service Interfaces
                    deleteServiceInterfaces(serviceSpecification);

                    // Delete Service Spec
                    this.serviceSpecificationDAO.remove(serviceSpecification.getId());
                }
            }
        }
    }

    private void deleteServiceInterfaces(final ServiceSpecification serviceSpecification) {
        final Set<ServiceInterface> interfaces = serviceSpecification.getServiceInterfaces();
        if (interfaces != null) {
            for (final ServiceInterface serviceInterface : interfaces) {
                this.serviceInterfaceDAO.remove(serviceInterface.getId());
            }
        }
    }

    private void deleteBindings(final ServiceSpecification serviceSpecification) {
        final Set<Binding> bindings = serviceSpecification.getBindings();
        if (bindings != null) {
            for (final Binding binding : bindings) {
                this.bindingDAO.remove(binding.getId());
            }
        }
    }

    private void deleteTechServices(final ServiceSpecification serviceSpecification) {
        final Set<TechnicalService> techServs = serviceSpecification.getTechnicalServices();
        if (techServs != null) {
            for (final TechnicalService technicalService2 : techServs) {
                this.technicalServiceUnifiedDAO.remove(technicalService2.getId());
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * searchTechService(java.lang.String, boolean, boolean, boolean, boolean,
     * boolean, boolean, org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<TechnicalServiceTO> searchTechService(final String searchCriteria,
            final List<ServiceSearchProperties> searchedProperties, final RequestOptionsTO options)
            throws DeploymentException {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<TechnicalService> techServices;
        try {
            techServices = this.technicalServiceUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toServiceRequestOptions(options));
        } catch (DAOLayerException e) {
            throw new DeploymentException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toTechServicesTO(result, techServices);
        return result;
    }

    public void setBindingDAO(final BindingDAO bindingDAO) {
        this.bindingDAO = bindingDAO;
    }

    public void setMetadataService(final MetadataService metadataService) {
        this.metadataService = metadataService;
    }

    public void setServiceInterfaceDAO(final ServiceInterfaceDAO serviceInterfaceDAO) {
        this.serviceInterfaceDAO = serviceInterfaceDAO;
    }

    public void setServiceSpecificationDAO(final ServiceSpecificationDAO serviceSpecificationDAO) {
        this.serviceSpecificationDAO = serviceSpecificationDAO;
    }

    public void setTechnicalServiceUnifiedDAO(
            final GenericUnifiedDAO<TechnicalService, String> technicalServiceUnifiedDAO) {
        this.technicalServiceUnifiedDAO = technicalServiceUnifiedDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * updateTechService
     * (org.ow2.dragon.ui.model.to.deployment.TechnicalServiceTO)
     */
    public String updateTechService(final TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        // Retrieve tech serv
        final TechnicalService technicalService = this.technicalServiceUnifiedDAO.get(techServiceTO
                .getId());
        if (technicalService == null) {
            throw new DeploymentException(
                    "Your are trying to update a non existing Technical Service with id: "
                            + techServiceTO.getId());
        }

        // update fields
        this.transfertObjectAssembler.toTechServiceBO(techServiceTO, technicalService);
        this.technicalServiceUnifiedDAO.save(technicalService);

        return technicalService.getId();
    }

    public String getRelatedDocContentAsString(String relatedDocId) throws DeploymentException {
        String result = null;
        try {
            InputStream inputStream = this.metadataService
                    .loadMetadataContentAsInputStream(relatedDocId);
            result = StringHelper.toString(inputStream);
        } catch (final MetadataServiceException e) {
            throw new DeploymentException("Can't load related doc content", e);
        } catch (final IOException e) {
            throw new DeploymentException("Can't convert related doc content to String", e);
        }
        return result;
    }

    public InputStream getRelatedDocContent(String relatedDocId) throws DeploymentException {
        InputStream result = null;
        try {
            result = this.metadataService.loadMetadataContentAsInputStream(relatedDocId);
        } catch (final MetadataServiceException e) {
            throw new DeploymentException("Can't load related doc content", e);
        }
        return result;
    }

    public List<SimpleFileTO> getRelatedDocsForTechServ(String techServiceId)
            throws DeploymentException {
        List<SimpleFileTO> result = new ArrayList<SimpleFileTO>();

        // retrieve tech serv bo
        final TechnicalService technicalServiceBO = this.technicalServiceUnifiedDAO
                .get(techServiceId);

        Set<SimpleFile> relatedDocs = technicalServiceBO.getRelatedDocs();
        for (SimpleFile simpleFile : relatedDocs) {
            result.add(transfertObjectAssembler.toSimpleFileTO(simpleFile));
        }

        return result;
    }

    private void addRelatedDocs(String serviceId, Set<SimpleFile> relatedDocs) {
        if (!relatedDocs.isEmpty()) {
            TechnicalService service = technicalServiceUnifiedDAO.get(serviceId);
            for (SimpleFile relatedDoc : relatedDocs) {
                service.addRelatedDoc(relatedDoc);
            }
            technicalServiceUnifiedDAO.save(service);
        }
    }

    public String registerRelatedDoc(String serviceId, String mimetype, byte[] docContent)
            throws DeploymentException {
        SimpleFile doc = null;
        try {
            doc = metadataService.storeMetadataAndIndexContent(mimetype, docContent);
        } catch (MetadataServiceException e) {
            throw new DeploymentException("Can't store document in registry", e);
        }
        Set<SimpleFile> docs = new HashSet<SimpleFile>();
        docs.add(doc);
        addRelatedDocs(serviceId, docs);
        return doc.getId();
    }

    public void removeRelatedDoc(String serviceId, String documentId) {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);
        SimpleFile simpleFile = metadataService.loadMetadata(documentId);
        technicalService.getRelatedDocs().remove(simpleFile);
        metadataService.deleteMetadata(documentId);
    }

    @CheckArgumentsNotNull
    public void addCategory(@NotNullParam String techServiceId, @NotNullParam String categoryId,
            @NotNullParam String categoryValue, String categoryDesc) throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        if (service != null && category != null) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(categoryDesc);
            keyedReference.setKeyValue(categoryValue);
            this.validateCategory(service, keyedReference);

            CategoryBag categoryBag = service.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                service.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown service, category system or category value");
        }

        technicalServiceUnifiedDAO.save(service);
    }

    @CheckAllArgumentsNotNull
    public void addCategory(String techServiceId, String categoryId, String categoryValueId)
            throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        // Retrieve category value
        CategoryValue value = (CategoryValue) universalUnifiedDAO.get(CategoryValue.class,
                categoryValueId);

        if (service != null && category != null) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(value.getDescription());
            keyedReference.setKeyValue(value.getValue());
            this.validateCategory(service, keyedReference);

            CategoryBag categoryBag = service.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                service.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown service, category system or category value");
        }

        technicalServiceUnifiedDAO.save(service);
    }

    private void validateCategory(TechnicalService techServ, KeyedReference keyedReference)
            throws DeploymentException {
        CategoryBag categoryBag = techServ.getCategoryBag();
        if (categoryBag != null) {
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            if (keyedReferences != null) {
                if (keyedReferences.contains(keyedReference)) {
                    throw new DeploymentException("Category already added to this service.");
                }
            }
        }
    }

    public void removeCategories(String techServiceId, List<String> categoryIds) {
        if (categoryIds != null) {
            // Retrieve service
            TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);

            // Unlink all categories
            List<KeyedReference> keyRefs = universalUnifiedDAO.getAll(KeyedReference.class,
                    categoryIds);
            service.getCategoryBag().getKeyedReferences().removeAll(keyRefs);

            // Delete categories
            universalUnifiedDAO.removeAll(keyRefs);
        }
    }

    // Added By TIAR Abderrahmane
    public float updateGlobalRating(String serviceId) {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);
        float globalRating = 0f;
        if (!technicalService.getRatings().isEmpty()) {
            Iterator<Rating> iter = technicalService.getRatings().iterator();
            int size = technicalService.getRatings().size();
            while (iter.hasNext()) {
                globalRating = globalRating + iter.next().getMyRating();
            }
            globalRating = globalRating / size;
        }
        return globalRating;
    }

    public void updateComments(String serviceId, CommentTO comment) {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);

        if (technicalService != null) {
            final Comment commentTemp = new Comment();

            commentTemp.setContent(comment.getContent());
            commentTemp.setDate(comment.getDate());
            commentTemp.setUserName(comment.getUserName());
            // commentTemp.setId(comment.getId()) ;

            technicalService.addComment(commentTemp);
        }
    }

    public List<TechnicalServiceTO> getTechServicesByTag(String tag) {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();
        final List<TechnicalService> allTechServices = this.technicalServiceUnifiedDAO.getAll();
        final List<TechnicalService> techServices = new ArrayList<TechnicalService>();

        Iterator<TechnicalService> iter = allTechServices.iterator();

        while (iter.hasNext()) {
            final TechnicalService techServiceTemp = iter.next();
            if (techServiceTemp.isTagged(tag)) {
                techServices.add(techServiceTemp);
            }
        }
        toTechServicesTO(result, techServices);
        return result;
    }

    public void removeTag(String tag, String serviceId) {
        TechnicalService techService = this.technicalServiceUnifiedDAO.get(serviceId);
        Set<String> tags = techService.getTags();
        tags.remove(tag);
        techService.setTags(tags);
    }

    public void removeComment(String serviceId, String commentId) {
        TechnicalService techService = this.technicalServiceUnifiedDAO.get(serviceId);
        Comment comment = this.commentDAO.get(commentId);
        List<Comment> comments = techService.getComments();

        comments.remove(comment);
        techService.setComments(comments);
    }

    public void setUniversalUnifiedDAO(UniversalUnifiedDAO universalUnifiedDAO) {
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

    public List<KeyedRefTO> getCategoriesForTechServ(String techServiceId)
            throws DeploymentException {
        TechnicalService service = this.technicalServiceUnifiedDAO.get(techServiceId);
        if (service == null) {
            throw new DeploymentException("No service found for the given id: " + techServiceId);
        }

        return transfertObjectAssembler.toCategoriesTO(service.getCategoryBag());
    }

}
