/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * MetadataServiceImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.metadata;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.parser.AutoDetectParser;
import org.apache.tika.parser.Parser;
import org.apache.tika.sax.BodyContentHandler;
import org.hibernate.Hibernate;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.dao.metadata.SimpleFileDAO;
import org.ow2.dragon.util.ContentType;
import org.ow2.dragon.util.InputStreamUtil;
import org.ow2.dragon.util.UDDIUseType;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class MetadataServiceImpl implements MetadataService {

    private SimpleFileDAO simpleFileDAO;

    private File repositoryRootFile;

    private RepositoryType repoType;

    private Logger logger = Logger.getLogger(this.getClass());

    public MetadataServiceImpl(String repositoryRoot, SimpleFileDAO simpleFileDAO, String repoType)
            throws IOException, MetadataServiceException {
        super();
        this.simpleFileDAO = simpleFileDAO;
        this.repoType = RepositoryType.fromString(repoType);
        if (this.repoType == null) {
            throw new MetadataServiceException("Repository type not supported: \"" + repoType
                    + "\". Supported types are: \"" + RepositoryType.DATABASE.toString()
                    + "\" and \"" + RepositoryType.FILE_SYSTEM.toString() + "\"");
        }
        switch (this.repoType) {
            case DATABASE:
                // Do nothing for the moment
                break;
            case FILE_SYSTEM:
                // Create Repository directory if it doesn't exist
                initRepository(repositoryRoot);
                break;
            default:
                break;
        }
    }

    private void initRepository(String repositoryRoot) throws IOException {
        repositoryRootFile = new File(repositoryRoot);
        FileUtils.forceMkdir(repositoryRootFile);
    }

    public void cleanupRepository() {
        switch (this.repoType) {
            case DATABASE:
                // Do nothing for the moment
                logger.info("A database repository cannot be cleanup");
                break;
            case FILE_SYSTEM:
                List<File> allFileList = Arrays.asList(repositoryRootFile.listFiles());
                for (File file : allFileList) {
                    if (!isMappedInDB(file)) {
                        FileUtils.deleteQuietly(file);
                    }
                }
                break;
            default:
                break;
        }
    }

    private boolean isMappedInDB(File file) {
        boolean bool = false;
        List<SimpleFile> result = simpleFileDAO.searchEquals(new String[] { file.getName() },
                new String[] { "location" }, null);
        if (result != null && !result.isEmpty()) {
            bool = true;
        }
        return bool;
    }

    public void deleteMetadata(final String metadataId) {
        this.simpleFileDAO.remove(metadataId);
    }

    public SimpleFileDAO getSimpleFileDAO() {
        return this.simpleFileDAO;
    }

    public byte[] loadMetadataContent(final String id) {
        throw new RuntimeException("not implemented method");
    }

    public byte[] loadMetadataContentAsBytes(final String id) {
        throw new RuntimeException("not implemented method");
    }

    public InputStream loadMetadataContentAsInputStream(final String id)
            throws MetadataServiceException {
        final SimpleFile simpleFile = this.simpleFileDAO.get(id);

        InputStream inputStream = null;
        switch (repoType) {
            case DATABASE:
                // Retrieve content from db
                final Blob blob = simpleFile.getContent();
                try {
                    inputStream = blob.getBinaryStream();
                } catch (final SQLException e) {
                    throw new MetadataServiceException("Can't read file content", e);
                }
                break;
            case FILE_SYSTEM:
                // Retrieve content from file system
                inputStream = retrieveContentFromFileSystem(simpleFile.getLocation());
                break;
            default:
                break;
        }

        return inputStream;
    }

    private InputStream retrieveContentFromFileSystem(String location)
            throws MetadataServiceException {
        File data = new File(repositoryRootFile, location);
        FileInputStream inputStream = null;
        try {
            inputStream = FileUtils.openInputStream(data);
        } catch (IOException e) {
            throw new MetadataServiceException("Can't load file from file system", e);
        }
        return inputStream;
    }

    public SimpleFile storeMetadataAndIndexContent(final ContentType type, final byte[] content,
            String fileName) throws MetadataServiceException {
        SimpleFile simpleFile = createSimpleFile(type, content, fileName, null);

        // Extract indexable information from original content
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(content);
        try {
            extractInformation(byteArrayInputStream, simpleFile, true);
        } catch (Exception e) {
            throw new MetadataServiceException("Can't extract information from the given content",
                    e);
        }

        // update simple file with extracted infos
        this.simpleFileDAO.save(simpleFile);

        return simpleFile;
    }

    public SimpleFile storeMetadataAndIndexContent(final byte[] content, String fileName)
            throws MetadataServiceException {
        SimpleFile simpleFile = createSimpleFile(null, content, fileName, null);

        // Extract indexable information from original content
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(content);
        try {
            extractInformation(byteArrayInputStream, simpleFile, true);
        } catch (Exception e) {
            throw new MetadataServiceException("Can't extract information from the given content",
                    e);
        }

        // update simple file with extracted infos
        this.simpleFileDAO.save(simpleFile);

        return simpleFile;
    }

    private String storeDataInFileSystem(String id, byte[] content, ContentType type)
            throws MetadataServiceException {
        String suffix = ContentType.getRelatedSuffix(type);

        String location = id + suffix;

        File data = new File(repositoryRootFile, location);
        try {
            FileUtils.writeByteArrayToFile(data, content);
        } catch (IOException e) {
            throw new MetadataServiceException("Can't write file to filesystem", e);
        }
        return data.getName();
    }

    private SimpleFile createSimpleFile(final ContentType type, final byte[] content,
            String fileName, UDDIUseType useType) throws MetadataServiceException {
        // Create the simple file
        SimpleFile simpleFile = new SimpleFile();
        simpleFile.setFileType(type);
        simpleFile.setFileName(fileName);
        simpleFile.setUseType(useType);

        // Save file to generate id
        simpleFile = this.simpleFileDAO.save(simpleFile);

        switch (repoType) {
            case DATABASE:
                // Store data in db
                Blob metadataContent = Hibernate.createBlob(content);
                simpleFile.setContent(metadataContent);
                break;
            case FILE_SYSTEM:
                // Store data in file system
                String location = storeDataInFileSystem(simpleFile.getId(), content, type);
                simpleFile.setLocation(location);
                break;
            default:
                break;
        }

        this.simpleFileDAO.save(simpleFile);

        return simpleFile;
    }

    private void extractInformation(ByteArrayInputStream byteArrayInputStream,
            SimpleFile simpleFile, boolean indexContent) throws IOException, SAXException,
            TikaException {
        // Create an universal parser
        Parser parser = new AutoDetectParser();

        // create the content handler
        StringWriter writer = new StringWriter();
        ContentHandler handler = new BodyContentHandler(writer);

        // Create metadata
        Metadata metadata = new Metadata();

        // Extract inputstream content to string and extract metadata
        parser.parse(byteArrayInputStream, handler, metadata);

        simpleFile.setAuthor(metadata.get(Metadata.AUTHOR));
        simpleFile.setTitle(metadata.get(Metadata.TITLE));
        if (simpleFile.getFileType() == null) {
            // TODO Test if its a supported content type
            simpleFile.setFileType(ContentType.fromString(metadata.get(Metadata.CONTENT_TYPE)));

        }

        if (indexContent) {
            simpleFile.setExtractedContent(writer.toString());
            logger.debug("#### Extracted Content : ");
            logger.debug(writer.toString());
        }

    }

    public SimpleFile storeMetadata(final ContentType type, final URI contentURI, String fileName)
            throws MetadataServiceException {
        return this.storeMetadata(type, contentURI, fileName, null);
    }

    public SimpleFile storeMetadata(final ContentType type, final URI contentURI, String fileName,
            UDDIUseType useType) throws MetadataServiceException {

        /*
         * Read the content of the file denoted by the given URI
         */
        /*
         * BufferedReader reader = null; String buffer = ""; try { String line =
         * ""; //reader = new BufferedReader(new
         * InputStreamReader(contentURI.toURL().openStream())); reader = new
         * BufferedReader(new
         * InputStreamReader(InputStreamUtil.getInputStream(contentURI))); while
         * ((line = reader.readLine()) != null) { buffer = buffer + line; }
         * 
         * } catch (final MalformedURLException e) { throw new
         * MetadataServiceException("Given data URL is invalid:" + contentURI,
         * e); } catch (final IOException e) { throw new
         * MetadataServiceException("Can't read data at the given URL:" +
         * contentURI, e); } catch (URISyntaxException e) { throw new
         * MetadataServiceException("Given data URI is invalid:" + contentURI,
         * e); } finally { if (reader != null) { try { reader.close(); } catch
         * (final IOException e) { throw new
         * MetadataServiceException("Impossible to close the stream", e); } } }
         * 
         * return storeMetadata(type, buffer, fileName, useType);
         */

        try {
            return storeMetadata(type, InputStreamUtil.getBytes(InputStreamUtil
                    .getInputStream(contentURI)), fileName, useType);
        } catch (final IOException e) {
            throw new MetadataServiceException("Can't read data at the given URL:" + contentURI, e);
        } catch (URISyntaxException e) {
            throw new MetadataServiceException("Given data URI is invalid:" + contentURI, e);
        }
    }

    public SimpleFile storeMetadata(final ContentType type, String fileContent, String fileName)
            throws MetadataServiceException {
        return this.storeMetadata(type, fileContent, fileName, null);
    }

    public SimpleFile storeMetadata(final ContentType type, String fileContent, String fileName,
            UDDIUseType useType) throws MetadataServiceException {
        byte[] fileContentAsByte = fileContent.getBytes();
        SimpleFile simpleFile = createSimpleFile(type, fileContentAsByte, fileName, useType);
        // Extract indexable information from original content
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(fileContentAsByte);
        try {
            extractInformation(byteArrayInputStream, simpleFile, false);
        } catch (Exception e) {
            throw new MetadataServiceException("Can't extract information from the given content",
                    e);
        }
        // update simple file with extracted infos
        this.simpleFileDAO.save(simpleFile);
        return simpleFile;
    }

    public SimpleFile loadMetadata(String id) {
        return this.simpleFileDAO.get(id);
    }

    public SimpleFile storeMetadataAndIndexContent(String mimetype, byte[] docContent,
            String fileName) throws MetadataServiceException {
        SimpleFile result = null;
        ContentType contentType = ContentType.fromString(mimetype);
        if (contentType != null) {
            result = storeMetadataAndIndexContent(contentType, docContent, fileName);
        } else {
            throw new MetadataServiceException(
                    "Can't register document. Content type not supported : " + mimetype);
        }
        return result;
    }

    public SimpleFile storeMetadata(ContentType type, byte[] content)
            throws MetadataServiceException {
        SimpleFile simpleFile = createSimpleFile(type, content, null, null);
        // Extract indexable information from original content
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(content);
        try {
            extractInformation(byteArrayInputStream, simpleFile, false);
        } catch (Exception e) {
            throw new MetadataServiceException("Can't extract information from the given content",
                    e);
        }
        // update simple file with extracted infos
        this.simpleFileDAO.save(simpleFile);
        return simpleFile;
    }

    public SimpleFile storeMetadata(ContentType type, String fileContent)
            throws MetadataServiceException {

        return storeMetadata(type, fileContent, null);
    }

    public SimpleFile storeMetadata(ContentType type, URI contentURI)
            throws MetadataServiceException {

        return storeMetadata(type, contentURI, null);
    }

    public SimpleFile storeMetadataAndIndexContent(byte[] content) throws MetadataServiceException {

        return storeMetadataAndIndexContent(content, null);
    }

    public SimpleFile storeMetadataAndIndexContent(ContentType type, byte[] content)
            throws MetadataServiceException {

        return storeMetadataAndIndexContent(type, content, null);
    }

    public SimpleFile storeMetadataAndIndexContent(String mimetype, byte[] docContent)
            throws MetadataServiceException {

        return storeMetadataAndIndexContent(mimetype, docContent, null);
    }

    public SimpleFile storeMetadata(ContentType type, byte[] content, String fileName)
            throws MetadataServiceException {
        return this.storeMetadata(type, content, fileName, null);
    }

    public SimpleFile storeMetadata(ContentType type, byte[] content, String fileName,
            UDDIUseType useType) throws MetadataServiceException {
        SimpleFile simpleFile = createSimpleFile(type, content, fileName, useType);
        // Extract indexable information from original content
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(content);
        try {
            extractInformation(byteArrayInputStream, simpleFile, false);
        } catch (Exception e) {
            throw new MetadataServiceException("Can't extract information from the given content",
                    e);
        }
        // update simple file with extracted infos
        this.simpleFileDAO.save(simpleFile);
        return simpleFile;
    }

    public String getRepositoryRootPath() {
        String rootPath = null;
        if (repositoryRootFile != null) {
            rootPath = repositoryRootFile.getAbsolutePath();
        }
        return rootPath;
    }
}
