/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleOfPartyManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.organization;

import java.util.ArrayList;
import java.util.List;

import org.ow2.dragon.api.service.common.CommonException;
import org.ow2.dragon.api.service.common.EntityLinkManager;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.service.organization.RoleOfPartyManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.OrgToEndpointTO;
import org.ow2.dragon.api.to.organization.OrgToTechServiceTO;
import org.ow2.dragon.api.to.organization.PersonToEndpointTO;
import org.ow2.dragon.api.to.organization.PersonToTechServiceTO;
import org.ow2.dragon.persistence.bo.common.Link;
import org.ow2.dragon.persistence.bo.common.LinkedEntity;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.dao.UniversalORMDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class RoleOfPartyManagerImpl implements RoleOfPartyManager {

    private EntityLinkManager entityLinkManager;

    private UniversalORMDAO universalORMDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    public String createEndpointRole(final String partyId, final String endpointId,
            final String type) throws OrganizationException {

        String roleId;
        try {
            // Create and persist Role
            roleId = this.entityLinkManager.createLink(partyId, endpointId, type);

            // Automatically add role to tech service because endpoint is part
            // of a
            // service
            Endpoint endpoint = (Endpoint) this.universalORMDAO.get(Endpoint.class, endpointId);
            this.entityLinkManager
                    .createLink(partyId, endpoint.getTechnicalService().getId(), type);
        } catch (CommonException e) {
            throw new OrganizationException(e);
        }
        return roleId;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.RoleOfPartyManager#createRole
     * (java.lang.String, java.lang.String, java.lang.String)
     */
    public String createTechServiceRole(final String partyId, final String technicalServiceId,
            final String type) throws OrganizationException {

        String roleId;
        try {
            // Create and persist Role
            roleId = this.entityLinkManager.createLink(partyId, technicalServiceId, type);
        } catch (CommonException e) {
            throw new OrganizationException(e);
        }

        return roleId;
    }

    public List<OrgToEndpointTO> getOrgRolesByEndpoint(final String endpointId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        // TODO improve by making filtering in DB
        final List<OrgToEndpointTO> result = new ArrayList<OrgToEndpointTO>();

        // retrieve all roles
        final List<Link> allRoles = this.entityLinkManager.getAllLinksForTo(endpointId,
                requestOptionsTO);

        // Search only Org roles
        for (final Link partyToEndpoint : allRoles) {
            final LinkedEntity party = partyToEndpoint.getFrom();
            if (party instanceof OrganizationUnit) {
                final OrgToEndpointTO orgToEndpointTO = new OrgToEndpointTO();
                orgToEndpointTO.setId(partyToEndpoint.getId());
                orgToEndpointTO.setEndpointTO(this.transfertObjectAssembler
                        .toEndpointTO((Endpoint) partyToEndpoint.getTo(), null));
                orgToEndpointTO.setPartyTO(this.transfertObjectAssembler
                        .toOrganizationUnitTO((OrganizationUnit) party));
                orgToEndpointTO.setType(partyToEndpoint.getType());
                result.add(orgToEndpointTO);
            }
        }
        return result;
    }

    public List<OrgToTechServiceTO> getOrgRolesByTechService(final String techServiceId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        // TODO improve by making filtering in DB
        final List<OrgToTechServiceTO> result = new ArrayList<OrgToTechServiceTO>();

        // retrieve all roles
        final List<Link> allRoles = this.entityLinkManager.getAllLinksForTo(techServiceId,
                requestOptionsTO);

        // Search only Org roles
        for (final Link partyToTechService : allRoles) {
            final LinkedEntity party = partyToTechService.getFrom();
            if (party instanceof OrganizationUnit) {
                final OrgToTechServiceTO orgToTechServiceTO = new OrgToTechServiceTO();
                orgToTechServiceTO.setId(partyToTechService.getId());
                orgToTechServiceTO.setTechnicalServiceTO(this.transfertObjectAssembler
                        .toTechServiceTO((TechnicalService) partyToTechService.getTo(),null));
                orgToTechServiceTO.setPartyTO(this.transfertObjectAssembler
                        .toOrganizationUnitTO((OrganizationUnit) party));
                orgToTechServiceTO.setType(partyToTechService.getType());
                result.add(orgToTechServiceTO);
            }
        }
        return result;
    }

    public List<PersonToEndpointTO> getPersonRolesByEndpoint(final String endpointId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        // TODO improve by making filtering in DB
        final List<PersonToEndpointTO> result = new ArrayList<PersonToEndpointTO>();

        // retrieve all roles
        final List<Link> allRoles = this.entityLinkManager.getAllLinksForTo(endpointId,
                requestOptionsTO);

        // Search only Person roles
        for (final Link partyToEndpoint : allRoles) {
            final LinkedEntity party = partyToEndpoint.getFrom();
            if (party instanceof Person) {
                final PersonToEndpointTO personToEndpointTO = new PersonToEndpointTO();
                personToEndpointTO.setId(partyToEndpoint.getId());
                personToEndpointTO.setEndpointTO(this.transfertObjectAssembler
                        .toEndpointTO((Endpoint) partyToEndpoint.getTo(), null));
                personToEndpointTO.setPartyTO(this.transfertObjectAssembler
                        .toPersonTO((Person) party));
                personToEndpointTO.setType(partyToEndpoint.getType());
                result.add(personToEndpointTO);
            }
        }
        return result;
    }

    public List<PersonToTechServiceTO> getPersonRolesByTechService(final String techServiceId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        // TODO improve by making filtering in DB
        final List<PersonToTechServiceTO> result = new ArrayList<PersonToTechServiceTO>();

        // retrieve all roles
        final List<Link> allRoles = this.entityLinkManager.getAllLinksForTo(techServiceId,
                requestOptionsTO);

        // Search only Person roles
        for (final Link partyToTechService : allRoles) {
            final LinkedEntity party = partyToTechService.getFrom();
            if (party instanceof Person) {
                final PersonToTechServiceTO personToTechServiceTO = new PersonToTechServiceTO();
                personToTechServiceTO.setId(partyToTechService.getId());
                personToTechServiceTO.setTechnicalServiceTO(this.transfertObjectAssembler
                        .toTechServiceTO((TechnicalService) partyToTechService.getTo(),null));
                personToTechServiceTO.setPartyTO(this.transfertObjectAssembler
                        .toPersonTO((Person) party));
                personToTechServiceTO.setType(partyToTechService.getType());
                result.add(personToTechServiceTO);
            }
        }
        return result;
    }

    public String[] getRolesTypes(final LINK_TYPE linkType, final RequestOptionsTO options)
            throws OrganizationException {
        // TODO Create roles enum or retrieve the differents lists in DB
        String[] result = null;
        final String[] personToTechServRoles = new String[] { "Architect", "Certifier", "Designer",
                "Developer", "Manager", "User" };
        final String[] orgToTechServRoles = new String[] { "Analyst", "Consumer", "Owner",
                "Provider" };
        final String[] personToEndpointRoles = new String[] { "Architect", "Certifier", "Designer",
                "Developer", "Manager", "User" };
        final String[] orgToEndpointRoles = new String[] { "Analyst", "Consumer", "Owner",
                "Provider" };

        if (linkType.equals(LINK_TYPE.PERSON_TO_TECHSERVICE)) {
            result = personToTechServRoles;
        } else if (linkType.equals(LINK_TYPE.ORG_TO_TECHSERVICE)) {
            result = orgToTechServRoles;
        } else if (linkType.equals(LINK_TYPE.PERSON_TO_ENDPOINT)) {
            result = personToEndpointRoles;
        } else if (linkType.equals(LINK_TYPE.ORG_TO_ENDPOINT)) {
            result = orgToEndpointRoles;
        }
        return result;
    }

    public void removeRole(final String roleId) throws OrganizationException {
        this.entityLinkManager.removeLink(roleId);
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    public void setUniversalORMDAO(UniversalORMDAO universalORMDAO) {
        this.universalORMDAO = universalORMDAO;
    }

    public void setEntityLinkManager(EntityLinkManager entityLinkManager) {
        this.entityLinkManager = entityLinkManager;
    }

}
