/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechnologyManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.technology;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.cxf.frontend.ClientFactoryBean;
import org.apache.cxf.jaxws.JaxWsProxyFactoryBean;
import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.technology.TechnologyException;
import org.ow2.dragon.api.service.technology.TechnologyManager;
import org.ow2.dragon.api.service.wsdl.WSDLManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.technology.EnvironmentFederationTO;
import org.ow2.dragon.api.to.technology.ExecEnvManagerSearchProperties;
import org.ow2.dragon.api.to.technology.ExecEnvSearchProperties;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentManagerTO;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentTO;
import org.ow2.dragon.api.to.technology.FederationSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorTO;
import org.ow2.dragon.connection.api.to.EnvironmentFederation.FedPattern;
import org.ow2.dragon.connection.api.to.ExecutionEnvironment.EEType;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.technology.EnvironmentFederation;
import org.ow2.dragon.persistence.bo.technology.ExecutionEnvironment;
import org.ow2.dragon.persistence.bo.technology.ExecutionEnvironmentManager;
import org.ow2.dragon.persistence.bo.technology.Processor;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.RequestOptions;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.SearchHelper;
import org.springframework.util.CollectionUtils;

/**
 * @author ofabre - eBM Websourcing
 * 
 */
public class TechnologyManagerImpl implements TechnologyManager {

    private Logger logger = Logger.getLogger(this.getClass());

    private GenericUnifiedDAO<ExecutionEnvironmentManager, String> executionEnvironmentManagerUnifiedDAO;

    private GenericUnifiedDAO<org.ow2.dragon.persistence.bo.technology.ExecutionEnvironment, String> executionEnvironmentUnifiedDAO;

    private GenericUnifiedDAO<org.ow2.dragon.persistence.bo.technology.EnvironmentFederation, String> environmentFederationUnifiedDAO;

    private GenericUnifiedDAO<Endpoint, String> endpointUnifiedDAO;

    private GenericUnifiedDAO<Processor, String> processorUnifiedDAO;

    private WSDLManager wsdlManager;

    private JaxWsProxyFactoryBean executionEnvironmentManagerFactory;

    private TransfertObjectAssembler transfertObjectAssembler;

    private TechnologyTransferObjectAssembler technologyTransferObjectAssembler;

    private Map<String, org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager> execEnvManagers = new HashMap<String, org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager>();

    public String addRuntimeManager(String address) throws TechnologyException {
        String result = null;

        // check if ExecEnvManager isn't already persisted
        List<ExecutionEnvironmentManager> execEnvManagers = executionEnvironmentManagerUnifiedDAO
                .searchEquals(new String[] { address }, new String[] { "address" }, null);

        if (CollectionUtils.isEmpty(execEnvManagers)) {
            // Create ExecEnvManager Proxy
            org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager execEnvManagerProxy = createExecEnvManagerProxy(address);

            // get ExecEnvManager properties from runtime env
            org.ow2.dragon.connection.api.to.ExecutionEnvironmentManager execEnvManagerTO = execEnvManagerProxy
                    .getProperties();

            // Create new ExecEnvManager BO
            if (execEnvManagerTO == null) {
                throw new TechnologyException(
                        "Can't retrieve execution environment manager properties");
            }
            ExecutionEnvironmentManager execEnvManagerBO = new ExecutionEnvironmentManager();
            execEnvManagerBO.setAddress(address);
            execEnvManagerBO.setName(execEnvManagerTO.getName());

            // persist new ExecEnvManager BO
            result = executionEnvironmentManagerUnifiedDAO.save(execEnvManagerBO).getId();
        } else if (execEnvManagers.size() > 1) {
            throw new TechnologyException(
                    "Database incoherence : two or more Execution Environment Manager with the same address");
        } else {
            result = execEnvManagers.get(0).getId();
        }
        return result;
    }

    private org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager createExecEnvManagerProxy(
            String address) throws TechnologyException {
        org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager execEnvManagerProxy = execEnvManagers
                .get(address);
        if (execEnvManagerProxy == null) {
            // Reinit factory
            ClientFactoryBean clientFactoryBean = executionEnvironmentManagerFactory
                    .getClientFactoryBean();

            if (clientFactoryBean != null) {
                // this fix isn't needed with cxf 2.2.5+
                clientFactoryBean.setClient(null);
            }

            // Set service address
            executionEnvironmentManagerFactory.setAddress(address);
            try {
                // Create service proxy
                execEnvManagerProxy = (org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager) executionEnvironmentManagerFactory
                        .create();
            } catch (RuntimeException e) {
                throw new TechnologyException("Can't connect to execution environment manager");
            }
            execEnvManagers.put(address, execEnvManagerProxy);
        }
        return execEnvManagerProxy;
    }

    public List<ExecutionEnvironmentTO> getManagedExecEnv(String runtimeManagerId,
            RequestOptionsTO requestOptionsTO) {
        List<ExecutionEnvironmentTO> result = new ArrayList<ExecutionEnvironmentTO>();

        List<ExecutionEnvironment> execEnvsBO = executionEnvironmentUnifiedDAO.searchEquals(
                new String[] { runtimeManagerId }, new String[] { "manager.id" },
                transfertObjectAssembler.toExecEnvRequestOptions(requestOptionsTO));
        if (execEnvsBO != null) {
            for (ExecutionEnvironment execEnvBO : execEnvsBO) {
                result.add(technologyTransferObjectAssembler.toExecutionEnvironmentTO(execEnvBO));
            }
        }
        return result;
    }

    public void synchronizeManagedExecEnv(String runtimeManagerId) throws TechnologyException {
        // Get ExecEnvManager BO
        ExecutionEnvironmentManager executionEnvironmentManager = executionEnvironmentManagerUnifiedDAO
                .get(runtimeManagerId);
        if (executionEnvironmentManager == null) {
            throw new TechnologyException("Can't retrieve Execution Environment Manager with id: "
                    + runtimeManagerId);
        }

        // Create ExecEnvManager Proxy
        org.ow2.dragon.connection.api.service.ExecutionEnvironmentManager execEnvManagerProxy = createExecEnvManagerProxy(executionEnvironmentManager
                .getAddress());

        // Get managed exec env from runtime
        List<org.ow2.dragon.connection.api.to.ExecutionEnvironment> execEnvProxies = execEnvManagerProxy
                .getManagedExecutionEnvironments();

        // Update data base
        if (!CollectionUtils.isEmpty(execEnvProxies)) {
            for (org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy : execEnvProxies) {
                // try to retrieve corresponding execEnv from DB (business key :
                // name)
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<ExecutionEnvironment> execEnvsBO = executionEnvironmentUnifiedDAO
                        .searchEquals(new String[] { execEnvProxy.getName() },
                                new String[] { "name" }, requestOptions);
                if (!CollectionUtils.isEmpty(execEnvsBO)) {
                    if (execEnvsBO.size() > 1) {
                        throw new TechnologyException(
                                "Database incoherence : two or more Execution Environment with the same name");
                    }
                    ExecutionEnvironment execEnvBO = execEnvsBO.get(0);
                    updateExecEnv(execEnvProxy, execEnvBO);
                    executionEnvironmentManager.addNode(execEnvBO);
                } else {
                    ExecutionEnvironment execEnvBO = createNewExecEnv(executionEnvironmentManager,
                            execEnvProxy);
                    executionEnvironmentManager.addNode(execEnvBO);
                }
            }
        }

        // Update exec env manager
        executionEnvironmentManagerUnifiedDAO.save(executionEnvironmentManager);
    }

    private ExecutionEnvironment createNewExecEnv(
            ExecutionEnvironmentManager executionEnvironmentManager,
            org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy)
            throws TechnologyException {
        ExecutionEnvironment execEnvBO = new ExecutionEnvironment();
        EEType type = execEnvProxy.getEnvType();
        if (type != null) {
            execEnvBO.setEnvType(ExecutionEnvironment.EEType.valueOf(type.toString()));
        }
        execEnvBO.setIpv4Address(execEnvProxy.getIpv4Address());
        execEnvBO.setName(execEnvProxy.getName());
        execEnvBO.setRoleInFederation(execEnvProxy.getRoleInFederation());
        execEnvBO.setType(execEnvProxy.getType());
        executionEnvironmentUnifiedDAO.save(execEnvBO);

        // Add links to other instances
        setEnpoints(execEnvProxy, execEnvBO);
        setHostProcessor(execEnvProxy, execEnvBO);
        setParentFederation(execEnvProxy, execEnvBO);
        executionEnvironmentUnifiedDAO.save(execEnvBO);

        return execEnvBO;
    }

    private void setParentFederation(
            org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy,
            ExecutionEnvironment execEnvBO) throws TechnologyException {
        org.ow2.dragon.connection.api.to.EnvironmentFederation federationProxy = execEnvProxy
                .getParentFederation();
        if (federationProxy != null) {
            EnvironmentFederation envFedBO;
            // try to retrieve corresponding federation from DB (business key :
            // name)
            RequestOptions requestOptions = new RequestOptions();
            requestOptions.setCaseSensitive(true);
            List<EnvironmentFederation> envFedsBO = environmentFederationUnifiedDAO.searchEquals(
                    new String[] { execEnvProxy.getParentFederation().getName().toString() },
                    new String[] { "name" }, requestOptions);
            if (!CollectionUtils.isEmpty(envFedsBO)) {
                if (envFedsBO.size() > 1) {
                    throw new TechnologyException(
                            "Database incoherence : two or more Environment Federations with the same name");
                }
                envFedBO = envFedsBO.get(0);
                updateEnvFed(federationProxy, envFedBO);
            } else {
                envFedBO = createNewEnvFed(federationProxy, execEnvBO);
            }
            envFedBO.addExecEnv(execEnvBO);
        }

    }

    private EnvironmentFederation createNewEnvFed(
            org.ow2.dragon.connection.api.to.EnvironmentFederation federationProxy,
            ExecutionEnvironment execEnvBO) {
        EnvironmentFederation environmentFederation = new EnvironmentFederation();
        environmentFederation.setName(federationProxy.getName());
        FedPattern pattern = federationProxy.getPattern();
        if (pattern != null) {
            environmentFederation
                    .setPattern(org.ow2.dragon.persistence.bo.technology.EnvironmentFederation.FedPattern
                            .valueOf(pattern.toString()));
        }
        List<String> possibleRoles = federationProxy.getPossibleRole();
        if (!CollectionUtils.isEmpty(possibleRoles)) {
            for (String possibleRole : possibleRoles) {
                environmentFederation.addPossibleRole(possibleRole);
            }
        }
        environmentFederationUnifiedDAO.save(environmentFederation);
        return environmentFederation;
    }

    private void updateEnvFed(
            org.ow2.dragon.connection.api.to.EnvironmentFederation federationProxy,
            EnvironmentFederation envFedBO) {
        // TODO Do nothing for the moment

    }

    private void setHostProcessor(
            org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy,
            ExecutionEnvironment execEnvBO) throws TechnologyException {
        org.ow2.dragon.connection.api.to.Processor processorProxy = execEnvProxy.getHostProcessor();
        if (processorProxy != null) {
            Processor processorBO;
            // try to retrieve corresponding ep from DB (business key :
            // name)
            RequestOptions requestOptions = new RequestOptions();
            requestOptions.setCaseSensitive(true);
            List<Processor> processorsBO = processorUnifiedDAO.searchEquals(
                    new String[] { execEnvProxy.getHostProcessor().getName().toString() },
                    new String[] { "name" }, requestOptions);
            if (!CollectionUtils.isEmpty(processorsBO)) {
                if (processorsBO.size() > 1) {
                    throw new TechnologyException(
                            "Database incoherence : two or more Processors with the same name");
                }
                processorBO = processorsBO.get(0);
                updateProcessor(processorProxy, processorBO);
            } else {
                processorBO = createNewProcessor(processorProxy, execEnvBO);
            }
            processorBO.addExecutionEnvironment(execEnvBO);
        }

    }

    private Processor createNewProcessor(org.ow2.dragon.connection.api.to.Processor processorProxy,
            ExecutionEnvironment execEnvBO) {
        Processor processor = new Processor();
        processor.setName(processorProxy.getName());
        processor.setIpv4Address(processorProxy.getIpv4Address());
        processor.setType(processorProxy.getType());
        processorUnifiedDAO.save(processor);
        return processor;
    }

    private void updateProcessor(org.ow2.dragon.connection.api.to.Processor processorProxy,
            Processor processorBO) {
        // TODO Do nothing for the moment

    }

    private void setEnpoints(org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy,
            ExecutionEnvironment execEnvBO) throws TechnologyException {
        List<org.ow2.dragon.connection.api.to.Endpoint> epsProxy = execEnvProxy.getEndpoints();
        if (!CollectionUtils.isEmpty(epsProxy)) {
            for (org.ow2.dragon.connection.api.to.Endpoint epProxy : epsProxy) {
                Endpoint epBO;
                // try to retrieve corresponding ep from DB (business key :
                // name)
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<Endpoint> epsBO = endpointUnifiedDAO.searchEquals(new String[] { epProxy
                        .getName().toString() }, new String[] { "name" }, requestOptions);
                if (!CollectionUtils.isEmpty(epsBO)) {
                    if (epsBO.size() > 1) {
                        throw new TechnologyException(
                                "Database incoherence : two or more Endpoints with the same name");
                    }
                    epBO = epsBO.get(0);
                    updateEndpoint(epProxy, epBO);
                } else {
                    epBO = createNewEndpoint(epProxy);
                }
                if (epBO != null) {
                    execEnvBO.addEndpoint(epBO);
                }
            }
        }
    }

    private Endpoint createNewEndpoint(org.ow2.dragon.connection.api.to.Endpoint epProxy)
            throws TechnologyException {
        Endpoint result = null;
        // Parse Endpoint description to create the new EP
        QName epName = epProxy.getName();
        if (epName == null) {
            throw new TechnologyException("Can't register an Endpoint without name");
        }
        try {
            wsdlManager.importServiceDefFile(epProxy.getWsdlDescription(), epProxy
                    .getWsdlDescriptionImports(), null);
        } catch (Exception e) {
            logger.warn("Failed to import endpoint description for ep:'" + epName.toString()
                    + "'. Endpoint will not be managed !", e);
        }
        RequestOptions requestOptions = new RequestOptions();
        requestOptions.setCaseSensitive(true);
        List<Endpoint> epsBO = endpointUnifiedDAO.searchEquals(new String[] { epName.toString() },
                new String[] { "name" }, requestOptions);
        if (!CollectionUtils.isEmpty(epsBO)) {
            result = epsBO.get(0);
        } else {
            logger.warn("You are trying to register an Endpoint without description: "
                    + epName.toString());
        }
        return result;
    }

    private void updateEndpoint(org.ow2.dragon.connection.api.to.Endpoint epProxy, Endpoint epBO) {
        // TODO Do nothing for the moment

    }

    private void updateExecEnv(org.ow2.dragon.connection.api.to.ExecutionEnvironment execEnvProxy,
            ExecutionEnvironment executionEnvironment) {
        // TODO Do nothing for the moment

    }

    public void setExecutionEnvironmentManagerUnifiedDAO(
            GenericUnifiedDAO<ExecutionEnvironmentManager, String> executionEnvironmentManagerUnifiedDAO) {
        this.executionEnvironmentManagerUnifiedDAO = executionEnvironmentManagerUnifiedDAO;
    }

    public void setExecutionEnvironmentUnifiedDAO(
            GenericUnifiedDAO<org.ow2.dragon.persistence.bo.technology.ExecutionEnvironment, String> executionEnvironmentUnifiedDAO) {
        this.executionEnvironmentUnifiedDAO = executionEnvironmentUnifiedDAO;
    }

    public void setExecutionEnvironmentManagerFactory(
            JaxWsProxyFactoryBean executionEnvironmentManagerFactory) {
        this.executionEnvironmentManagerFactory = executionEnvironmentManagerFactory;
    }

    public void setTransfertObjectAssembler(TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    public void setTechnologyTransferObjectAssembler(
            TechnologyTransferObjectAssembler technologyTransferObjectAssembler) {
        this.technologyTransferObjectAssembler = technologyTransferObjectAssembler;
    }

    public ExecutionEnvironmentManagerTO getRuntimeManager(String arg0) {
        ExecutionEnvironmentManager environmentManager = executionEnvironmentManagerUnifiedDAO
                .get(arg0);
        return technologyTransferObjectAssembler
                .toExecutionEnvironmentManagerTO(environmentManager);
    }

    public void removeRuntimeManager(String runtimeManagerId) {
        executionEnvironmentManagerUnifiedDAO.remove(runtimeManagerId);
    }

    public ExecutionEnvironmentTO getExecutionEnvironment(String execEnvId) {
        ExecutionEnvironment environment = executionEnvironmentUnifiedDAO.get(execEnvId);
        return technologyTransferObjectAssembler.toExecutionEnvironmentTO(environment);
    }

    public void removeExecutionEnvironment(String execEnvId) {
        executionEnvironmentUnifiedDAO.remove(execEnvId);
    }

    public void setEndpointUnifiedDAO(GenericUnifiedDAO<Endpoint, String> endpointUnifiedDAO) {
        this.endpointUnifiedDAO = endpointUnifiedDAO;
    }

    public void setWsdlManager(WSDLManager wsdlManager) {
        this.wsdlManager = wsdlManager;
    }

    public List<ExecutionEnvironmentManagerTO> getAllExecutionEnvironmentManagers(
            RequestOptionsTO requestOptionsTO) {
        List<ExecutionEnvironmentManagerTO> result = new ArrayList<ExecutionEnvironmentManagerTO>();
        final List<ExecutionEnvironmentManager> runtimeManagers = this.executionEnvironmentManagerUnifiedDAO
                .getAll(this.transfertObjectAssembler.toExecEnvManRequestOptions(requestOptionsTO));
        toRuntimeManagersTO(result, runtimeManagers);
        return result;
    }

    private void toRuntimeManagersTO(List<ExecutionEnvironmentManagerTO> result,
            List<ExecutionEnvironmentManager> runtimeManagers) {
        if ((runtimeManagers != null) && !runtimeManagers.isEmpty()) {
            for (final ExecutionEnvironmentManager runtimeManager : runtimeManagers) {
                result.add(toRuntimeManagerTO(runtimeManager));
            }
        }
    }

    private ExecutionEnvironmentManagerTO toRuntimeManagerTO(
            final ExecutionEnvironmentManager runtimeManager) {
        return this.technologyTransferObjectAssembler
                .toExecutionEnvironmentManagerTO(runtimeManager);
    }

    public List<ExecutionEnvironmentManagerTO> searchExecutionEnvironmentManagers(
            String searchCriteria, List<ExecEnvManagerSearchProperties> searchedProperties,
            RequestOptionsTO requestOptionsTO) throws TechnologyException {
        List<ExecutionEnvironmentManagerTO> result = new ArrayList<ExecutionEnvironmentManagerTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createRuntimeManagerSearchProperties(searchedProperties);

        // Search for bo
        List<ExecutionEnvironmentManager> runtimeManagers;
        try {
            runtimeManagers = this.executionEnvironmentManagerUnifiedDAO.searchORMResult(criteria,
                    properties, transfertObjectAssembler
                            .toExecEnvManRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new TechnologyException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toRuntimeManagersTO(result, runtimeManagers);
        return result;
    }

    private String[] createRuntimeManagerSearchProperties(
            List<ExecEnvManagerSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(ExecEnvManagerSearchProperties.NAME)) {
                propertiesList.add("name");
            }
        } else {
            // search on all properties
            propertiesList.add("name");
        }
        return propertiesList.toArray(new String[0]);
    }

    public List<ExecutionEnvironmentTO> getAllExecutionEnvironments(
            RequestOptionsTO requestOptionsTO) {
        List<ExecutionEnvironmentTO> result = new ArrayList<ExecutionEnvironmentTO>();
        final List<ExecutionEnvironment> execEnvs = this.executionEnvironmentUnifiedDAO
                .getAll(this.transfertObjectAssembler.toExecEnvRequestOptions(requestOptionsTO));
        toExecEnvsTO(result, execEnvs);
        return result;
    }

    private void toExecEnvsTO(List<ExecutionEnvironmentTO> result,
            List<ExecutionEnvironment> execEnvs) {
        if ((execEnvs != null) && !execEnvs.isEmpty()) {
            for (final ExecutionEnvironment execEnv : execEnvs) {
                result.add(toExecEnvTO(execEnv));
            }
        }
    }

    private ExecutionEnvironmentTO toExecEnvTO(ExecutionEnvironment execEnv) {
        return this.technologyTransferObjectAssembler.toExecutionEnvironmentTO(execEnv);
    }

    public List<ExecutionEnvironmentTO> searchExecutionEnvironments(String searchCriteria,
            List<ExecEnvSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException {
        List<ExecutionEnvironmentTO> result = new ArrayList<ExecutionEnvironmentTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createExecEnvSearchProperties(searchedProperties);

        // Search for bo
        List<ExecutionEnvironment> execEnvs;
        try {
            execEnvs = this.executionEnvironmentUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toExecEnvRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new TechnologyException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toExecEnvsTO(result, execEnvs);
        return result;
    }

    private String[] createExecEnvSearchProperties(List<ExecEnvSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(ExecEnvSearchProperties.NAME)) {
                propertiesList.add("name");
            }
            if (searchedProperties.contains(ExecEnvSearchProperties.ENV_TYPE)) {
                propertiesList.add("envType");
            }
            if (searchedProperties.contains(ExecEnvSearchProperties.ADDRESS)) {
                propertiesList.add("ipv4Address");
            }
            if (searchedProperties.contains(ExecEnvSearchProperties.ROLE_IN_FEDERATION)) {
                propertiesList.add("roleInFederation");
            }
            if (searchedProperties.contains(ExecEnvSearchProperties.FEDERATION)) {
                propertiesList.add("parentFederation.name");
            }
            if (searchedProperties.contains(ExecEnvSearchProperties.HOST)) {
                propertiesList.add("hostProcessor.name");
            }
        } else {
            // search on all properties
            propertiesList.add("name");
            propertiesList.add("envType");
            propertiesList.add("ipv4Address");
            propertiesList.add("roleInFederation");
            propertiesList.add("parentFederation.name");
            propertiesList.add("hostProcessor.name");
        }
        return propertiesList.toArray(new String[0]);
    }

    public List<EndpointTO> getEpsHostedOnExecEnv(String execEnvId,
            RequestOptionsTO requestOptionsTO) {
        List<EndpointTO> result = new ArrayList<EndpointTO>();

        List<Endpoint> epsBO = endpointUnifiedDAO.searchEquals(new String[] { execEnvId },
                new String[] { "hostNode.id" }, transfertObjectAssembler
                        .toEndpointRequestOptions(requestOptionsTO));
        if (epsBO != null) {
            for (Endpoint epBO : epsBO) {
                result.add(transfertObjectAssembler.toEndpointTO(epBO, null));
            }
        }
        return result;
    }

    public List<EndpointTO> getEpsHostedOnProcessor(String processorId,
            RequestOptionsTO requestOptionsTO) {
        List<EndpointTO> result = new ArrayList<EndpointTO>();

        List<Endpoint> epsBO = endpointUnifiedDAO.searchEquals(new String[] { processorId },
                new String[] { "hostNode.hostProcessor.id" }, transfertObjectAssembler
                        .toEndpointRequestOptions(requestOptionsTO));
        if (epsBO != null) {
            for (Endpoint epBO : epsBO) {
                result.add(transfertObjectAssembler.toEndpointTO(epBO, null));
            }
        }
        return result;
    }

    public List<ExecutionEnvironmentTO> getExecEnvsHostedOnProcessor(String processorId,
            RequestOptionsTO requestOptionsTO) {
        List<ExecutionEnvironmentTO> result = new ArrayList<ExecutionEnvironmentTO>();

        List<ExecutionEnvironment> execEnvsBO = executionEnvironmentUnifiedDAO.searchEquals(
                new String[] { processorId }, new String[] { "hostProcessor.id" },
                transfertObjectAssembler.toExecEnvRequestOptions(requestOptionsTO));
        if (execEnvsBO != null) {
            for (ExecutionEnvironment execEnvBO : execEnvsBO) {
                result.add(technologyTransferObjectAssembler.toExecutionEnvironmentTO(execEnvBO));
            }
        }
        return result;
    }

    public ProcessorTO getProcessor(String processorId) {
        Processor processor = processorUnifiedDAO.get(processorId);
        return technologyTransferObjectAssembler.toProcessorTO(processor);
    }

    public void removeProcessor(String processorId) {
        processorUnifiedDAO.remove(processorId);
    }

    public void setProcessorUnifiedDAO(GenericUnifiedDAO<Processor, String> processorUnifiedDAO) {
        this.processorUnifiedDAO = processorUnifiedDAO;
    }

    public List<ProcessorTO> getAllProcessors(RequestOptionsTO requestOptionsTO) {
        List<ProcessorTO> result = new ArrayList<ProcessorTO>();
        final List<Processor> processors = this.processorUnifiedDAO
                .getAll(this.transfertObjectAssembler.toProcessorRequestOptions(requestOptionsTO));
        toProcessorsTO(result, processors);
        return result;
    }

    private void toProcessorsTO(List<ProcessorTO> result, List<Processor> processors) {
        if ((processors != null) && !processors.isEmpty()) {
            for (final Processor processor : processors) {
                result.add(toProcessorTO(processor));
            }
        }
    }

    private ProcessorTO toProcessorTO(Processor processor) {
        return this.technologyTransferObjectAssembler.toProcessorTO(processor);
    }

    public List<ProcessorTO> searchProcessors(String searchCriteria,
            List<ProcessorSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException {
        List<ProcessorTO> result = new ArrayList<ProcessorTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createProcessorSearchProperties(searchedProperties);

        // Search for bo
        List<Processor> processors;
        try {
            processors = this.processorUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toProcessorRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new TechnologyException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toProcessorsTO(result, processors);
        return result;
    }

    private String[] createProcessorSearchProperties(
            List<ProcessorSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(ProcessorSearchProperties.NAME)) {
                propertiesList.add("name");
            }
            if (searchedProperties.contains(ProcessorSearchProperties.ADDRESS)) {
                propertiesList.add("ipv4Address");
            }
        } else {
            // search on all properties
            propertiesList.add("name");
            propertiesList.add("ipv4Address");
        }
        return propertiesList.toArray(new String[0]);
    }

    public EnvironmentFederationTO getEnvironmentFederation(String envFedId) {
        EnvironmentFederation environmentFederation = environmentFederationUnifiedDAO.get(envFedId);
        return technologyTransferObjectAssembler.toEnvironmentFederationTO(environmentFederation);
    }

    public List<ExecutionEnvironmentTO> getExecEnvsInFederation(String envFedId,
            RequestOptionsTO requestOptionsTO) {
        List<ExecutionEnvironmentTO> result = new ArrayList<ExecutionEnvironmentTO>();

        List<ExecutionEnvironment> execEnvsBO = executionEnvironmentUnifiedDAO.searchEquals(
                new String[] { envFedId }, new String[] { "parentFederation.id" },
                transfertObjectAssembler.toExecEnvRequestOptions(requestOptionsTO));
        if (execEnvsBO != null) {
            for (ExecutionEnvironment execEnvBO : execEnvsBO) {
                result.add(technologyTransferObjectAssembler.toExecutionEnvironmentTO(execEnvBO));
            }
        }
        return result;
    }

    public void removeEnvironmentFederation(String envFedId) {
        environmentFederationUnifiedDAO.remove(envFedId);
    }

    public void setEnvironmentFederationUnifiedDAO(
            GenericUnifiedDAO<org.ow2.dragon.persistence.bo.technology.EnvironmentFederation, String> environmentFederationUnifiedDAO) {
        this.environmentFederationUnifiedDAO = environmentFederationUnifiedDAO;
    }

    public List<EnvironmentFederationTO> getAllEnvironmentFederations(
            RequestOptionsTO requestOptionsTO) {
        List<EnvironmentFederationTO> result = new ArrayList<EnvironmentFederationTO>();
        final List<EnvironmentFederation> envFeds = this.environmentFederationUnifiedDAO
                .getAll(this.transfertObjectAssembler.toFederationRequestOptions(requestOptionsTO));
        toEnvFedsTO(result, envFeds);
        return result;
    }

    private void toEnvFedsTO(List<EnvironmentFederationTO> result,
            List<EnvironmentFederation> envFeds) {
        if (!CollectionUtils.isEmpty(envFeds)) {
            for (final EnvironmentFederation envFed : envFeds) {
                result.add(toEnvFedTO(envFed));
            }
        }
    }

    private EnvironmentFederationTO toEnvFedTO(EnvironmentFederation envFed) {
        return this.technologyTransferObjectAssembler.toEnvironmentFederationTO(envFed);
    }

    public List<EnvironmentFederationTO> searchEnvironmentFederations(String searchCriteria,
            List<FederationSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException {
        List<EnvironmentFederationTO> result = new ArrayList<EnvironmentFederationTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createEnvFedSearchProperties(searchedProperties);

        // Search for bo
        List<EnvironmentFederation> envFeds;
        try {
            envFeds = this.environmentFederationUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toFederationRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new TechnologyException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toEnvFedsTO(result, envFeds);
        return result;
    }

    private String[] createEnvFedSearchProperties(
            List<FederationSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(FederationSearchProperties.NAME)) {
                propertiesList.add("name");
            }
            if (searchedProperties.contains(FederationSearchProperties.PATTERN)) {
                propertiesList.add("pattern");
            }
        } else {
            // search on all properties
            propertiesList.add("name");
            propertiesList.add("pattern");
        }
        return propertiesList.toArray(new String[0]);
    }
}
