/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * WSDLManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.wsdl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeoutException;

import javax.xml.namespace.QName;
import javax.xml.transform.TransformerException;

import org.apache.log4j.Logger;
import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.api.service.wsdl.WSDLManager;
import org.ow2.dragon.api.service.wsdl.WSDLServiceException;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.KRKeyNames;
import org.ow2.dragon.persistence.bo.common.KRKeyValues;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Protocol;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.deployment.Transport;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.bo.specification.SpecifiedOperation;
import org.ow2.dragon.persistence.bo.specification.SpecifiedOperation.OpStyleType;
import org.ow2.dragon.persistence.bo.specification.SpecifiedOperation.OpTmType;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.RequestOptions;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.dao.deployment.BindingDAO;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.deployment.ProtocolDAO;
import org.ow2.dragon.persistence.dao.deployment.TechnicalServiceDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceInterfaceDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceSpecificationDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.ContentType;
import org.ow2.dragon.util.InputStreamUtil;
import org.ow2.dragon.util.StringHelper;
import org.ow2.dragon.util.UDDIStandardTModelKeys;
import org.ow2.dragon.util.XMLUtil;
import org.ow2.easywsdl.schema.SchemaFactory;
import org.ow2.easywsdl.schema.api.Documentation;
import org.ow2.easywsdl.schema.api.SchemaException;
import org.ow2.easywsdl.schema.api.SchemaReader;
import org.ow2.easywsdl.schema.api.absItf.AbsItfSchema;
import org.ow2.easywsdl.wsdl.WSDLFactory;
import org.ow2.easywsdl.wsdl.api.Binding;
import org.ow2.easywsdl.wsdl.api.BindingOperation;
import org.ow2.easywsdl.wsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.Endpoint;
import org.ow2.easywsdl.wsdl.api.InterfaceType;
import org.ow2.easywsdl.wsdl.api.Operation;
import org.ow2.easywsdl.wsdl.api.Service;
import org.ow2.easywsdl.wsdl.api.WSDLException;
import org.ow2.easywsdl.wsdl.api.WSDLReader;
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfDescription;
import org.ow2.easywsdl.wsdl.api.abstractItf.AbsItfBinding.BindingConstants;
import org.ow2.easywsdl.wsdl.util.InputStreamForker;
import org.springframework.util.ResourceUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class WSDLManagerImpl implements WSDLManager {

    private MetadataService metadataService;

    private ServiceSpecificationDAO serviceSpecificationDAO;

    private GenericUnifiedDAO<ServiceSpecification, String> serviceSpecificationUnifiedDAO;

    private ServiceInterfaceDAO serviceInterfaceDAO;

    private BindingDAO bindingDAO;

    private TechnicalServiceDAO technicalServiceDAO;

    private EndpointDAO endpointDAO;

    private ProtocolDAO protocolDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    private WSDLReader wsdlReader;

    private SchemaReader schemaReader;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private Logger logger = Logger.getLogger(this.getClass());

    public WSDLManagerImpl() throws WSDLServiceException {
        super();
        WSDLFactory wsdlFactory = null;
        try {
            wsdlFactory = WSDLFactory.newInstance();
        } catch (WSDLException e) {

        }
        try {
            this.wsdlReader = wsdlFactory.newWSDLReader();
        } catch (WSDLException e1) {
            throw new WSDLServiceException("Can't create wsdl reader", e1);
        }

        SchemaFactory schemaFactory = null;
        try {
            schemaFactory = SchemaFactory.newInstance();
            this.schemaReader = schemaFactory.newSchemaReader();
        } catch (SchemaException e) {
            throw new WSDLServiceException("Can't create schema reader factory", e);
        }

    }

    private void createBindingOps(final Map<Operation, SpecifiedOperation> createdIntOps,
            final Binding binding,
            final org.ow2.dragon.persistence.bo.deployment.Binding dragonBinding) {
        final List<BindingOperation> operations = binding.getBindingOperations();
        if (operations != null) {
            for (final BindingOperation operation : operations) {
                // TODO Improve signature extraction
                String signature = operation.getOperation().getQName().toString();

                // Binding ops are linked to a binding so if the binding already
                // exist, operation already exist too.
                final org.ow2.dragon.persistence.bo.deployment.BindingOperation dragonBindingOp = new org.ow2.dragon.persistence.bo.deployment.BindingOperation();
                dragonBindingOp.setSignature(signature);
                dragonBindingOp.setStyle(this.getBindingOpStyle(operation));

                createdIntOps.get(operation.getOperation()).addBindingOp(dragonBindingOp);
                dragonBinding.addBindingOp(dragonBindingOp);
            }
        }
    }

    private void createBindings(final Description description,
            final ServiceSpecification serviceSpecification,
            final Map<String, ServiceInterface> createdInterfaces,
            final Map<Operation, SpecifiedOperation> createdIntOps,
            final Map<String, org.ow2.dragon.persistence.bo.deployment.Binding> createdBindings)
            throws WSDLServiceException {
        final List<Binding> bindings = description.getBindings();
        if (bindings != null) {
            for (final Binding binding : bindings) {
                final QName bindingName = binding.getQName();
                org.ow2.dragon.persistence.bo.deployment.Binding dragonBinding = null;

                final String interfaceName = binding.getInterface().getQName().toString();
                ServiceInterface linkedInterface = createdInterfaces.get(interfaceName);

                /*
                 * Try to identify an already existing binding with the same
                 * QName
                 */
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<org.ow2.dragon.persistence.bo.deployment.Binding> bindingsBO = bindingDAO
                        .searchEquals(new String[] { bindingName.toString() },
                                new String[] { "fullName" }, requestOptions);
                if (bindingsBO != null && !bindingsBO.isEmpty()) {
                    // Binding already exist, skip creation/update for
                    // the moment
                    dragonBinding = bindingsBO.get(0);

                } else {
                    dragonBinding = new org.ow2.dragon.persistence.bo.deployment.Binding();
                    dragonBinding.setFullName(bindingName.toString());

                    Name nameObject = new Name();
                    nameObject.setLangCode("en");
                    nameObject.setName(bindingName.getLocalPart());
                    dragonBinding.setName(nameObject);

                    /*
                     * Add descriptions
                     */
                    final Documentation bindingDoc = binding.getDocumentation();
                    String bindingDocContent = null;
                    if (bindingDoc != null) {
                        bindingDocContent = bindingDoc.getContent();
                    }
                    if (!StringHelper.isNullOrEmpty(bindingDocContent)) {
                        org.ow2.dragon.persistence.bo.common.Description description1 = new org.ow2.dragon.persistence.bo.common.Description();
                        description1.setLangCode("en");
                        description1.setDescription(bindingDocContent);
                        dragonBinding.addDescription(description1);
                    }
                    org.ow2.dragon.persistence.bo.common.Description description2 = new org.ow2.dragon.persistence.bo.common.Description();
                    description2.setLangCode("en");
                    description2.setDescription("This is a wsdl binding specification.");
                    dragonBinding.addDescription(description2);

                    /*
                     * Add categories
                     */
                    setTModelCategory(dragonBinding, UDDIStandardTModelKeys.XML_NAMESPACE,
                            KRKeyNames.BINDING_NS, bindingName.getNamespaceURI());
                    setTModelCategory(dragonBinding, UDDIStandardTModelKeys.WSDL_TYPES,
                            KRKeyNames.WSDL_TYPE, KRKeyValues.WSDL_TYPE_BINDING);
                    setTModelCategory(dragonBinding, UDDIStandardTModelKeys.WSDL_PORTTYPEREFERENCE,
                            KRKeyNames.PORTTYPE_REFERENCE, linkedInterface.getId());
                    setTModelCategory(dragonBinding, UDDIStandardTModelKeys.CATEGORIZATION_TYPES,
                            KRKeyNames.UDDI_TYPES, KRKeyValues.WSDL_SPEC);

                    /*
                     * Link Protocol and Transport
                     */
                    this.linkProtocolAndTransport(binding, dragonBinding);

                    /*
                     * Add binding operations
                     */
                    this.createBindingOps(createdIntOps, binding, dragonBinding);
                }

                /*
                 * Link binding to interface
                 */
                linkedInterface.addBinding(dragonBinding);

                serviceSpecification.addBinding(dragonBinding);
                createdBindings.put(bindingName.toString(), dragonBinding);
                this.bindingDAO.save(dragonBinding);
            }
        }
    }

    private String[] createDragonObjectModel(final Description description,
            final SimpleFile storedWSDL, final String wsdlSha1) throws WSDLServiceException {

        /*
         * Validate description : for the moment, test if its a concrete service
         * description (with service(s), endpoint(s), binding(s) and
         * interface(s)).
         */

        validateDescription(description);

        /*
         * Dragon service specification and service definition language file
         */
        final ServiceSpecification serviceSpecification = this.createServiceSpecification(
                description, storedWSDL, wsdlSha1);

        /*
         * Dragon interface and interface ops
         */
        final Map<String, ServiceInterface> createdInterfaces = new HashMap<String, ServiceInterface>();
        final Map<Operation, SpecifiedOperation> createdIntOps = new HashMap<Operation, SpecifiedOperation>();
        this.createInterfacesAndIntOps(description, serviceSpecification, createdInterfaces,
                createdIntOps);

        /*
         * Dragon bindings, protocol and binding ops
         */
        final Map<String, org.ow2.dragon.persistence.bo.deployment.Binding> createdBindings = new HashMap<String, org.ow2.dragon.persistence.bo.deployment.Binding>();
        this.createBindings(description, serviceSpecification, createdInterfaces, createdIntOps,
                createdBindings);

        /*
         * Dragon Services and endpoints
         */
        final List<String> result = new ArrayList<String>();
        this.createServices(serviceSpecification, createdInterfaces, createdBindings, description,
                result);

        return result.toArray(new String[0]);
    }

    /**
     * Validate description : for the moment, test if its a concrete service
     * description (with service(s), endpoint(s), binding(s) and interface(s)).
     * 
     * @param description
     * @throws WSDLServiceException
     */
    private void validateDescription(Description description) throws WSDLServiceException {
        List<Service> services = description.getServices();
        List<Binding> bindings = description.getBindings();
        List<InterfaceType> interfaceTypes = description.getInterfaces();
        if (services == null || services.isEmpty() || bindings == null || bindings.isEmpty()
                || interfaceTypes == null || interfaceTypes.isEmpty()) {
            throw new WSDLServiceException("Abstract wsdl aren't supported for the moment. "
                    + "Supported wsdl must contains PortTypes "
                    + "(Interfaces), Bindings, Services.");
        }
    }

    private void createEndpoints(
            final Map<String, org.ow2.dragon.persistence.bo.deployment.Binding> createdBindings,
            final Service service, final TechnicalService dragonTechnicalService,
            ServiceSpecification serviceSpecification) {
        final List<Endpoint> endpoints = service.getEndpoints();
        if (endpoints != null) {
            for (final Endpoint endpoint : endpoints) {
                String dragonEndpointName = endpoint.getName();
                final org.ow2.dragon.persistence.bo.deployment.Endpoint dragonEndpoint;

                /*
                 * Try to identify an already existing endpoint with the same
                 * name
                 */
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<org.ow2.dragon.persistence.bo.deployment.Endpoint> endpointsBO = endpointDAO
                        .searchEquals(new String[] { dragonEndpointName }, new String[] { "name" },
                                requestOptions);
                if (endpointsBO != null && !endpointsBO.isEmpty()) {
                    // Endpoint already exist, skip creation/update for the
                    // moment
                    dragonEndpoint = endpointsBO.get(0);

                } else {
                    dragonEndpoint = new org.ow2.dragon.persistence.bo.deployment.Endpoint();
                    dragonEndpoint.setName(dragonEndpointName);
                    final URI endpointAddress = URI.create(endpoint.getAddress());
                    if (endpointAddress != null) {
                        dragonEndpoint.setNetworkAddress(endpointAddress.toString());
                    } else {
                        dragonEndpoint.setNetworkAddress("Unknown");
                    }
                }
                dragonTechnicalService.addEndpoint(dragonEndpoint);

                /*
                 * Link endpoint to Service spec
                 */
                dragonEndpoint.addServiceSpec(serviceSpecification);

                /*
                 * Link endpoint to binding
                 */
                final String bindingName = endpoint.getBinding().getQName().toString();
                createdBindings.get(bindingName).addEndpoint(dragonEndpoint);
            }
        }
    }

    private void createInterfacesAndIntOps(final Description description,
            final ServiceSpecification serviceSpecification,
            final Map<String, ServiceInterface> createdInterfaces,
            final Map<Operation, SpecifiedOperation> createdIntOps) throws WSDLServiceException {
        final List<InterfaceType> interfaces = description.getInterfaces();
        if (interfaces != null) {
            for (final InterfaceType wsdlInterface : interfaces) {
                final QName interfaceName = wsdlInterface.getQName();
                final ServiceInterface serviceInterface;

                /*
                 * Try to identify an already existing interface with the same
                 * QName
                 */
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<ServiceInterface> serviceInterfacesBO = serviceInterfaceDAO.searchEquals(
                        new String[] { interfaceName.toString() }, new String[] { "fullName" },
                        requestOptions);
                if (serviceInterfacesBO != null && !serviceInterfacesBO.isEmpty()) {
                    // Service interface already exist, skip creation/update for
                    // the moment
                    serviceInterface = serviceInterfacesBO.get(0);
                    // Simply map persisted int op to wsdl int op
                    this.mapIntOps(createdIntOps, wsdlInterface, serviceInterface);
                } else {
                    serviceInterface = new ServiceInterface();
                    serviceInterface.setFullName(interfaceName.toString());

                    Name nameObject = new Name();
                    nameObject.setLangCode("en");
                    nameObject.setName(interfaceName.getLocalPart());
                    serviceInterface.setName(nameObject);

                    /*
                     * Add descriptions
                     */
                    final Documentation interfaceDoc = wsdlInterface.getDocumentation();
                    String interfaceDocContent = null;
                    if (interfaceDoc != null) {
                        interfaceDocContent = interfaceDoc.getContent();
                    }
                    if (!StringHelper.isNullOrEmpty(interfaceDocContent)) {
                        org.ow2.dragon.persistence.bo.common.Description description1 = new org.ow2.dragon.persistence.bo.common.Description();
                        description1.setLangCode("en");
                        description1.setDescription(interfaceDocContent);
                        serviceInterface.addDescription(description1);
                    }
                    org.ow2.dragon.persistence.bo.common.Description description2 = new org.ow2.dragon.persistence.bo.common.Description();
                    description2.setLangCode("en");
                    description2.setDescription("This is a wsdl interface specification.");
                    serviceInterface.addDescription(description2);

                    /*
                     * Add categories
                     */
                    setTModelCategory(serviceInterface, UDDIStandardTModelKeys.XML_NAMESPACE,
                            KRKeyNames.PORTTYPE_NS, interfaceName.getNamespaceURI());
                    setTModelCategory(serviceInterface, UDDIStandardTModelKeys.WSDL_TYPES,
                            KRKeyNames.WSDL_TYPE, KRKeyValues.WSDL_TYPE_PORTTYPE);

                    /*
                     * Dragon interface operations
                     */
                    this.createIntOps(createdIntOps, wsdlInterface, serviceInterface);
                }
                serviceSpecification.addServiceInterface(serviceInterface);
                createdInterfaces.put(interfaceName.toString(), serviceInterface);
                this.serviceInterfaceDAO.save(serviceInterface);
            }
        }
    }

    private void mapIntOps(Map<Operation, SpecifiedOperation> createdIntOps,
            InterfaceType wsdlInterface, ServiceInterface serviceInterface)
            throws WSDLServiceException {
        List<Operation> wsdlOps = wsdlInterface.getOperations();
        Set<SpecifiedOperation> specOps = serviceInterface.getSpecifiedOps();
        if (wsdlOps.size() == specOps.size()) {
            for (Operation operation : wsdlOps) {
                for (SpecifiedOperation specifiedOperation : specOps) {
                    /*
                     * if(specifiedOperation.getSignature().equals(operation.
                     * getSignature())) {
                     */
                    if (specifiedOperation.getName().equals(operation.getQName().toString())) {
                        createdIntOps.put(operation, specifiedOperation);
                        break;
                    }
                }
            }
            if (createdIntOps.keySet() != null
                    && createdIntOps.keySet().size() < (Math.max(wsdlOps.size(), specOps.size()))) {
                throw new WSDLServiceException("You're trying to register an interface that "
                        + "already exist in registry but contains different operations: "
                        + serviceInterface.getFullName());
            }
        } else {
            throw new WSDLServiceException("You're trying to register an interface "
                    + "that already exist in registry but contains different operations: "
                    + serviceInterface.getFullName());
        }
    }

    private void createIntOps(final Map<Operation, SpecifiedOperation> createdIntOps,
            final InterfaceType wsdlInterface, final ServiceInterface serviceInterface) {
        final List<Operation> intOps = wsdlInterface.getOperations();
        if (intOps != null) {
            for (final Operation interfaceOperation : intOps) {
                String signature = interfaceOperation.getSignature();
                String name = interfaceOperation.getQName().toString();
                Documentation documentation = interfaceOperation.getDocumentation();
                String docContent = null;
                if (documentation != null) {
                    docContent = documentation.getContent();
                }

                // Operations are linked to an interface so they can't exist if
                // the interface doesn't exist

                final SpecifiedOperation specOp = new SpecifiedOperation();
                specOp.setName(name);
                specOp.setSignature(signature);
                specOp.setPurpose(docContent);
                specOp.setTransmission(this.getInterfaceOpTransmission(interfaceOperation));

                serviceInterface.addSpecifiedOp(specOp);
                createdIntOps.put(interfaceOperation, specOp);
            }
        }
    }

    private void linkProtocolAndTransport(final Binding binding,
            final org.ow2.dragon.persistence.bo.deployment.Binding dragonBinding) {

        // Retrieve protocol
        BindingConstants protocol = binding.getTypeOfBinding();

        if (protocol != null) {
            // SOAP 1.1 case
            if (protocol.equals(BindingConstants.SOAP11_BINDING4WSDL11)) {
                // Link a SOAP 1.1 protocol
                Protocol soap11Protocol = protocolDAO.get(UDDIStandardTModelKeys.PROTOCOL_SOAP);
                dragonBinding.addProtocol(soap11Protocol);

                // detect and link the transport
                URI transportURI = URI.create(binding.getTransportProtocol());
                Transport transport = detectTransport(transportURI);
                if (transport != null) {
                    dragonBinding.addTransport(transport);
                }
            }
            // SOAP 1.2 case
            else if (protocol.equals(BindingConstants.SOAP12_BINDING4WSDL11)
                    || protocol.equals(BindingConstants.SOAP_BINDING4WSDL20)) {
                // Link a SOAP 1.2 protocol
                Protocol soap12Protocol = protocolDAO.get(UDDIStandardTModelKeys.PROTOCOL_SOAP12);
                dragonBinding.addProtocol(soap12Protocol);

                // detect and link the transport
                URI transportURI = URI.create(binding.getTransportProtocol());
                Transport transport = detectTransport(transportURI);
                if (transport != null) {
                    dragonBinding.addTransport(transport);
                }
            }
            // HTTP case
            else if (protocol.equals(BindingConstants.HTTP11_BINDING4WSDL11)
                    || protocol.equals(BindingConstants.HTTP_BINDING4WSDL20)) {
                // Link a HTTP protocol and a HTTP transport
                Protocol httpProtocol = protocolDAO.get(UDDIStandardTModelKeys.PROTOCOL_HTTP);
                Transport httpTransport = (Transport) universalUnifiedDAO.get(Transport.class,
                        UDDIStandardTModelKeys.TRANSPORT_HTTP);

                dragonBinding.addProtocol(httpProtocol);
                dragonBinding.addTransport(httpTransport);
            }
        }
    }

    // FIXME find a better way to detect transport based on a list of known uris
    private Transport detectTransport(URI transportURI) {
        Transport transport = null;
        if (transportURI != null) {
            if (transportURI.toString().toLowerCase().contains("http")) {
                transport = (Transport) universalUnifiedDAO.get(Transport.class,
                        UDDIStandardTModelKeys.TRANSPORT_HTTP);
            } else if (transportURI.toString().toLowerCase().contains("ftp")) {
                transport = (Transport) universalUnifiedDAO.get(Transport.class,
                        UDDIStandardTModelKeys.TRANSPORT_FTP);
            } else if (transportURI.toString().toLowerCase().contains("smtp")) {
                transport = (Transport) universalUnifiedDAO.get(Transport.class,
                        UDDIStandardTModelKeys.TRANSPORT_SMTP);
            }
        }
        return transport;
    }

    private void createServices(final ServiceSpecification serviceSpecification,
            final Map<String, ServiceInterface> createdInterfaces,
            final Map<String, org.ow2.dragon.persistence.bo.deployment.Binding> createdBindings,
            final Description description, final List<String> result) {
        final List<Service> services = description.getServices();
        if (services != null) {
            for (final Service service : services) {
                final QName serviceName = service.getQName();
                final Documentation serviceDoc = service.getDocumentation();
                String serviceDocContent = null;
                if (serviceDoc != null) {
                    serviceDocContent = serviceDoc.getContent();
                }
                final TechnicalService dragonTechnicalService;

                /*
                 * Try to identify an already existing service with the same
                 * name
                 */
                RequestOptions requestOptions = new RequestOptions();
                requestOptions.setCaseSensitive(true);
                List<TechnicalService> servicesBO = technicalServiceDAO.searchEquals(
                        new String[] { serviceName.toString() }, new String[] { "fullName" },
                        requestOptions);
                if (servicesBO != null && !servicesBO.isEmpty()) {
                    // Service already exist, skip creation/update for the
                    // moment
                    dragonTechnicalService = servicesBO.get(0);
                    /*
                     * Link service to interface Removed 'cause wsdl 1.1 spec
                     * allow multiple interfaces for the same service
                     */
                    /*
                     * String interfaceName = null; try { interfaceName =
                     * service.getInterface().getQName().toString(); } catch
                     * (WSDLException e) { throw new
                     * WSDLServiceException("Can't retrieve interface from service"
                     * , e); } createdInterfaces.get(interfaceName)
                     * .addTechnicalService(dragonTechnicalService);
                     */

                    /*
                     * Create Dragon endpoints cause it could have new ones
                     */
                    this.createEndpoints(createdBindings, service, dragonTechnicalService,
                            serviceSpecification);
                } else {
                    dragonTechnicalService = new TechnicalService();
                    dragonTechnicalService.setFullName(serviceName.toString());

                    // Only one name (extracted from the WSDL Service name)
                    Name nameObject = new Name();
                    nameObject.setLangCode("en");
                    nameObject.setName(serviceName.getLocalPart());
                    dragonTechnicalService.addName(nameObject);

                    // Add categories for each part of the name
                    setServiceCategory(dragonTechnicalService,
                            UDDIStandardTModelKeys.XML_LOCALNAME, KRKeyNames.SERVICE_LOCAL_NAME,
                            serviceName.getLocalPart());
                    setServiceCategory(dragonTechnicalService,
                            UDDIStandardTModelKeys.XML_NAMESPACE, KRKeyNames.SERVICE_NS,
                            serviceName.getNamespaceURI());
                    setServiceCategory(dragonTechnicalService, UDDIStandardTModelKeys.WSDL_TYPES,
                            KRKeyNames.WSDL_TYPE, KRKeyValues.WSDL_TYPE_SERVICE);

                    /*
                     * Add a description extracted from the WSDL service
                     * documentation
                     */
                    if (!StringHelper.isNullOrEmpty(serviceDocContent)) {
                        org.ow2.dragon.persistence.bo.common.Description descriptionObject = new org.ow2.dragon.persistence.bo.common.Description();
                        descriptionObject.setLangCode("en");
                        descriptionObject.setDescription(serviceDocContent);
                        dragonTechnicalService.addDescription(descriptionObject);
                    }

                    /*
                     * Link service to interface Removed 'cause wsdl 1.1 spec
                     * allow multiple interfaces for the same service
                     */
                    /*
                     * String interfaceName = null; try { interfaceName =
                     * service.getInterface().getQName().toString(); } catch
                     * (WSDLException e) { throw new
                     * WSDLServiceException("Can't retrieve interface from service"
                     * , e); } createdInterfaces.get(interfaceName)
                     * .addTechnicalService(dragonTechnicalService);
                     */
                    this.technicalServiceDAO.save(dragonTechnicalService);
                    /*
                     * Dragon endpoints
                     */
                    this.createEndpoints(createdBindings, service, dragonTechnicalService,
                            serviceSpecification);

                    result.add(dragonTechnicalService.getId());
                }

                serviceSpecification.addTechnicalService(dragonTechnicalService);
                this.technicalServiceDAO.save(dragonTechnicalService);
            }
        }
    }

    private void setTModelCategory(TModel tmodel, String tModelKey, String categoryName,
            String categoryValue) throws WSDLServiceException {

        // Create the category bag if it doesn't exist
        CategoryBag bag = tmodel.getCategoryBag();
        if (bag == null) {
            bag = new CategoryBag();
            tmodel.setCategoryBag(bag);
        }

        // Try to find a keyed ref with the same TModel key, if it exist, simply
        // replace the value...
        List<KeyedReference> keyedReferences = bag.getKeyedReferences();
        boolean serviceCategoryExist = false;
        if (keyedReferences != null) {
            for (KeyedReference keyedReference : keyedReferences) {
                if (tModelKey.equals(keyedReference.getTmodel().getId())) {
                    keyedReference.setKeyValue(categoryValue);
                    serviceCategoryExist = true;
                    break;
                }
            }
        }

        // ... if not, create a new category
        if (!serviceCategoryExist) {
            // search the tModel to link
            TModel tModelToLink = (TModel) universalUnifiedDAO.get(TModel.class, tModelKey);
            if (tModelToLink == null) {
                throw new WSDLServiceException("Can't find tmodel for key: " + tModelKey);
            }
            // add the new category
            addKeyedReference(bag, categoryName, categoryValue, tModelToLink);
        }
    }

    private void setServiceCategory(TechnicalService service, String tModelKey,
            String categoryName, String categoryValue) {

        // Create the category bag if it doesn't exist
        CategoryBag bag = service.getCategoryBag();
        if (bag == null) {
            bag = new CategoryBag();
            service.setCategoryBag(bag);
        }

        // Try to find a keyed ref with the same TModel key, if it exist, simply
        // replace the value...
        List<KeyedReference> keyedReferences = bag.getKeyedReferences();
        boolean serviceCategoryExist = false;
        if (keyedReferences != null) {
            for (KeyedReference keyedReference : keyedReferences) {
                if (tModelKey.equals(keyedReference.getTmodel().getId())) {
                    keyedReference.setKeyValue(categoryValue);
                    serviceCategoryExist = true;
                    break;
                }
            }
        }

        // ... if not, create a new category
        if (!serviceCategoryExist) {
            // search the tModel to link
            TModel tModelToLink = (TModel) universalUnifiedDAO.get(TModel.class, tModelKey);
            // add the new category
            addKeyedReference(bag, categoryName, categoryValue, tModelToLink);
        }
    }

    private void addKeyedReference(CategoryBag bag, String name, String value, TModel tModel) {
        KeyedReference keyedReference = new KeyedReference();
        keyedReference.setKeyName(name);
        keyedReference.setKeyValue(value);
        keyedReference.setTmodel(tModel);
        bag.addKeyedReference(keyedReference);
    }

    private ServiceSpecification createServiceSpecification(Description description,
            final SimpleFile storedWSDL, final String wsdlSha1) throws WSDLServiceException {
        final ServiceSpecification serviceSpecification = new ServiceSpecification();

        // Add spec name
        Name nameObject = new Name();
        nameObject.setLangCode("en");
        nameObject.setName(this.retrieveServSpecName(description));
        serviceSpecification.setName(nameObject);

        // Add spec descriptions
        final Documentation descriptionDoc = description.getDocumentation();
        String descriptionDocContent = null;
        if (descriptionDoc != null) {
            descriptionDocContent = descriptionDoc.getContent();
        }
        if (!StringHelper.isNullOrEmpty(descriptionDocContent)) {
            org.ow2.dragon.persistence.bo.common.Description description1 = new org.ow2.dragon.persistence.bo.common.Description();
            description1.setLangCode("en");
            description1.setDescription(descriptionDocContent);
            serviceSpecification.addDescription(description1);
        }
        org.ow2.dragon.persistence.bo.common.Description description2 = new org.ow2.dragon.persistence.bo.common.Description();
        description2.setLangCode("en");
        description2.setDescription("This is a wsdl specification.");
        serviceSpecification.addDescription(description2);

        serviceSpecification.setWsdlSha1(wsdlSha1);

        // TODO Only attach the original WSDL file to the service spec
        serviceSpecification.addOverviewDoc(storedWSDL);

        this.serviceSpecificationDAO.save(serviceSpecification);
        return serviceSpecification;
    }

    /**
     * Retrieve the description qname if specified or create a new one based on
     * the tns + "Undefined" local part
     * 
     * @param description
     * @return
     * @throws WSDLServiceException
     */
    private String retrieveServSpecName(Description description) throws WSDLServiceException {
        String name = null;
        try {
            QName qname = description.getQName();
            if (qname != null) {
                name = qname.toString();
            } else {
                name = "{" + description.getTargetNamespace() + "}Undefined";
            }
        } catch (WSDLException e) {
            throw new WSDLServiceException("Can't retrieve service spec name", e);
        }
        return name;
    }

    public BindingDAO getBindingDAO() {
        return this.bindingDAO;
    }

    private OpStyleType getBindingOpStyle(final BindingOperation bindingOperation) {
        OpStyleType opStyle = OpStyleType.UNKNOWN;
        final Binding.StyleConstant style = bindingOperation.getStyle();
        if (style != null) {
            if (Binding.StyleConstant.RPC.equals(style)) {
                opStyle = OpStyleType.PARAMETERS;
            } else if (Binding.StyleConstant.DOCUMENT.equals(style)) {
                opStyle = OpStyleType.DOCUMENT;
            }
        }
        return opStyle;
    }

    private OpTmType getInterfaceOpTransmission(final Operation interfaceOperation) {
        OpTmType opTransmission = OpTmType.UNKNOWN;
        if (Operation.MEPPatternConstants.IN_ONLY.equals(interfaceOperation.getPattern())
                || Operation.MEPPatternConstants.ROBUST_IN_ONLY.equals(interfaceOperation
                        .getPattern())
                || Operation.MEPPatternConstants.OUT_ONLY.equals(interfaceOperation.getPattern())
                || Operation.MEPPatternConstants.ROBUST_OUT_ONLY.equals(interfaceOperation
                        .getPattern())) {
            opTransmission = OpTmType.ONEWAY;
        } else if (Operation.MEPPatternConstants.IN_OUT.equals(interfaceOperation.getPattern())
                || Operation.MEPPatternConstants.IN_OPTIONAL_OUT.equals(interfaceOperation
                        .getPattern())
                || Operation.MEPPatternConstants.OUT_IN.equals(interfaceOperation.getPattern())
                || Operation.MEPPatternConstants.OUT_OPTIONAL_IN.equals(interfaceOperation
                        .getPattern())) {
            opTransmission = OpTmType.REQUEST_RESPONSE;
        }
        return opTransmission;
    }

    private String getLanguageFileName(final String wsdlURI) throws WSDLServiceException {

        final URI uri = URI.create(wsdlURI);
        final String path = uri.getPath();
        final String[] pathElements = path.split("/");
        String languageFileName = pathElements[pathElements.length - 1];

        return languageFileName;
    }

    public MetadataService getMetadataManager() {
        return this.metadataService;
    }

    public ServiceInterfaceDAO getServiceInterfaceDAO() {
        return this.serviceInterfaceDAO;
    }

    public ServiceSpecificationDAO getServiceSpecificationDAO() {
        return this.serviceSpecificationDAO;
    }

    public TechnicalServiceDAO getTechnicalServiceDAO() {
        return this.technicalServiceDAO;
    }

    @CheckAllArgumentsNotNull
    public List<ServiceSpecificationTO> getServSpecsForEndpoint(final String epId)
            throws WSDLServiceException {
        List<ServiceSpecificationTO> result = new ArrayList<ServiceSpecificationTO>();
        // Retrieve tech service
        final org.ow2.dragon.persistence.bo.deployment.Endpoint endpoint = this.endpointDAO
                .get(epId);
        if (endpoint == null) {
            throw new WSDLServiceException("Can't find endpoint for the given id: " + epId);
        }

        final Set<ServiceSpecification> specs = endpoint.getServiceSpecifications();

        for (ServiceSpecification serviceSpecification : specs) {
            result.add(transfertObjectAssembler.toServiceSpecTO(serviceSpecification));
        }
        return result;
    }

    @CheckAllArgumentsNotNull
    public List<ServiceSpecificationTO> getServSpecsForTechServ(final String techServiceId)
            throws WSDLServiceException {
        List<ServiceSpecificationTO> result = new ArrayList<ServiceSpecificationTO>();

        // Retrieve tech service
        final TechnicalService technicalService = this.technicalServiceDAO.get(techServiceId);
        if (technicalService == null) {
            throw new WSDLServiceException("Can't find technical service for the given id: "
                    + techServiceId);
        }

        // Location is only a SimpleFile Id for the moment
        final Set<ServiceSpecification> specs = technicalService.getServiceSpecifications();

        for (ServiceSpecification serviceSpecification : specs) {
            result.add(transfertObjectAssembler.toServiceSpecTO(serviceSpecification));
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.service.wsdl.WSDLImporter#importServiceDefFile(org.w3c
     * .dom.Document)
     */
    @CheckAllArgumentsNotNull
    public String[] importServiceDefFile(final Document domDocument) throws WSDLServiceException,
            TimeoutException {
        return this.importServiceDefFile(domDocument, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.service.wsdl.WSDLImporter#importServiceDefFile(org.w3c
     * .dom.Element)
     */
    public String[] importServiceDefFile(final Element domElement) throws WSDLServiceException {
        throw new RuntimeException("not implemented method");
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.service.wsdl.WSDLImporter#importServiceDefFile(org.xml
     * .sax.InputSource)
     */
    @CheckArgumentsNotNull
    public String[] importServiceDefFile(final InputSource inputSource, final String wsdlFileName)
            throws WSDLServiceException, TimeoutException {
        return this.importServiceDefFile(inputSource, null, wsdlFileName);
    }

    @CheckAllArgumentsNotNull
    public String[] importServiceDefFile(final InputSource inputSource)
            throws WSDLServiceException, TimeoutException {
        return this.importServiceDefFile(inputSource, null, null);
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(String wsdlContent, String wsdlFileName)
            throws WSDLServiceException, TimeoutException {
        return this.importServiceDefFile(wsdlContent, null, wsdlFileName);
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(String wsdlContent, Map<String, String> imports,
            String wsdlFileName) throws WSDLServiceException, TimeoutException {
        // Create the source wsdl InputSource
        InputSource wsdlSource = new InputSource(new ByteArrayInputStream(wsdlContent.getBytes()));

        // Create imports InputSources
        Map<String, InputSource> importSources = new HashMap<String, InputSource>();
        if (imports != null && imports.size() > 0) {
            Set<String> keys = imports.keySet();
            for (String key : keys) {
                InputSource importSource = new InputSource(new ByteArrayInputStream(imports
                        .get(key).getBytes()));
                importSources.put(key, importSource);
            }
        }
        return importServiceDefFile(wsdlSource, importSources, wsdlFileName);
    }

    private boolean isSchema(Document importDoc) {
        boolean result = false;
        String ns = importDoc.getDocumentElement().getNamespaceURI();
        if (org.ow2.easywsdl.schema.impl.Constants.SCHEMA_NAMESPACE.equals(ns)) {
            result = true;
        }
        return result;
    }

    private boolean isWSDL(Document importDoc) {
        boolean result = false;
        String ns = importDoc.getDocumentElement().getNamespaceURI();
        if (org.ow2.easywsdl.wsdl.impl.wsdl20.Constants.WSDL_20_NAMESPACE.equals(ns)
                || org.ow2.easywsdl.wsdl.impl.wsdl11.Constants.WSDL_11_NAMESPACE.equals(ns)) {
            result = true;
        }
        return result;
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(final URI wsdlURI, final Map<String, URI> imports,
            final String wsdlFileName) throws WSDLServiceException, TimeoutException {

        InputSource inputSource;
        try {
            // InputStream inputStream = wsdlURI.toURL().openStream();
            InputStream inputStream = ResourceUtils.getURL(wsdlURI.toString()).openStream();
            inputSource = new InputSource(inputStream);
        } catch (Exception e) {
            throw new WSDLServiceException("Can't read WSDL file at URL: " + wsdlURI.toString());
        }

        // Create imports InputSources
        Map<String, InputSource> importSources = null;
        if (imports != null && imports.size() > 0) {
            importSources = new HashMap<String, InputSource>();
            Set<String> keys = imports.keySet();
            for (String key : keys) {
                InputSource importSource = null;
                try {
                    importSource = new InputSource(ResourceUtils.getURL(wsdlURI.toString())
                            .openStream());
                } catch (Exception e) {
                    throw new WSDLServiceException(
                            "Can't create input source from the given import Document.", e);
                }
                importSources.put(key, importSource);
            }
        }

        return this.importServiceDefFile(inputSource);
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(final URI wsdlURI, final String wsdlFileName)
            throws WSDLServiceException, TimeoutException {

        return this.importServiceDefFile(wsdlURI, null, wsdlFileName);
    }

    @CheckAllArgumentsNotNull
    public String[] importServiceDefFile(final URI wsdlURI) throws WSDLServiceException,
            TimeoutException {

        return this.importServiceDefFile(wsdlURI, null, null);
    }

    /**
     * We consider that a description is already registered if we find a
     * ServiceSpecification in DB which is related to an overview doc with : <br/>
     * a {@link ContentType} equals to "XML", <br/>
     * a useType equals to "wsdl" and <br/>
     * a sha1 signature equals to the given sha1
     * 
     * @param originalXmlSHA1
     * @return true if already registered, false otherwise
     */
    private boolean alreadyRegisteredDesc(String originalXmlSHA1) {
        boolean result = false;

        logger.debug("Provided wsdl hash : " + originalXmlSHA1);

        String[] criteria = new String[] { originalXmlSHA1 };

        String[] searchedProperties = new String[] { "wsdlSha1" };

        List<ServiceSpecification> specs = serviceSpecificationUnifiedDAO.searchEquals(criteria,
                searchedProperties, null);
        if (specs != null && !specs.isEmpty()) {
            result = true;
            logger.info("You try to register an already registered WSDL");
        }

        return result;
    }

    public void setBindingDAO(final BindingDAO bindingDAO) {
        this.bindingDAO = bindingDAO;
    }

    public void setEndpointDAO(final EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

    public void setMetadataService(final MetadataService metadataService) {
        this.metadataService = metadataService;
    }

    public void setServiceInterfaceDAO(final ServiceInterfaceDAO serviceInterfaceDAO) {
        this.serviceInterfaceDAO = serviceInterfaceDAO;
    }

    public void setServiceSpecificationDAO(final ServiceSpecificationDAO serviceSpecificationDAO) {
        this.serviceSpecificationDAO = serviceSpecificationDAO;
    }

    public void setTechnicalServiceDAO(final TechnicalServiceDAO technicalServiceDAO) {
        this.technicalServiceDAO = technicalServiceDAO;
    }

    public ProtocolDAO getProtocolDAO() {
        return protocolDAO;
    }

    public void setProtocolDAO(ProtocolDAO protocolDAO) {
        this.protocolDAO = protocolDAO;
    }

    public void setTransfertObjectAssembler(TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    @CheckAllArgumentsNotNull
    public String getWsdlDescAsString(String servSpecId) throws WSDLServiceException {
        String result = null;
        try {
            InputStream inputStream = getWsdlDesc(servSpecId);
            result = StringHelper.toString(inputStream);
        } catch (final IOException e) {
            throw new WSDLServiceException("Can't convert WSDL file content to String", e);
        }
        return result;
    }

    @CheckAllArgumentsNotNull
    public InputStream getWsdlDesc(String servSpecId) throws WSDLServiceException {
        InputStream result = null;
        // Retrieve ServSpec BO
        ServiceSpecification serviceSpecification = serviceSpecificationDAO.get(servSpecId);
        if (serviceSpecification != null) {
            // TODO For the moment there's only one file attached to a service
            // spec wich is the original wsdl doc
            SimpleFile descFile = serviceSpecification.getOverviewDocs().get(0);
            // Retrieve WSDL file content
            try {
                result = this.metadataService.loadMetadataContentAsInputStream(descFile.getId());
            } catch (final MetadataServiceException e) {
                throw new WSDLServiceException("Can't load WSDL file content", e);
            }

        } else {
            throw new WSDLServiceException("Can't find a service specification for id: "
                    + servSpecId);
        }
        return result;
    }

    @CheckAllArgumentsNotNull
    public String[] importServiceDefFile(String wsdlContent) throws WSDLServiceException,
            TimeoutException {
        return this.importServiceDefFile(wsdlContent, null);
    }

    private void addRelatedDocs(String serviceId, Set<SimpleFile> relatedDocs) {
        if (!relatedDocs.isEmpty()) {
            TechnicalService service = technicalServiceDAO.get(serviceId);
            for (SimpleFile relatedDoc : relatedDocs) {
                service.addRelatedDoc(relatedDoc);
            }
            technicalServiceDAO.save(service);
        }
    }

    public String registerRelatedDoc(String serviceId, String mimetype, byte[] docContent)
            throws WSDLServiceException {
        SimpleFile doc = null;
        try {
            doc = metadataService.storeMetadataAndIndexContent(mimetype, docContent);
        } catch (MetadataServiceException e) {
            throw new WSDLServiceException("Can't store document in registry", e);
        }
        Set<SimpleFile> docs = new HashSet<SimpleFile>();
        docs.add(doc);
        addRelatedDocs(serviceId, docs);
        return doc.getId();
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(Document domDocument, String wsdlFileName)
            throws WSDLServiceException, TimeoutException {
        return this.importServiceDefFile(domDocument, null, wsdlFileName);
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(Document domDocument, Map<String, Document> imports,
            String wsdlFileName) throws WSDLServiceException, TimeoutException {
        // Create the source wsdl InputSource
        InputSource wsdlSource = null;
        try {
            wsdlSource = new InputSource(new ByteArrayInputStream(XMLUtil
                    .createStringFromDOMDocument(domDocument).getBytes()));
        } catch (TransformerException e) {
            throw new WSDLServiceException(
                    "Can't create input source from the given wsdl Document.", e);
        }

        // Create imports InputSources
        Map<String, InputSource> importSources = null;
        if (imports != null && imports.size() > 0) {
            importSources = new HashMap<String, InputSource>();
            Set<String> keys = imports.keySet();
            for (String key : keys) {
                InputSource importSource = null;
                try {
                    importSource = new InputSource(new ByteArrayInputStream(XMLUtil
                            .createStringFromDOMDocument(imports.get(key)).getBytes()));
                } catch (TransformerException e) {
                    throw new WSDLServiceException(
                            "Can't create input source from the given import Document.", e);
                }
                importSources.put(key, importSource);
            }
        }

        return this.importServiceDefFile(wsdlSource, importSources, wsdlFileName);
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile2(Document domDocument, String wsdlFileName)
            throws WSDLServiceException, TimeoutException {

        String xmlString;
        try {
            xmlString = XMLUtil.createStringFromDOMDocument(domDocument);
        } catch (TransformerException e) {
            throw new WSDLServiceException("Can't convert Dom to String", e);
        }
        String wsdlSha1;
        try {
            wsdlSha1 = XMLUtil.getXMLSHA1Sign(new ByteArrayInputStream(xmlString.getBytes()), true);
        } catch (Exception e2) {
            throw new WSDLServiceException("Can't get WSDL SHA1 signature", e2);
        }

        /*
         * Check if the given description isn't already registered
         */
        String[] result = null;
        if (!alreadyRegisteredDesc(wsdlSha1)) {
            /*
             * Parse wsdl with easywsdl
             */
            EasyWSDLTimeout easyWSDLTimeout = new EasyWSDLTimeout(wsdlReader, null, null, null);
            Description description = easyWSDLTimeout.execute(5000).getResult();

            /*
             * Store WSDL file in persistent storage
             */
            SimpleFile storedWSDL = null;
            try {
                storedWSDL = this.metadataService.storeMetadata(ContentType.XML, xmlString,
                        wsdlFileName);
            } catch (final MetadataServiceException e1) {
                throw new WSDLServiceException("Can't register wsdl file to persistent storage", e1);
            }

            /*
             * Convert common-wsdl object model to Dragon object model and
             * persist
             */
            result = this.createDragonObjectModel(description, storedWSDL, wsdlSha1);
        }
        return result;
    }

    @CheckArgumentsNotNull
    public String[] importServiceDefFile(InputSource wsdlSource, Map<String, InputSource> imports,
            String wsdlFileName) throws WSDLServiceException, TimeoutException {
        // You must create a number of forker equals to the number of time you
        // use the InputStream-1, to avoid premature end of file errors, cause
        // inputsource stream are consumed
        final InputStream originalInputStream = wsdlSource.getByteStream();
        final InputStreamForker isf1 = new InputStreamForker(originalInputStream);
        final InputStreamForker isf2 = new InputStreamForker(isf1.getInputStreamOne());

        String wsdlSha1;
        try {
            wsdlSha1 = XMLUtil.getXMLSHA1Sign(isf1.getInputStreamTwo(), true);
        } catch (Exception e2) {
            throw new WSDLServiceException("Can't get WSDL SHA1 signature", e2);
        }

        /*
         * Check if the given description isn't already registered
         */
        String[] result = null;
        if (!alreadyRegisteredDesc(wsdlSha1)) {

            /*
             * Parse wsdl with easywsdl
             */
            Description description = null;
            try {
                // Create import Map
                Map<URI, AbsItfDescription> importedWsdls = null;
                Map<URI, AbsItfSchema> importedSchemas = null;
                if (imports != null && imports.size() > 0) {
                    importedWsdls = new HashMap<URI, AbsItfDescription>();
                    importedSchemas = new HashMap<URI, AbsItfSchema>();
                    Set<String> keys = imports.keySet();
                    for (String key : keys) {
                        InputSource importSource = imports.get(key);
                        final InputStream originalImportInputStream = importSource.getByteStream();
                        final InputStreamForker isfImp1 = new InputStreamForker(
                                originalImportInputStream);
                        Document importDoc = XMLUtil.createDocument(isfImp1.getInputStreamOne());
                        importSource.setByteStream(isfImp1.getInputStreamTwo());
                        // TODO try to find a better URI for this type of
                        // document
                        importDoc.setDocumentURI(".");
                        // Read import
                        if (isWSDL(importDoc)) {
                            AbsItfDescription desc = this.wsdlReader.read(importSource);
                            importedWsdls.put(URI.create(key), desc);
                        } else if (isSchema(importDoc)) {
                            AbsItfSchema schema = this.schemaReader.read(importSource);
                            importedSchemas.put(URI.create(key), schema);
                        }

                    }
                }
                /*
                 * Parse wsdl with easywsdl
                 */
                wsdlSource.setByteStream(isf2.getInputStreamOne());
                EasyWSDLTimeout easyWSDLTimeout = new EasyWSDLTimeout(wsdlReader, wsdlSource,
                        importedWsdls, importedSchemas);
                WSDLImportReport wsdlImportReport = easyWSDLTimeout.execute(500000);
                description = wsdlImportReport.getResult();
                if (description == null) {
                    throw wsdlImportReport.getError();
                }
                // description = this.wsdlReader.read(wsdlSource, importedWsdls,
                // importedSchemas);
            } catch (WSDLException e) {
                throw new WSDLServiceException(
                        "Can't parse wsdl content. Check the wsdl validity.", e);
            } catch (SchemaException e) {
                throw new WSDLServiceException(
                        "Can't parse imported schema. Check the schema validity.", e);
            } catch (MalformedURLException e) {
                throw new WSDLServiceException(
                        "Can't parse wsdl content. Check the wsdl validity.", e);
            } catch (URISyntaxException e) {
                throw new WSDLServiceException(
                        "Can't parse wsdl content. Check the wsdl validity.", e);
            } catch (IllegalArgumentException e) {
                throw new WSDLServiceException(
                        "Can't parse wsdl content. Check the wsdl validity.", e);
            } catch (SAXException e) {
                throw new WSDLServiceException("Can't create document from InputSource.", e);
            } catch (IOException e) {
                throw new WSDLServiceException("Can't create document from InputSource.", e);
            }

            /*
             * Store WSDL file in persistent storage
             */
            SimpleFile storedWSDL = null;
            try {
                storedWSDL = this.metadataService.storeMetadata(ContentType.XML, InputStreamUtil
                        .getBytes(isf2.getInputStreamTwo()), wsdlFileName);
            } catch (final MetadataServiceException e1) {
                throw new WSDLServiceException("Can't register wsdl file to persistent storage", e1);
            } catch (IOException e) {
                throw new WSDLServiceException("Can't register wsdl file to persistent storage", e);
            }

            /*
             * Convert common-wsdl object model to Dragon object model and
             * persist
             */
            result = this.createDragonObjectModel(description, storedWSDL, wsdlSha1);
        }
        return result;
    }

    public void setServiceSpecificationUnifiedDAO(
            GenericUnifiedDAO<ServiceSpecification, String> serviceSpecificationUnifiedDAO) {
        this.serviceSpecificationUnifiedDAO = serviceSpecificationUnifiedDAO;
    }

    public void setUniversalUnifiedDAO(UniversalUnifiedDAO universalUnifiedDAO) {
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

}
