/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * OrganizationManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.organization;

import java.util.List;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.organization.OrganizationSearchProperties;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Identifier;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.springframework.transaction.annotation.Transactional;

/**
 * A manager of organization units. Provides methods to create, retrieve, delete
 * or update organization units and to manage categories, identifiers and post
 * related to organizations. Method arguments must be non null, unless the
 * contrary is explicitly specified.
 * 
 * @author ambarthe, ofabre - eBM WebSourcing
 * 
 */
@Transactional
public interface OrganizationManager {

    /**
     * Add the post matching the given postId to the organization unit matching
     * the given organizationId
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     * @param postId
     *            a {@link Post} id
     * @throws OrganizationException
     *             if at least one of the given id doesn't exist in database
     */
    public void addPost(String organizationId, String postId) throws OrganizationException;

    /**
     * Return the list of all posts that are linked to the given organization
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link PostTO}s that are linked to the
     *         given organization
     * @throws OrganizationException
     *             if an error occurs during posts retrieval
     */
    public List<PostTO> getPostsByOrganization(@NotNullParam String orgId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Add a category to the given organization. The added category is of the
     * type of the category denoted by the categoryId argument and the selected
     * value is denoted by the categoryValueId argument.
     * 
     * @param orgId
     *            a {@link OrganizationUnit} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValueId
     *            an identifier of a {@link CategoryValue} containing category
     *            info to add
     * @throws OrganizationException
     *             if at least one of the given id doesn't exist in database or
     *             if you try to add the same category twice
     */
    void addCategory(String orgId, String categoryId, String categoryValueId)
            throws OrganizationException;

    /**
     * Add a category to the given organization. The added category is of the
     * type of the category denoted by the categoryId argument and the selected
     * value is denoted by the categoryValue argument. The categoryDesc is used
     * to describe the created category
     * 
     * @param orgId
     *            a {@link OrganizationUnit} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValue
     *            a category value (UDDI: categoryValue)
     * @param categoryDesc
     *            a category description (UDDI: categoryName), could be null or
     *            empty
     * @throws OrganizationException
     *             if at least one of the given id doesn't exist in database or
     *             if you try to add the same category twice
     */
    void addCategory(@NotNullParam String orgId, @NotNullParam String categoryId,
            @NotNullParam String categoryValue, String categoryDesc) throws OrganizationException;

    /**
     * Add an identifier to the given organization. The added identifier is of
     * the type of the identifier system denoted by the identifierId argument
     * and the identifier value is denoted by the identifierValue argument.
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @param identifierId
     *            an {@link Identifier} system id
     * @param identifierValue
     *            an identifier value
     * @param identifierDesc
     *            an identifier description, could be null or empty
     * @throws OrganizationException
     *             if at least one of the given id doesn't exist in database
     */
    void addIdentifier(@NotNullParam String orgId, @NotNullParam String identifierId,
            @NotNullParam String identifierValue, String identifierDesc)
            throws OrganizationException;

    /**
     * Return all the categories related to the given organization unit.
     * Categories are keyed references as specified in the UDDI specification
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @return a {@link List} of categories ( {@link KeyedRefTO} )
     * @throws OrganizationException
     *             if not organization found for the given id
     */
    @Transactional(readOnly = true)
    List<KeyedRefTO> getCategoriesForOrg(String orgId) throws OrganizationException;

    /**
     * Remove a list of categories from the given organization unit.
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @param categoryIds
     *            a {@link List} of category ids
     * @throws OrganizationException
     *             if no organization found for the given id
     */
    void removeCategories(String orgId, List<String> categoryIds) throws OrganizationException;

    /**
     * Return all identifiers related to the given organization unit.
     * Identifiers are keyed references as specified in the UDDI specification
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @return a {@link List} of identifiers ( {@link KeyedRefTO} )
     * @throws OrganizationException
     *             if no organization found for the given id
     */
    @Transactional(readOnly = true)
    List<KeyedRefTO> getIdentifiersForOrg(String orgId) throws OrganizationException;

    /**
     * Remove a list of identifiers from the given organization unit.
     * 
     * @param orgId
     *            an {@link OrganizationUnit} id
     * @param categoryIds
     *            a {@link List} of identifier ids
     * @throws OrganizationException
     *             if no organization found for the given id
     */
    void removeIdentifiers(String orgId, List<String> categoryIds) throws OrganizationException;

    /**
     * Add a new organization unit in registry. The organization name and city
     * must be specified.
     * 
     * @param organizationUnitTO
     *            the {@link OrganizationUnitTO} handling data of the
     *            {@link OrganizationUnit} to create
     * @return the id of the added {@link OrganizationUnit}
     * @throws OrganizationException
     *             if you try to create an organization with name/city that
     *             already exists or an organization without name and city
     */
    public String createOrganization(OrganizationUnitTO organizationUnitTO)
            throws OrganizationException;

    /**
     * Return the list of all organization units in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link OrganizationUnitTO} in the
     *         registry, couldn't be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<OrganizationUnitTO> getAllOrganizations(RequestOptionsTO requestOptionsTO);

    /**
     * Return the list of all organization units in the registry excluding me
     * and my children
     * 
     * @return the {@link List} of all {@link OrganizationUnitTO} in the
     *         registry excluding me and my children
     * @throws OrganizationException
     *             if an error occurs during organization units retrieval
     */
    @Transactional(readOnly = true)
    public List<OrganizationUnitTO> getAllOrgsWithoutMeAndMyChildren(String initialOrg)
            throws OrganizationException;;

    /**
     * Retrieve the organization unit matching the given id
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     * @return the {@link OrganizationUnitTO} handling data of the
     *         {@link OrganizationUnit} that matches the given id
     * @throws OrganizationException
     *             if no organization found for the given id
     */
    @Transactional(readOnly = true)
    public OrganizationUnitTO getOrganization(String organizationId) throws OrganizationException;

    /**
     * Remove the organization unit matching the given id from the registry.
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     */
    public void removeOrganization(String organizationId);

    /**
     * Remove the post matching the given postId from the list of posts of the
     * organization unit matching the given organizationId. The post isn't
     * removed from the registry and can be re-affected to any organization
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     * @param postId
     *            a {@link Post} id
     * @throws OrganizationException
     *             if no organization found for the given id
     */
    public void removePost(String organizationId, String postId) throws OrganizationException;

    /**
     * Retrieve a list of organization units matching the given criteria for the
     * given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), mustn't be empty
     * @param searchedProperties
     *            the searched properties, mustn't be empty
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link OrganizationUnitTO} matching the given
     *         criteria for the given properties
     * @throws OrganizationException
     *             if you try to search orgs without specifying search criteria
     */
    @Transactional(readOnly = true)
    public List<OrganizationUnitTO> searchOrganization(@NotNullParam String searchCriteria,
            List<OrganizationSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws OrganizationException;

    /**
     * Update the organization unit in the registry.
     * 
     * @param organizationUnitTO
     *            the {@link OrganizationUnitTO} handling data of the
     *            {@link OrganizationUnit} to update
     * @return the updated organization unit ID
     * @throws OrganizationException
     *             if an error occurs during organization unit update :
     *             organization to update doesn't exist or have the same name
     *             and city as an other registered organization
     */
    public String updateOrganization(OrganizationUnitTO organizationUnitTO)
            throws OrganizationException;

}
