/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechnologyManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.technology;

import java.util.List;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.technology.EnvironmentFederationTO;
import org.ow2.dragon.api.to.technology.ExecEnvManagerSearchProperties;
import org.ow2.dragon.api.to.technology.ExecEnvSearchProperties;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentManagerTO;
import org.ow2.dragon.api.to.technology.ExecutionEnvironmentTO;
import org.ow2.dragon.api.to.technology.FederationSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorSearchProperties;
import org.ow2.dragon.api.to.technology.ProcessorTO;
import org.ow2.dragon.persistence.bo.technology.EnvironmentFederation;
import org.ow2.dragon.persistence.bo.technology.ExecutionEnvironment;
import org.ow2.dragon.persistence.bo.technology.ExecutionEnvironmentManager;
import org.ow2.dragon.persistence.bo.technology.Processor;
import org.springframework.transaction.annotation.Transactional;

/**
 * This API allows to connect to Execution Environments (Petals ESB...) and
 * retrieve information about these enviroments like hosted services and
 * endpoints. Method arguments must be non null, unless the contrary is
 * explicitly specified.
 * 
 * @author ofabre - eBM Websourcing
 * 
 */
@Transactional
public interface TechnologyManager {

    /**
     * Add a new runtime manager to the registry. Uses the provided address to
     * connect to the manager interface and retrieve manager properties
     * 
     * @param address
     *            a manager service interface address like
     *            'http://com.foo/ManagerService'. Must be a valid URL
     * @return the added manager ID
     * @throws TechnologyException
     *             if an error occurs during runtime manager creation or if the
     *             given URL isn't a valid URL
     */
    public String addRuntimeManager(String address) throws TechnologyException;

    /**
     * Retrieve the runtime manager matching the given ID
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @return the {@link ExecutionEnvironmentTO} (aka runtime manager) matching
     *         the given ID
     * @throws TechnologyException
     *             if no runtime manager found matching the given id
     */
    @Transactional(readOnly = true)
    public ExecutionEnvironmentManagerTO getRuntimeManager(String runtimeManagerId)
            throws TechnologyException;

    /**
     * Retrieve the environment federation matching the given ID
     * 
     * @param envFedId
     *            an environment federation ID
     * @return the {@link EnvironmentFederationTO} matching the given ID
     * @throws TechnologyException
     *             if no {@link EnvironmentFederation} found for the given id
     */
    @Transactional(readOnly = true)
    public EnvironmentFederationTO getEnvironmentFederation(String envFedId)
            throws TechnologyException;

    /**
     * Retrieve the list of execution environments managed by the given runtime
     * manager (aka execution environment manager)
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s managed by
     *         the given runtime manager (aka
     *         {@link ExecutionEnvironmentManagerTO}). Can't be null, could be
     *         empty
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getManagedExecEnv(@NotNullParam String runtimeManagerId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of execution environments that are parts of the given
     * environment federation
     * 
     * @param envFedId
     *            an environment federation ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s that are
     *         parts of the given {@link EnvironmentFederationTO}. Can't be
     *         null, could be empty
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getExecEnvsInFederation(@NotNullParam String envFedId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Remove the runtime manager matching the given ID from the registry
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @throws TechnologyException
     *             if no {@link ExecutionEnvironmentManager} can be found for
     *             the given id
     */
    public void removeRuntimeManager(String runtimeManagerId) throws TechnologyException;

    /**
     * Remove the environment federation matching the given ID from the registry
     * 
     * @param envFedId
     *            an environment federation ID
     * @throws TechnologyException
     *             if no {@link EnvironmentFederation} found matching the given
     *             id
     */
    public void removeEnvironmentFederation(String envFedId) throws TechnologyException;

    /**
     * Synchronize all execution environments managed by this execution runtime
     * manager. All linked endpoints, environment federations and processors
     * will also be synchronized. Synchronization consist in adding elements to
     * the registry
     * 
     * @param runtimeManagerId
     *            a runtime manager ID
     * @throws TechnologyException
     *             if no runtime manager found for the given id or if a database
     *             incoherence is detected
     */
    public void synchronizeManagedExecEnv(String runtimeManagerId) throws TechnologyException;

    /**
     * Retrieve the execution environment matching the given ID
     * 
     * @param execEnvId
     *            an execution environment ID
     * @return the {@link ExecutionEnvironmentTO} matching the given ID
     * @throws TechnologyException
     *             if no {@link ExecutionEnvironment} found matching the given
     *             id
     */
    @Transactional(readOnly = true)
    public ExecutionEnvironmentTO getExecutionEnvironment(String execEnvId)
            throws TechnologyException;

    /**
     * Retrieve the processor matching the given ID
     * 
     * @param processorId
     *            a processor ID
     * @return the {@link ProcessorTO} matching the given ID
     * @throws TechnologyException
     *             if no processor found matching the given id
     */
    @Transactional(readOnly = true)
    public ProcessorTO getProcessor(String processorId) throws TechnologyException;

    /**
     * Remove the execution environment matching the given ID from the registry
     * 
     * @param execEnvId
     *            an execution environment ID
     * @throws TechnologyException
     *             if no {@link ExecutionEnvironment} found matching the given
     *             id
     */
    public void removeExecutionEnvironment(String execEnvId) throws TechnologyException;

    /**
     * Remove the processor matching the given ID from the registry
     * 
     * @param processorId
     *            a processor ID
     * @throws TechnologyException
     *             if no {@link Processor} found matching the given id
     */
    public void removeProcessor(String processorId) throws TechnologyException;

    /**
     * Retrieve the list of endpoints that are hosted by the given execution
     * environment
     * 
     * @param execEnvId
     *            an execution environment ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of {@link EndpointTO}s that are hosted by the
     *         given {@link ExecutionEnvironmentTO}. Can't be null, could be
     *         empty
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEpsHostedOnExecEnv(@NotNullParam String execEnvId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of endpoints that are hosted by the given processor. It
     * means all endpoints hosted by execution environment hosted on the given
     * processor
     * 
     * @param processorId
     *            a processor ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of {@link EndpointTO}s that are hosted by the
     *         given {@link ProcessorTO}
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEpsHostedOnProcessor(@NotNullParam String processorId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of execution environments that are hosted by the given
     * processor.
     * 
     * @param processorId
     *            a processor ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of {@link ExecutionEnvironmentTO}s that are
     *         hosted by the given {@link ProcessorTO}. Can't be null, could be
     *         empty
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getExecEnvsHostedOnProcessor(
            @NotNullParam String processorId, RequestOptionsTO requestOptionsTO);

    /**
     * Return the list of all runtime managers (aka Execution Environment
     * Managers) in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return {@link List} of all runtime managers (aka
     *         {@link ExecutionEnvironmentManagerTO}) in the registry. Can't be
     *         null, could be empty
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentManagerTO> getAllExecutionEnvironmentManagers(
            RequestOptionsTO requestOptionsTO);

    /**
     * Return the list of all environment federations in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link EnvironmentFederationTO}s in the
     *         registry. Can't be null, could be empty
     */
    @Transactional(readOnly = true)
    public List<EnvironmentFederationTO> getAllEnvironmentFederations(
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all runtime managers (aka execution environment
     * managers) matching the given criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), musn't be empty
     * @param searchedProperties
     *            the searched properties, musn't be empty
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all runtime managers (aka
     *         {@link ExecutionEnvironmentManagerTO}s) matching the given
     *         criteria for the given properties. Can't be null, could be empty
     * @throws TechnologyException
     *             if you try to search without specifying criteria
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentManagerTO> searchExecutionEnvironmentManagers(
            @NotNullParam String searchCriteria,
            List<ExecEnvManagerSearchProperties> searchedProperties,
            RequestOptionsTO requestOptionsTO) throws TechnologyException;

    /**
     * Retrieve the list of all environment federations matching the given
     * criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), mustn't be empty
     * @param searchedProperties
     *            the searched properties, mustn't be empty
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link EnvironmentFederationTO}s matching
     *         the given criteria for the given properties. Can't be null, could
     *         be empty
     * @throws TechnologyException
     *             if you try to search without specifying criteria
     */
    @Transactional(readOnly = true)
    public List<EnvironmentFederationTO> searchEnvironmentFederations(
            @NotNullParam String searchCriteria,
            List<FederationSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException;

    /**
     * Return the list of all execution environments in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link ExecutionEnvironmentTO}s in the
     *         registry. Can't be null, could be empty
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> getAllExecutionEnvironments(
            RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all execution environments matching the given
     * criteria for the given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), musn't be empty
     * @param searchedProperties
     *            the searched properties, musn't be empty
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link ExecutionEnvironmentTO}s matching
     *         the given criteria for the given properties. Can't be null, could
     *         be empty
     * @throws TechnologyException
     *             if you try to search without specifying criteria
     */
    @Transactional(readOnly = true)
    public List<ExecutionEnvironmentTO> searchExecutionEnvironments(
            @NotNullParam String searchCriteria, List<ExecEnvSearchProperties> searchedProperties,
            RequestOptionsTO requestOptionsTO) throws TechnologyException;

    /**
     * Return the list of all processors in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link ProcessorTO}s in the registry.
     *         Can't be null, could be empty
     */
    @Transactional(readOnly = true)
    public List<ProcessorTO> getAllProcessors(RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the list of all processors matching the given criteria for the
     * given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), musn't be empty
     * @param searchedProperties
     *            the searched properties, musn't be empty
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null
     * @return the {@link List} of all {@link ProcessorTO}s matching the given
     *         criteria for the given properties. Can't be null, could be empty
     * @throws TechnologyException
     *             if you try to search without specifying criteria
     */
    @Transactional(readOnly = true)
    public List<ProcessorTO> searchProcessors(@NotNullParam String searchCriteria,
            List<ProcessorSearchProperties> searchedProperties, RequestOptionsTO requestOptionsTO)
            throws TechnologyException;

}
