/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDIIdentifierGenerator.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.persistence.util;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.Logger;
import org.hibernate.HibernateException;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.exception.JDBCExceptionHelper;
import org.hibernate.id.IdentifierGenerator;
import org.ow2.dragon.util.RegistryConfig;
import org.ow2.dragon.util.StringHelper;

/**
 * The UDDI key generator. Generates a key like this:
 * 
 * uddiScheme : RootDomain : UUID
 * 
 * @author ofabre
 * 
 */
public class UDDIIdentifierGenerator implements IdentifierGenerator {

    public static final String UDDI_V3_SCHEME = "uddi";

    public static final String UDDI_V2_SCHEME = "uuid";

    private static final String PARTITION_SEPARATOR = ":";

    private static final String ROOT_DOMAIN = "root.domain";

    private Logger log = Logger.getLogger(UDDIIdentifierGenerator.class);

    private static DualHashBidiMap uddiV2toV3Key;

    static {
        uddiV2toV3Key = new DualHashBidiMap();

        // Base tmodels
        uddiV2toV3Key.put("uuid:C1ACF26D-9672-4404-9D70-39B756E62AB4",
                "uddi:uddi.org:categorization:types");
        uddiV2toV3Key.put("uuid:A035A07C-F362-44dd-8F95-E2B134BF43B4",
                "uddi:uddi.org:categorization:general_keywords");
        uddiV2toV3Key.put("uuid:327A56F0-3299-4461-BC23-5CD513E95C55",
                "uddi:uddi.org:categorization:nodes");
        uddiV2toV3Key.put("uuid:807A2C6A-EE22-470D-ADC7-E0424A337C03",
                "uddi:uddi.org:relationships");
        uddiV2toV3Key.put("uuid:4064C064-6D14-4F35-8953-9652106476A9",
                "uddi:uddi.org:categorization:owningbusiness");
        uddiV2toV3Key.put("uuid:E59AE320-77A5-11D5-B898-0004AC49CC1E",
                "uddi:uddi.org:identifier:isreplacedby");
        uddiV2toV3Key.put("uuid:25b22e3e-3dfa-3024-b02a-3438b9050b59",
                "uddi:uddi.org:categorization:validatedby");
        uddiV2toV3Key.put("uuid:5678dd4f-f95d-35f9-9ea6-f79a7dd64656",
                "uddi:uddi.org:categorization:derivedfrom");
        uddiV2toV3Key.put("uuid:916b87bf-0756-3919-8eae-97dfa325e5a4",
                "uddi:uddi.org:categorization:entitykeyvalues");

        // Protocol tmodels
        uddiV2toV3Key.put("uuid:c8aea832-3faf-33c6-b32a-bbfd1b926294",
                "uddi:uddi.org:protocol:serverauthenticatedssl3");
        uddiV2toV3Key.put("uuid:9555b5b6-55d4-3b0e-bb17-e084fed4e33f",
                "uddi:uddi.org:protocol:mutualauthenticatedssl3");
        uddiV2toV3Key.put("uuid:68DE9E80-AD09-469D-8A37-088422BFBC36",
                "uddi:uddi.org:transport:http");
        uddiV2toV3Key.put("uuid:93335D49-3EFB-48A0-ACEA-EA102B60DDC6",
                "uddi:uddi.org:transport:smtp");
        uddiV2toV3Key.put("uuid:5FCF5CD0-629A-4C50-8B16-F94E9CF2A674",
                "uddi:uddi.org:transport:ftp");
        uddiV2toV3Key.put("uuid:1A2B00BE-6E2C-42F5-875B-56F32686E0E7",
                "uddi:uddi.org:transport:fax");
        uddiV2toV3Key.put("uuid:38E12427-5536-4260-A6F9-B5B530E63A07",
                "uddi:uddi.org:transport:telephone");

        // Categorization tmodel
        uddiV2toV3Key.put("uuid:c0b9fe13-179f-413d-8a5b-5004db8e5bb2",
                "uddi:uddi.org:ubr:categorization:naics:1997");
        uddiV2toV3Key.put("uuid:1ff729f2-1948-46cf-b660-31ec107f1663",
                "uddi:uddi.org:ubr:categorization:naics:2002");
        uddiV2toV3Key.put("uuid:cd153257-086a-4237-b336-6bdcbdcc6634",
                "uddi:cd153257-086a-4237-b336-6bdcbdcc6634");
        uddiV2toV3Key.put("uuid:4614c240-b483-11d7-8be8-000629dc0a53",
                "uddi:uddi.org:ubr:categorization:unspsc");
        uddiV2toV3Key.put("uuid:4e49a8d6-d5a2-4fc2-93a0-0411d8d19e88",
                "uddi:uddi.org:ubr:categorization:iso3166");

        // Identitifer tmodels
        uddiV2toV3Key.put("uuid:8609c81e-ee1f-4d5a-b202-3eb13ad01823",
                "uddi:uddi.org:ubr:identifier:dnb.com:d-u-n-s");
        uddiV2toV3Key.put("uuid:B1B1BAF5-2329-43E6-AE13-BA8E97195039",
                "uddi:uddi.org:ubr:identifier:thomasregister.com:supplierid");
        uddiV2toV3Key.put("uuid:f1b347da-6cbb-3a10-93e7-7cd4328b88d3",
                "uddi:uddi.org:ubr:identifier:iso6523icd");

        // UDDI API tmodels
        uddiV2toV3Key.put("uuid:01b9bbff-a8f5-3735-9a5e-5ea5ade7daaf", "uddi:uddi.org:v3_inquiry");
        uddiV2toV3Key.put("uuid:72ade754-c6cc-315b-b014-7c94791fe15c",
                "uddi:uddi.org:v3_publication");

        // WSDL mapping to model
        uddiV2toV3Key.put("uuid:6e090afa-33e5-36eb-81b7-1ca18373f457", "uddi:uddi.org:wsdl:types");
        uddiV2toV3Key.put("uuid:d01987d1-ab2e-3013-9be2-2a66eb99d824",
                "uddi:uddi.org:xml:namespace");
        uddiV2toV3Key.put("uuid:2ec65201-9109-3919-9bec-c9dbefcaccf6",
                "uddi:uddi.org:xml:localname");
        uddiV2toV3Key.put("uuid:082b0851-25d8-303c-b332-f24a6d53e38e",
                "uddi:uddi.org:wsdl:porttypereference");
        uddiV2toV3Key.put("uuid:aa254698-93de-3870-8df3-a5c075d64a0e",
                "uddi:uddi.org:protocol:soap");
        uddiV2toV3Key.put("uuid:6e10b91b-babc-3442-b8fc-5a3c8fde0794",
                "uddi:uddi.org:protocol:http");
        uddiV2toV3Key.put("uuid:4dc74177-7806-34d9-aecd-33c57dc3a865",
                "uddi:uddi.org:wsdl:categorization:protocol");
        uddiV2toV3Key.put("uuid:e5c43936-86e4-37bf-8196-1d04b35c0099",
                "uddi:uddi.org:wsdl:categorization:transport");
        uddiV2toV3Key
                .put("uuid:ad61de98-4db8-31b2-a299-a2373dc97212", "uddi:uddi.org:wsdl:address");

    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.hibernate.id.IdentifierGenerator#generate(org.hibernate.engine.
     * SessionImplementor, java.lang.Object)
     */
    public Serializable generate(SessionImplementor session, Object object)
            throws HibernateException {

        Serializable id = session.getEntityPersister(null, object)
        // TODO: cache the persister, this shows up in yourkit
                .getIdentifier(object, session.getEntityMode());

        // If the id isn't specified by the application, generate it
        if (id == null || (id instanceof String && StringHelper.isNullOrEmpty((String) id))) {
            id = getUddiV3KeyPrefix() + getUUID();
        }

        // Based on JAVA UUID
        return id;

        // Based on SQL GUID
        // return UDDI_SCHEME + PARTITION_SEPARATOR + rootDomain +
        // PARTITION_SEPARATOR
        // + getGUID(session);
    }

    private synchronized String getUUID() {
        return UUID.randomUUID().toString();
    }

    private String getGUID(SessionImplementor session) {
        final String sql = session.getFactory().getDialect().getSelectGUIDString();
        try {
            PreparedStatement st = session.getBatcher().prepareSelectStatement(sql);
            try {
                ResultSet rs = st.executeQuery();
                final String result;
                try {
                    rs.next();
                    result = rs.getString(1);
                } finally {
                    rs.close();
                }
                log.debug("GUID identifier generated: " + result);
                return result;
            } finally {
                session.getBatcher().closeStatement(st);
            }
        } catch (SQLException sqle) {
            throw JDBCExceptionHelper.convert(session.getFactory().getSQLExceptionConverter(),
                    sqle, "could not retrieve GUID", sql);
        }
    }

    public static String toUddiV2Id(String uddiV3Id) {
        String v2Key = null;
        if (!StringHelper.isNullOrEmpty(uddiV3Id)) {
            uddiV3Id = uddiV3Id.toLowerCase();
            // Try to find it in conversion map
            v2Key = (String) uddiV2toV3Key.getKey(uddiV3Id);
            if (v2Key == null) {
                // Uses a simple algorithm to convert it to uddi v3 key
                if (!uddiV3Id.startsWith(getUddiV3KeyPrefix())) {
                    throw new HibernateException("Trying to use an invalid uddi v3 key ('"
                            + uddiV3Id + "'). Valid v3 key for this registry are: '"
                            + getUddiV3KeyPrefix() + "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'");
                } else {
                    v2Key = getUddiV2KeyPrefix()
                            + uddiV3Id.substring(getUddiV3KeyPrefix().length());
                }
            }
        }
        return v2Key;
    }

    private static String getUddiV3KeyPrefix() {
        String rootDomain = "";
        try {
            rootDomain = RegistryConfig.getConfiguration().getString(ROOT_DOMAIN);
        } catch (ConfigurationException ce) {
            throw new HibernateException(
                    "Can't retrieve root.domain property from registry config file", ce);
        }
        return UDDI_V3_SCHEME + PARTITION_SEPARATOR + rootDomain + PARTITION_SEPARATOR;
    }

    private static String getUddiV2KeyPrefix() {
        return UDDI_V2_SCHEME + PARTITION_SEPARATOR;
    }

    public static String toUddiV3Id(String uddiV2Id) {
        String v3Key = null;
        if (!StringHelper.isNullOrEmpty(uddiV2Id)) {
            uddiV2Id = uddiV2Id.toLowerCase();
            // Try to find it in conversion map
            v3Key = (String) uddiV2toV3Key.get(uddiV2Id);
            if (v3Key == null) {
                // Uses a simple algorithm to convert it to uddi v3 key
                if (!uddiV2Id.startsWith(getUddiV2KeyPrefix())) {
                    throw new HibernateException("Trying to use an invalid uddi v2 key ('"
                            + uddiV2Id + "'). Valid v2 key are: '" + getUddiV2KeyPrefix()
                            + "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'");
                } else {
                    v3Key = getUddiV3KeyPrefix()
                            + uddiV2Id.substring(getUddiV2KeyPrefix().length());
                }
            }
        }

        return v3Key;
    }

    /**
     * A valid id is an Id starting with "uddi:'root domain'" where root domain
     * is the "root.domain" property defined in dragon.properties file or an id
     * registered in the uddiV2toV3 map
     * 
     * @param uddiV3Id
     *            a v3 id
     * @return true if it's a valid Id false otherwise
     */
    public static boolean isUddiV3ValidId(String uddiV3Id) {
        boolean result = uddiV2toV3Key.containsValue(uddiV3Id);
        if (!result) {
            result = uddiV3Id.startsWith(getUddiV3KeyPrefix());
        }
        return result;
    }

}
