/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PersonManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.organization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.service.organization.PersonManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.PersonSearchProperties;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.organization.PersonDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.StringHelper;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class PersonManagerImpl implements PersonManager {

    private TransfertObjectAssembler transfertObjectAssembler;

    private GenericUnifiedDAO<Person, String> personUnifiedDAO;

    private PersonDAO personDAO;

    @CheckAllArgumentsNotNull
    public String createPerson(final PersonTO personTO) throws OrganizationException {
        // Validate person before creation
        this.validatePersonBeforeCreation(personTO);

        // create organization unit bo from to
        final Person personBO = new Person();
        this.transfertObjectAssembler.toPersonBO(personTO, personBO);
        // Persist new Organization Unit
        return this.personUnifiedDAO.save(personBO).getId();
    }

    private void validatePersonBeforeCreation(PersonTO personTO) throws OrganizationException {
        // Check if the name and city are null
        if (StringHelper.isNullOrEmpty(personTO.getFirstname())
                || StringHelper.isNullOrEmpty(personTO.getLastname())
                || StringHelper.isNullOrEmpty(personTO.getEmail())) {
            throw new OrganizationException(
                    "Person firstname, lastname and email must be specified.");
        }

        // Check if a person with the same name exists
        Person person = personDAO.getPersonByFirstNameLastNameEmail(personTO.getFirstname(),
                personTO.getLastname(), personTO.getEmail());
        if (person != null) {
            throw new OrganizationException(
                    "A person with the same firstName, lastName and email address already exists.");
        }
    }

    private String[] createSearchProperties(List<PersonSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(PersonSearchProperties.NAME)) {
                propertiesList.add("firstName");
                propertiesList.add("lastName");
                propertiesList.add("middleName");
            }
            if (searchedProperties.contains(PersonSearchProperties.PARENT_ORG)) {
                propertiesList.add("parentOrganization.names.name");
            }
            if (searchedProperties.contains(PersonSearchProperties.POST)) {
                propertiesList.add("post.name");
                propertiesList.add("post.nature");
                propertiesList.add("post.description");
            }
        } else {
            // search on all properties
            propertiesList.add("firstName");
            propertiesList.add("lastName");
            propertiesList.add("middleName");
            propertiesList.add("parentOrganization.names.name");
            propertiesList.add("post.name");
            propertiesList.add("post.nature");
            propertiesList.add("post.description");

        }
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PersonManager#getAllPersons
     * (org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<PersonTO> getAllPersons(final RequestOptionsTO requestOptionsTO) {
        final List<PersonTO> result = new ArrayList<PersonTO>();
        final List<Person> persons = this.personUnifiedDAO.getAll(this.transfertObjectAssembler
                .toPersonRequestOptions(requestOptionsTO));
        this.toPersonsTO(result, persons);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PersonManager#getPerson
     * (java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public PersonTO getPerson(final String personId) throws OrganizationException {
        // retrieve person bo
        final Person personBO = this.personUnifiedDAO.get(personId);
        if (personBO == null) {
            throw new OrganizationException("No person found for the given id: " + personId);
        }
        // create person to from bo
        final PersonTO personTO = this.toPersonTO(personBO);
        return personTO;
    }

    public GenericUnifiedDAO<Person, String> getPersonUnifiedDAO() {
        return this.personUnifiedDAO;
    }

    public List<PersonTO> getPersonsByOrgAndPost(final String organizationId, final String postId,
            final RequestOptionsTO optionsTO) {
        final List<PersonTO> result = new ArrayList<PersonTO>();

        // Split searchCriteria
        final String[] criteria = { organizationId, postId };

        // Create search properties
        final String[] properties = { "parentOrganization.id", "post.id" };

        // Search for bo
        final List<Person> persons = this.personUnifiedDAO.searchEquals(criteria, properties,
                this.transfertObjectAssembler.toPersonRequestOptions(optionsTO));

        this.toPersonsTO(result, persons);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.PersonManager#
     * getPersonsByOrganization(java.lang.String,
     * org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<PersonTO> getPersonsByOrganization(final String organizationId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        final List<PersonTO> result = new ArrayList<PersonTO>();

        // Split searchCriteria
        final String[] criteria = { organizationId };

        // Create search properties
        final String[] properties = { "parentOrganization.id" };

        // Search for bo
        final List<Person> persons = this.personUnifiedDAO.searchEquals(criteria, properties,
                this.transfertObjectAssembler.toPersonRequestOptions(requestOptionsTO));

        this.toPersonsTO(result, persons);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.PersonManager#
     * getPersonsByPost(java.lang.String,
     * org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<PersonTO> getPersonsByPost(final String postId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        final List<PersonTO> result = new ArrayList<PersonTO>();

        // Split searchCriteria
        final String[] criteria = { postId };

        // Create search properties
        final String[] properties = { "post.id" };

        // Search for bo
        final List<Person> persons = this.personUnifiedDAO.searchEquals(criteria, properties,
                this.transfertObjectAssembler.toPersonRequestOptions(requestOptionsTO));

        this.toPersonsTO(result, persons);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PersonManager#removePerson
     * (java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public void removePerson(final String personId) {
        this.personUnifiedDAO.remove(personId);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PersonManager#searchPersons
     * (java.lang.String, boolean, boolean, boolean,
     * org.ow2.dragon.ui.model.to.RequestOptions)
     */
    @CheckArgumentsNotNull
    public List<PersonTO> searchPersons(final String searchCriteria,
            List<PersonSearchProperties> searchedProperties, final RequestOptionsTO options)
            throws OrganizationException {
        final List<PersonTO> result = new ArrayList<PersonTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<Person> persons;
        try {
            persons = this.personUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toPersonRequestOptions(options));
        } catch (DAOLayerException e) {
            throw new OrganizationException(
                    "You must specified non empty search criteria and properties.", e);
        }

        this.toPersonsTO(result, persons);
        return result;
    }

    public void setPersonUnifiedDAO(final GenericUnifiedDAO<Person, String> personUnifiedDAO) {
        this.personUnifiedDAO = personUnifiedDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    private void toPersonsTO(final List<PersonTO> result, final Collection<Person> persons) {
        if ((persons != null) && !persons.isEmpty()) {
            for (final Person person : persons) {
                result.add(this.toPersonTO(person));
            }
        }
    }

    private PersonTO toPersonTO(final Person person) {
        return this.transfertObjectAssembler.toPersonTO(person);
    }

    @CheckAllArgumentsNotNull
    public String updatePerson(final PersonTO personTO) throws OrganizationException {
        // Retrieve person
        String persId = personTO.getId();
        if (persId == null) {
            throw new NullPointerException("Person to update must have an id.");
        }
        final Person person = this.personUnifiedDAO.get(persId);
        if (person == null) {
            throw new OrganizationException(
                    "Your are trying to update a non existing Person with id: " + persId);
        }

        // Validate person before update
        this.validatePersonBeforeUpdate(personTO);

        // update fields
        this.transfertObjectAssembler.toPersonBO(personTO, person);
        this.personUnifiedDAO.save(person);

        return person.getId();
    }

    private void validatePersonBeforeUpdate(PersonTO personTO) throws OrganizationException {
        // Check if the name and city are null
        if (StringHelper.isNullOrEmpty(personTO.getFirstname())
                || StringHelper.isNullOrEmpty(personTO.getLastname())
                || StringHelper.isNullOrEmpty(personTO.getEmail())) {
            throw new OrganizationException(
                    "Person firstname, lastname and email must be specified.");
        }

        // Check if a person with the same name exists
        Person person = personDAO.getPersonByFirstNameLastNameEmail(personTO.getFirstname(),
                personTO.getLastname(), personTO.getEmail());
        if (person != null && !person.getId().equals(personTO.getId())) {
            throw new OrganizationException(
                    "A person with the same firstName, lastName and email address already exists.");
        }

    }

    public void setPersonDAO(PersonDAO personDAO) {
        this.personDAO = personDAO;
    }
}
