/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PostManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.organization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.service.organization.PostManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.PostSearchProperties;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.organization.OrganizationUnitDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.StringHelper;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class PostManagerImpl implements PostManager {

    private TransfertObjectAssembler transfertObjectAssembler;

    private GenericUnifiedDAO<Post, String> postDAO;

    private OrganizationUnitDAO organizationUnitDAO;

    private GenericUnifiedDAO<Person, String> personUnifiedDAO;

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PostManager#createPost
     * (org.ow2.dragon.ui.model.to.organization.PostTO)
     */
    @CheckAllArgumentsNotNull
    public String createPost(final PostTO postTO) throws OrganizationException {
        // Validate created post
        this.validatePostBeforeCreation(postTO);

        // create person bo from to
        final Post postBO = new Post();
        this.transfertObjectAssembler.toPostBO(postTO, postBO);
        // Persist new Post
        return this.postDAO.save(postBO).getId();
    }

    private void validatePostBeforeCreation(PostTO postTO) throws OrganizationException {
        // check if the postn ame isn't specified
        if (postTO.getName() == null) {
            throw new OrganizationException("Post name must be specified");
        }

        // Check if a post with the same name exists
        List<Post> posts = postDAO.searchEquals(new String[] { postTO.getName() },
                new String[] { "name" }, null);
        if (posts != null && !posts.isEmpty()) {
            throw new OrganizationException("A post with the same name already exists");
        }
    }

    private void validatePostBeforeUpdate(PostTO postTO) throws OrganizationException {
        // Check if the name isn't null or empty
        if (StringHelper.isNullOrEmpty(postTO.getName())) {
            throw new OrganizationException(
                    "You're trying to update a post with a null or empty name");
        }

        // Check if a post with the same name exists
        List<Post> posts = postDAO.searchEquals(new String[] { postTO.getName() },
                new String[] { "name" }, null);
        if (posts != null && !posts.isEmpty() && !posts.get(0).getId().equals(postTO.getId())) {
            throw new OrganizationException("A post with the same name already exists");
        }
    }

    @CheckArgumentsNotNull
    public List<PostTO> searchPost(final String searchCriteria,
            List<PostSearchProperties> searchedProperties, final RequestOptionsTO requestOptionsTO)
            throws OrganizationException {
        final List<PostTO> result = new ArrayList<PostTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<Post> posts;
        try {
            posts = this.postDAO.searchORMResult(criteria, properties, transfertObjectAssembler
                    .toPostRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new OrganizationException(
                    "You must specified non empty search criteria and properties.", e);
        }

        // Create result array
        this.toPostsTO(result, posts);

        // Is post in use ?
        for (PostTO post : result) {
            post.setPostInUse(isPostInUse(post.getId()));
        }

        return result;
    }

    private String[] createSearchProperties(List<PostSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(PostSearchProperties.NAME)) {
                propertiesList.add("name");
            }
            if (searchedProperties.contains(PostSearchProperties.NATURE)) {
                propertiesList.add("nature");
            }
            if (searchedProperties.contains(PostSearchProperties.DESCRIPTION)) {
                propertiesList.add("description");
            }
        } else {
            // Search on all properties
            propertiesList.add("name");
            propertiesList.add("nature");
            propertiesList.add("description");
        }
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PostManager#getAllPosts
     * (org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<PostTO> getAllPosts(final RequestOptionsTO requestOptionsTO) {
        final List<PostTO> result = new ArrayList<PostTO>();
        final List<Post> posts = this.postDAO.getAll(this.transfertObjectAssembler
                .toPostRequestOptions(requestOptionsTO));
        this.toPostsTO(result, posts);
        // Is post in use ?
        for (PostTO post : result) {
            try {
                post.setPostInUse(isPostInUse(post.getId()));
            } catch (OrganizationException e) {
                // Can't be thrown
            }
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PostManager#getPost(java
     * .lang.String)
     */
    @CheckAllArgumentsNotNull
    public PostTO getPost(final String postId) throws OrganizationException {
        // retrieve person bo
        final Post postBO = this.postDAO.get(postId);
        if (postBO == null) {
            throw new OrganizationException("No post found for the given id: " + postId);
        }
        // create person to from bo
        final PostTO postTO = this.toPostTO(postBO);
        return postTO;
    }

    public List<PostTO> getPostsNotLinkedToOrganization(final String orgId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        final List<PostTO> result = new ArrayList<PostTO>();
        final OrganizationUnit org = this.organizationUnitDAO.get(orgId);
        Set<Post> postsInOrg = new HashSet<Post>();
        if (org != null) {
            postsInOrg = org.getPosts();
        } else {
            throw new OrganizationException("No organization with id: " + orgId);
        }
        final List<Post> allPost = this.postDAO.getAll(this.transfertObjectAssembler
                .toPostRequestOptions(requestOptionsTO));
        if (allPost != null) {
            if (!postsInOrg.isEmpty()) {
                allPost.removeAll(postsInOrg);
            }
            for (final Post post : allPost) {
                result.add(this.toPostTO(post));
            }
        }
        return result;
    }

    public List<PostTO> getPostsNotLinkedToPersons(final RequestOptionsTO requestOptionsTO)
            throws OrganizationException {
        final List<PostTO> result = new ArrayList<PostTO>();
        final List<Post> allPost = this.postDAO.getAll(this.transfertObjectAssembler
                .toPostRequestOptions(requestOptionsTO));
        if (allPost != null) {
            // TODO dirty !!! find a better way to do it
            for (Post post : allPost) {
                if (!isPostInUse(post.getId())) {
                    result.add(transfertObjectAssembler.toPostTO(post));
                }
            }
        }
        return result;
    }

    private boolean isPostInUse(String postId) throws OrganizationException {
        Post post = postDAO.get(postId);
        if (post == null) {
            throw new OrganizationException("No post found for the given id: " + postId);
        }
        List<Person> persons = personUnifiedDAO.searchEquals(new String[] { post.getName() },
                new String[] { "post.name" }, null);
        List<OrganizationUnit> orgs = organizationUnitDAO.searchEquals(new String[] { post
                .getName() }, new String[] { "posts.name" }, null);
        return (persons != null && !persons.isEmpty()) || (orgs != null && !orgs.isEmpty());
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PostManager#removePost
     * (java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public void removePost(final String postId) throws OrganizationException {
        if (!isPostInUse(postId)) {
            this.postDAO.remove(postId);
        }
    }

    public void setOrganizationUnitDAO(final OrganizationUnitDAO organizationUnitDAO) {
        this.organizationUnitDAO = organizationUnitDAO;
    }

    public void setPostDAO(final GenericUnifiedDAO<Post, String> postDAO) {
        this.postDAO = postDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    private void toPostsTO(final List<PostTO> result, final Collection<Post> posts) {
        if ((posts != null) && !posts.isEmpty()) {
            for (final Post post : posts) {
                result.add(this.toPostTO(post));
            }
        }
    }

    private PostTO toPostTO(final Post post) {
        return this.transfertObjectAssembler.toPostTO(post);
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.organization.PostManager#updatePost
     * (org.ow2.dragon.ui.model.to.organization.PostTO)
     */
    @CheckAllArgumentsNotNull
    public String updatePost(final PostTO postTO) throws OrganizationException {
        // Retrieve post
        if (postTO.getId() == null) {
            throw new NullPointerException("Post id must be specified");
        }
        final Post post = this.postDAO.get(postTO.getId());
        if (post == null) {
            throw new OrganizationException(
                    "Your are trying to update a non existing Person with id: " + postTO.getId());
        }

        // Validate post
        this.validatePostBeforeUpdate(postTO);

        // update fields
        this.transfertObjectAssembler.toPostBO(postTO, post);
        this.postDAO.save(post);

        return post.getId();
    }

    public void setPersonUnifiedDAO(GenericUnifiedDAO<Person, String> personUnifiedDAO) {
        this.personUnifiedDAO = personUnifiedDAO;
    }
}
