/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EntityLinkManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.common;

import java.util.List;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.persistence.bo.common.Link;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
@Transactional
public interface EntityLinkManager {

    /**
     * Create a link between two linked entities (like person, organization,
     * technical service, endpoint, etc.)
     * 
     * @param fromId
     *            identifier of the "from end" of the link
     * @param toId
     *            identifier of the "to end" of the link
     * @param type
     *            the link type
     * @return the link identifier
     * @throws CommonException
     *             if one of the linked entity doesn't exist
     */
    public String createLink(final String fromId, final String toId, final String type)
            throws CommonException;

    /**
     * Remove a link from the registry
     * 
     * @param linkId
     *            a link identifier
     */
    public void removeLink(final String linkId);

    /**
     * Retrieve all links with the specified "to end"
     * 
     * @param toId
     *            the "to end" identifier
     * @param requestOptions
     *            sort, pagination and case sensitive parameters. Could be null.
     * @return all links with the specified "to end"
     */
    @Transactional(readOnly = true)
    public List<Link> getAllLinksForTo(@NotNullParam String toId, RequestOptionsTO requestOptions);

    /**
     * Retrieve all links with the specified "from end"
     * 
     * @param fromId
     *            the to end identifier
     * @param requestOptions
     *            sort, pagination and case sensitive parameters. Could be null.
     * @return all links with the specified "from end"
     */
    @Transactional(readOnly = true)
    public List<Link> getAllLinksForFrom(@NotNullParam String fromId,
            RequestOptionsTO requestOptions);

}
