/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EndpointManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.deployment;

import java.net.URI;
import java.util.List;
import java.util.concurrent.TimeoutException;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.springframework.transaction.annotation.Transactional;
import org.w3c.dom.Document;

/**
 * A manager of endpoints. Provides methods to create, retrieve, delete or
 * update endpoints. Method arguments must be non null, unless the contrary is
 * explicitly specified.
 * 
 * @author ambarthe, ofabre - eBM WebSourcing
 * 
 */
@Transactional
public interface EndpointManager {

    /**
     * Return the list of all endpoints in the registry.
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters. Could be null.
     * @return the {@link List} of all {@link EndpointTO} in the registry, not
     *         null, can be empty
     * @throws DeploymentException
     *             if an error occurs during endpoints retrieval
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getAllEndpoints(RequestOptionsTO requestOptionsTO)
            throws DeploymentException;

    /**
     * Retrieve an endpoint matching the given ID
     * 
     * @param endpointId
     *            an endpoint ID
     * @param user
     *            a user login, could be null
     * @return an {@link EndpointTO} matching the given ID, can be null
     * @throws DeploymentException
     *             if no {@link Endpoint} found matching the given id
     */
    @Transactional(readOnly = true)
    public EndpointTO getEndpoint(@NotNullParam String endpointId, String user)
            throws DeploymentException;

    /**
     * Return the list of all endpoints linked to the binding matching the given
     * ID. Not implemented yet
     * 
     * @param bindingId
     *            a binding ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link EndpointTO} linked to the binding
     *         matching the given ID, not null, can be empty
     * @throws DeploymentException
     *             if an error occurs during endpoints retrieval
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEndpointsByBinding(@NotNullParam String bindingId,
            RequestOptionsTO requestOptionsTO) throws DeploymentException;

    /**
     * Return the list of all endpoints linked to the technical service matching
     * the given ID.
     * 
     * @param techServiceId
     *            a tech service ID
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link EndpointTO} linked to the tech
     *         service matching the given ID, not null, can be empty
     */
    @Transactional(readOnly = true)
    public List<EndpointTO> getEndpointsByTechService(@NotNullParam String techServiceId,
            RequestOptionsTO requestOptionsTO);

    /**
     * Add a new endpoint to the registry linked to the specified service
     * 
     * @param epName
     *            the endpoint name
     * @param epAddress
     *            the enpoint network address. Could be null or empty
     * @param servId
     *            the parent service id
     * @return id of the created endpoint
     * @throws DeploymentException
     *             if an endpoint already exist with the given name linked to
     *             the given service or the ep name is empty or the service id
     *             is empty or the given service doesn't exist into the registry
     */
    String createEndpoint(@NotNullParam final String epName, final String epAddress,
            @NotNullParam final String servId) throws DeploymentException;

    /**
     * Remove the endpoint matching the given ID from the registry
     * 
     * @param endpointId
     *            an endpoint ID
     * @throws DeploymentException
     *             if an error occurs during endpoint removal
     */
    public void removeEndpoint(String endpointId) throws DeploymentException;

    /**
     * Update the endpoint in the registry.
     * 
     * @param endpointTO
     *            {@link EndpointTO} to update
     * @return the ID of the updated endpoint
     * @throws DeploymentException
     *             if no {@link Endpoint} found for the given techServ id to
     *             update or if an endpoint already exist with the given name
     *             linked to the given service or the ep name is null or empty
     *             or the service name is null or empty or the given service
     *             doesn't exist into the registry
     */
    public String updateEndpoint(EndpointTO endpointTO) throws DeploymentException;

    /**
     * Return the list of categories related to the endpoint matching the given
     * id
     * 
     * @param endpointId
     *            an {@link Endpoint} identifier
     * @return the list of categories as {@link KeyedRefTO} related to the
     *         endpoint matching the given id
     * @throws DeploymentException
     *             if no {@link Endpoint} found for the given id
     */
    List<KeyedRefTO> getCategoriesForEndpoint(String endpointId) throws DeploymentException;

    /**
     * Add a category to the given endpoint. The added category is of the type
     * of the category denoted by the categoryId argument and the selected value
     * is denoted by the categoryValueId argument.
     * 
     * @param endpointId
     *            a {@link Endpoint} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValueId
     *            an identifier of a {@link CategoryValue} containing category
     *            info to add.
     * @throws DeploymentException
     *             if at least one of the given id doesn't exist in database
     */
    void addCategory(String endpointId, String categoryId, String categoryValueId)
            throws DeploymentException;

    /**
     * Add a category to the given endpoint. The added category is of the type
     * of the category denoted by the categoryId argument and the selected value
     * is denoted by the categoryValueId argument.
     * 
     * @param endpointId
     *            a {@link Endpoint} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValue
     *            a category value (UDDI: categoryValue)
     * @param categoryDesc
     *            a category description (UDDI: categoryName), could be null
     * @throws DeploymentException
     *             if at least one of the given id doesn't exist in database or
     *             if value is null or empty
     */
    void addCategory(@NotNullParam String endpointId, @NotNullParam String categoryId,
            @NotNullParam String categoryValue, String categoryDesc) throws DeploymentException;

    /**
     * Remove the selected categories linked to a technical service
     * 
     * @param endpointId
     *            an {@link Endpoint} identifier
     * @param categoryIds
     *            ids of categories to remove, can be empty
     * @throws DeploymentException
     *             if no endpoint found for the given id or if no category found
     *             for the given ids
     */
    void removeCategories(String endpointId, List<String> categoryIds) throws DeploymentException;

    // Added By TIAR Abderrahmane
    /**
     * Calculate the global rating of a technical service.
     * 
     * @param endpointId
     *            the id of this endpoint.
     * 
     */
    float updateGlobalRating(String endpointId);

    /**
     * Update the Comments of an Endpoint.
     * 
     * @param endpointId
     *            The endpoint ID
     * 
     * @param comment
     *            The comment to add to this endpoint.
     */
    void updateComments(String endpointId, CommentTO comment);

    /**
     * Get all Endpoints tagged by the String tag.
     * 
     * @param tag
     *            the tag.
     * @return Endpoints related to tag.
     */
    List<EndpointTO> getEndpointsByTag(String tag);

    /**
     * Remove a String tag from the endpoint.
     * 
     * @param endpointId
     *            The endpoint ID
     * 
     * @param tag
     *            The tag to remove from.
     */
    void removeTag(String tag, String endpointId);

    /**
     * Remove a Comment from the endpoint.
     * 
     * @param endpointId
     *            The endpoint ID
     * 
     * @param commentId
     *            The comment ID.
     */
    void removeComment(String serviceId, String commentId);

    void removeServiceSpec(String endpointId, String serviceSpecificationId)
            throws DeploymentException;

    String addServiceDefFile(@NotNullParam Document wsdlDoc, String wsdlFileName,
            @NotNullParam String endpointId, boolean storeEntities) throws DeploymentException,
            TimeoutException;

    String addServiceDefFile(@NotNullParam URI wsdlURI, String wsdlFileName,
            @NotNullParam String endpointId, boolean storeEntities) throws DeploymentException,
            TimeoutException;
}
