/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechServiceManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.deployment;

import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.concurrent.TimeoutException;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.ServiceSearchProperties;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.metadata.SimpleFileTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.util.ContentType;
import org.springframework.transaction.annotation.Transactional;
import org.w3c.dom.Document;

/**
 * A manager of tech services. Provides methods to create, retrieve, delete or
 * update tech services, add or remove related docs, categories. Method
 * arguments must be non null, unless the contrary is explicitly specified.
 * 
 * @author ambarthe, ofabre - eBM Websourcing
 * 
 */
@Transactional
public interface TechServiceManager {

    /**
     * Add a new tech service in registry.
     * 
     * @param techServiceTO
     *            the tech service to add
     * @return the ID of the added tech service
     * @throws DeploymentException
     *             if a service already exist with the given name or the name is
     *             null or empty
     */
    String createTechService(TechnicalServiceTO techServiceTO) throws DeploymentException;

    /**
     * Return the list of all tech services in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters , could be null
     * @return the {@link List} of all {@link TechnicalServiceTO} in the
     *         registry, not null, can be empty
     */
    @Transactional(readOnly = true)
    List<TechnicalServiceTO> getAllTechServices(RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the endpoint with the given name, which is part of the tech
     * service with the given name.
     * 
     * @param techServiceName
     *            A tech service name
     * @param endpointName
     *            An endpoint name
     * @return the {@link EndpointTO} with the given name, which is part of the
     *         tech service with the given name. Can be null
     * @throws DeploymentException
     *             if an error occurs during endpoint retrieval
     */
    @Transactional(readOnly = true)
    EndpointTO getEndpoint(String techServiceName, String endpointName) throws DeploymentException;

    /**
     * Retrieve an technical service matching the given ID
     * 
     * @param techServiceId
     *            a tech service ID
     * @param user
     *            a user login. Will be used to retrieve related rating for this
     *            service. Can be null
     * @return an {@link TechnicalServiceTO} matching the given ID
     * @throws DeploymentException
     *             if no {@link TechnicalService} can be found for the given Id
     */
    @Transactional(readOnly = true)
    TechnicalServiceTO getTechService(@NotNullParam String techServiceId, String user)
            throws DeploymentException;

    /**
     * Retrieve a list of tech services matching the given IDs
     * 
     * @param servicesId
     *            a {@link List} of {@link String} service IDs
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters
     * @return a {@link List} of {@link TechnicalServiceTO} matching the given
     *         IDs
     * @throws DeploymentException
     *             if an error occurs during tech services retrieval
     */
    @Transactional(readOnly = true)
    List<TechnicalServiceTO> loadServicesFromWSDL(String[] servicesId,
            RequestOptionsTO requestOptionsTO) throws DeploymentException;

    /**
     * Remove the tech services matching the given ID from the registry
     * 
     * @param techServiceId
     *            a tech service ID
     * @throws DeploymentException
     *             if no service found for the given techService id
     */
    void removeTechService(String techServiceId) throws DeploymentException;

    /**
     * Retrieve a list of tech services matching the given criteria for the
     * given properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces), mustn't be empty
     * @param searchedProperties
     *            the searched properties as a list of
     *            {@link ServiceSearchProperties}, mustn't be empty
     * @param iOptions
     *            sort, pagination and case sensitive parameters. Could be null
     * @return a {@link List} of {@link TechnicalServiceTO} matching the given
     *         criteria for the given properties
     * @throws DeploymentException
     *             if you try to search services without specifying search
     *             criteria
     */
    @Transactional(readOnly = true)
    List<TechnicalServiceTO> searchTechService(@NotNullParam String searchCriteria,
            List<ServiceSearchProperties> searchedProperties, RequestOptionsTO iOptions)
            throws DeploymentException;

    /**
     * Update the tech service in the registry.
     * 
     * @param techServiceTO
     *            the tech service to update
     * @return the ID of the updated tech service
     * @throws DeploymentException
     *             if no service found for the given techServ id to update or if
     *             a service already exist with the given name or the name is
     *             null or empty
     */
    String updateTechService(TechnicalServiceTO techServiceTO) throws DeploymentException;

    /**
     * Return a {@link List} of {@link SimpleFileTO} that maps service related
     * docs
     * 
     * @param techServiceId
     *            a service id
     * @return {@link List} of {@link SimpleFileTO} that maps service related
     *         docs
     * @throws DeploymentException
     *             if an error occurs during related docs retrieval
     */
    List<SimpleFileTO> getRelatedDocsForTechServ(String techServiceId) throws DeploymentException;

    /**
     * Return a {@link List} of {@link KeyedRefTO} that maps service related
     * categories
     * 
     * @param techServiceId
     *            a service id
     * @return {@link List} of {@link KeyedRefTO} that maps service related
     *         categories
     * @throws DeploymentException
     *             if the given service doesn't exist
     */
    List<KeyedRefTO> getCategoriesForTechServ(String techServiceId) throws DeploymentException;

    /**
     * Return the content of the specified related document as a {@link String}
     * 
     * @param relatedDocId
     *            a document id
     * @return the content of the specified related document as a {@link String}
     * @throws DeploymentException
     *             if an error occurs during related doc content retrieval
     */
    String getRelatedDocContentAsString(String relatedDocId) throws DeploymentException;

    /**
     * Return the content of the specified related document as an
     * {@link InputStream}
     * 
     * @param relatedDocId
     *            a document id
     * @return the content of the specified related document as an
     *         {@link InputStream}
     * @throws DeploymentException
     *             if no document found for the given id or if the document
     *             content can't be loaded
     */
    InputStream getRelatedDocContent(String relatedDocId) throws DeploymentException;

    /**
     * Registers a new document in the registry (supported types are: DOC, PDF,
     * XML, HTML, XLS, PPT, TXT, RTF, ODT, ODS, ODP). Indexes its content. Links
     * it to the given service (denoted by the given serviceId).
     * 
     * @param serviceId
     *            a {@link TechnicalService} identifier
     * @param mimetype
     *            a mimetype (supported Mimetypes are : application/msword,
     *            application/pdf, text/html, text/xml,
     *            application/vnd.ms-excel, application/vnd.ms-powerpoint,
     *            text/plain, application/rtf,
     *            application/vnd.oasis.opendocument.text,
     *            application/vnd.oasis.opendocument.spreadsheet,
     *            application/vnd.oasis.opendocument.presentation). See
     *            {@link ContentType} for supported content types
     * @param docContent
     *            the content of the document as a byte array.
     * @param fileName
     *            the file name, could be null
     * @return the created file identifier
     * @throws DeploymentException
     *             if no {@link TechnicalService} found for the given serviceId
     *             or if the document can't be registered or indexed
     */
    String registerRelatedDoc(@NotNullParam String serviceId, @NotNullParam String mimetype,
            @NotNullParam byte[] docContent, String fileName) throws DeploymentException;

    /**
     * Remove a related document from the registry associate to the given
     * service. Service and document are denoted by the given identifiers
     * 
     * @param serviceId
     *            a service identifier
     * @param documentId
     *            a document identifier
     * @throws DeploymentException
     *             if no service or document found for the given ids
     */
    void removeRelatedDoc(String serviceId, String documentId) throws DeploymentException;

    /**
     * Remove the selected categories linked to a technical service
     * 
     * @param techServiceId
     *            a service identifier
     * @param categoryIds
     *            ids of categories to remove, can be empty
     * @throws DeploymentException
     *             if no service found for the given id or if no category found
     *             for the given ids
     */
    void removeCategories(String techServiceId, List<String> categoryIds)
            throws DeploymentException;

    /**
     * Add a category to the given tech service. The added category is of the
     * type of the category denoted by the categoryId argument and the selected
     * value is denoted by the categoryValueId argument.
     * 
     * @param techServiceId
     *            a {@link TechnicalService} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValueId
     *            an identifier of a {@link CategoryValue} containing category
     *            info to add.
     * @throws DeploymentException
     *             if at least one of the given id doesn't exist in database or
     *             if you try to add the same category twice
     */
    void addCategory(String techServiceId, String categoryId, String categoryValueId)
            throws DeploymentException;

    /**
     * Add a category to the given tech service. The added category is of the
     * type of the category denoted by the categoryId argument and the selected
     * value is denoted by the categoryValueId argument.
     * 
     * @param techServiceId
     *            a {@link TechnicalService} id
     * @param categoryId
     *            a {@link Category} id
     * @param categoryValue
     *            a category value (UDDI: categoryValue)
     * @param categoryDesc
     *            a category description (UDDI: categoryName). Could be null
     * @throws DeploymentException
     *             if at least one of the given id doesn't exist in database or
     *             if you try to add the same category twice
     */
    void addCategory(@NotNullParam String techServiceId, @NotNullParam String categoryId,
            @NotNullParam String categoryValue, String categoryDesc) throws DeploymentException;

    // Added By TIAR Abderrahmane
    /**
     * Calculate the global rating of a technical service.
     * 
     * @param serviceId
     *            the id of this service.
     * 
     */
    float updateGlobalRating(String serviceId);

    /**
     * Update the Comments of a techService.
     * 
     * @param serviceId
     *            The technical Service ID.
     * 
     * @param comment
     *            The comment to add to this technical Service.
     */
    void updateComments(String serviceId, CommentTO comment);

    /**
     * Get all Technical Services tagged by the String tag.
     * 
     * @param tag
     *            the tag.
     * @return Techservices related to tag.
     */
    List<TechnicalServiceTO> getTechServicesByTag(String tag);

    /**
     * Remove a String tag from the Technical Service.
     * 
     * @param serviceId
     *            The technical Service ID
     * 
     * @param tag
     *            The tag to remove from.
     */
    void removeTag(String tag, String serviceId);

    /**
     * Remove a Comment from the endpoint.
     * 
     * @param serviceId
     *            The technical Service ID.
     * 
     * @param commentID
     *            The EndPoint ID.
     */
    void removeComment(String serviceId, String commentId);

    /**
     * Add the given specification to the given service. If "storeEntities" is
     * true, register described entities too. If a "wsdlFileName" is specified,
     * use it as service spec name.
     * 
     * @param wsdlURI
     *            an {@link URI} pointing on a service spec file
     * @param wsdlFileName
     *            an optional service spec content
     * @param techServiceId
     *            a service identitifer
     * @param storeEntities
     *            a boolean that specifies if entities described in spec file
     *            must be registered too
     * @return the registered service specification id
     * @throws DeploymentException
     *             if the service specification can't be registered
     * @throws TimeoutException
     *             if service spec registration timeout
     */
    String addServiceDefFile(URI wsdlURI, String wsdlFileName, String techServiceId,
            boolean storeEntities) throws DeploymentException, TimeoutException;

    /**
     * Add the given specification to the given service. If "storeEntities" is
     * true, register described entities too. If a "wsdlFileName" is specified,
     * use it as service spec name.
     * 
     * @param wsdlDoc
     *            the service spec content as {@link Document}
     * @param wsdlFileName
     *            an optional service spec content
     * @param techServiceId
     *            a service identitifer
     * @param storeEntities
     *            a boolean that specifies if entities described in spec file
     *            must be registered too
     * @return the registered service specification id
     * @throws DeploymentException
     *             if the service specification can't be registered
     * @throws TimeoutException
     *             if service spec registration timeout
     */
    String addServiceDefFile(Document wsdlDoc, String wsdlFileName, String techServiceId,
            boolean storeEntities) throws DeploymentException, TimeoutException;

    /**
     * Unlink the given service specification from the given service. If the
     * service spec isn't connected to a service after this unlink, it is totaly
     * removed from registry.
     * 
     * @param serviceId
     *            a technical service id
     * @param serviceSpecificationId
     *            a service specification id
     * @throws DeploymentException
     *             if the specified service is registered or if the specified
     *             specification isn't registered
     */
    void removeServiceSpec(String serviceId, String serviceSpecificationId)
            throws DeploymentException;

    void addEndpoint(String endpointId, String serviceId) throws DeploymentException;
}
