/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * PersonManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.organization;

import java.util.List;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.organization.PersonSearchProperties;
import org.ow2.dragon.api.to.organization.PersonTO;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.springframework.transaction.annotation.Transactional;

/**
 * A manager of persons. Provides methods to create, retrieve, delete or update
 * persons
 * 
 * @author ambarthe, ofabre - eBM WebSourcing
 * 
 */
@Transactional
public interface PersonManager {

    /**
     * Add a new person in registry. Firstname, Lastname and email must be
     * specified
     * 
     * @param personTO
     *            the {@link PersonTO} handling data of the {@link Person} to
     *            add
     * @return the id of the added {@link Person}
     * @throws OrganizationException
     *             if you try to create a person with firstname/lastname/email
     *             that already exists or a person without firstname, lastname
     *             and email
     */
    public String createPerson(PersonTO personTO) throws OrganizationException;

    /**
     * Return the list of all persons in the registry
     * 
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of all {@link PersonTO} handling {@link Person}
     *         data in the registry, couldn't be null, can be empty
     */
    @Transactional(readOnly = true)
    public List<PersonTO> getAllPersons(RequestOptionsTO requestOptionsTO);

    /**
     * Retrieve the person matching the given ID
     * 
     * @param personId
     *            a {@link Person} id
     * @return the {@link PersonTO} handling data of the {@link Person} matching
     *         the given id
     * @throws OrganizationException
     *             if no person found for the given id
     */
    @Transactional(readOnly = true)
    public PersonTO getPerson(String personId) throws OrganizationException;

    /**
     * Retrieve the list of persons that are assigned to the given post in the
     * given organization unit
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     * @param postId
     *            a {@link Post} id
     * @param optionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of {@link PersonTO}s that are assigned to the
     *         given post in the given organization unit
     */
    public List<PersonTO> getPersonsByOrgAndPost(String organizationId, String postId,
            RequestOptionsTO optionsTO);

    /**
     * Retrieve the list of persons that are parts of the given organization
     * unit
     * 
     * @param organizationId
     *            an {@link OrganizationUnit} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of {@link PersonTO}s that are parts of the given
     *         {@link OrganizationUnit}
     * @throws OrganizationException
     *             if an error occurs during persons retrieval
     */
    @Transactional(readOnly = true)
    public List<PersonTO> getPersonsByOrganization(String organizationId,
            RequestOptionsTO requestOptionsTO) throws OrganizationException;

    /**
     * Retrieve the list of persons that are assigned to the given post
     * 
     * @param postId
     *            a {@link Post} id
     * @param requestOptionsTO
     *            sort, pagination and case sensitive parameters, could be null
     * @return the {@link List} of {@link PersonTO}s that are assigned to the
     *         given {@link Post}
     * @throws OrganizationException
     *             if an error occurs during persons retrieval
     */
    @Transactional(readOnly = true)
    public List<PersonTO> getPersonsByPost(String postId, RequestOptionsTO requestOptionsTO)
            throws OrganizationException;

    /**
     * Remove the person matching the given id from the registry
     * 
     * @param personId
     *            a {@link Person} id
     */
    public void removePerson(String personId);

    /**
     * Retrieve a list of persons matching the given criteria for the given
     * properties
     * 
     * @param searchCriteria
     *            the search criteria (a String containing criteria separated
     *            with whitespaces)
     * @param searchedProperties
     *            the searched properties
     * @param iOptions
     *            sort, pagination and case sensitive parameters, could be null
     * @return a {@link List} of {@link PersonTO} matching the given criteria
     *         for the given properties
     * @throws OrganizationException
     *             if an error occurs during persons retrieval
     */
    @Transactional(readOnly = true)
    public List<PersonTO> searchPersons(@NotNullParam String searchCriteria,
            List<PersonSearchProperties> searchedProperties, RequestOptionsTO iOptions)
            throws OrganizationException;

    /**
     * Update the person in the registry
     * 
     * @param personTO
     *            the {@link PersonTO} handling information of the
     *            {@link Person} to update
     * @return the updated {@link Person} id
     * @throws OrganizationException
     *             if an error occurs during person update
     */
    public String updatePerson(PersonTO personTO) throws OrganizationException;

}
