/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * WSDLManager.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.api.service.wsdl;

import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import org.ow2.dragon.aop.annotation.NotNullParam;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.specification.ServiceSpecificationTO;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.service.wsdl.importreport.WsdlEntitiesImportReport;
import org.springframework.transaction.annotation.Transactional;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

/**
 * This class allows to import WSDL file descriptions into the registry and to
 * retrieve them. It also allows to retrieve created
 * {@link ServiceSpecification} entities linked to {@link Endpoint} or
 * {@link TechnicalService}. All method parameters must not be null unless the
 * contrary is specified.
 * 
 * @author ofabre - eBM WebSourcing
 * 
 */
public interface WSDLManager {

    /**
     * Retrieve the list of service specifications (like WSDL files...) where
     * the given endpoint is described
     * 
     * @param epId
     *            an endpoint ID
     * @return the {@link List} of {@link ServiceSpecificationTO}s (like WSDL
     *         files...) where the given {@link EndpointTO} is described
     * @throws WSDLServiceException
     *             if no endpoint found for the given id
     */
    @Transactional(readOnly = true)
    List<ServiceSpecificationTO> getServSpecsForEndpoint(String epId) throws WSDLServiceException;

    /**
     * Retrieve the list of service specifications (like WSDL files...) where
     * the given tech service is described
     * 
     * @param techServiceId
     *            a tech service ID
     * @return the {@link List} of {@link ServiceSpecificationTO}s (like WSDL
     *         files...) where the given {@link TechnicalServiceTO} is described
     * @throws WSDLServiceException
     *             if no tech service found for the given id
     */
    @Transactional(readOnly = true)
    List<ServiceSpecificationTO> getServSpecsForTechServ(String techServiceId)
            throws WSDLServiceException;

    /**
     * Retrieve the WSDL description linked to the given service specification
     * as an input stream
     * 
     * @param servSpecId
     *            a service spec ID
     * @return the WSDL description linked to the given
     *         {@link ServiceSpecificationTO}, as an {@link InputStream}
     * @throws WSDLServiceException
     *             if no service spec found for the given id or if the wsdl file
     *             content can't be retrieved
     */
    @Transactional(readOnly = true)
    InputStream getWsdlDesc(String servSpecId) throws WSDLServiceException;

    /**
     * Retrieve the WSDL description linked to the given service specification
     * as a String
     * 
     * @param servSpecId
     *            a service spec ID
     * @return the WSDL description linked to the given
     *         {@link ServiceSpecificationTO}, as an {@link String}
     * @throws WSDLServiceException
     *             if no service spec found for the given id, if the wsdl file
     *             content can't be retrieved or if the wsdl file content can't
     *             be converted as string
     */
    @Transactional(readOnly = true)
    String getWsdlDescAsString(String servSpecId) throws WSDLServiceException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods mustn't be used to import WSDL
     * that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(Document, Map, String)})
     * 
     * @param domDocument
     *            the WSDL description as a DOM {@link Document}
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(Document domDocument) throws WSDLServiceException,
            TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param domDocument
     *            the WSDL description as a DOM {@link Document}
     * @param imports
     *            a {@link Map} (key = import location --> value = import
     *            content as {@link Document}) of all imported WSDL. Could be
     *            null
     * @param wsdlFileName
     *            the wsdl file name. Could be null
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam Document domDocument,
            Map<String, Document> imports, String wsdlFileName) throws WSDLServiceException,
            TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods mustn't be used to import WSDL
     * that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(Document, Map, String)})
     * 
     * @param domDocument
     *            the WSDL description as a DOM {@link Document}
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam Document domDocument, String wsdlFileName)
            throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry
     * 
     * @param domElement
     *            the WSDL description as a DOM {@link Element}
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if an error occurs during description import
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(Element domElement) throws WSDLServiceException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlURI
     *            {@link URI} pointing on a WSDL description
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if an error occurs during description import
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(URI wsdlURI) throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlURI
     *            {@link URI} pointing on a WSDL description
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if an error occurs during description import
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam final URI wsdlURI, final String wsdlFileName)
            throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlURI
     *            {@link URI} pointing on a WSDL description
     * @param imports
     *            a {@link Map} (key = import location --> value = import
     *            content as {@link URI}) of all imported WSDL. Could be null
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import
     * @throws WSDLServiceException
     *             if an error occurs during description import
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam final URI wsdlURI, final Map<String, URI> imports,
            final String wsdlFileName) throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. This methods mustn't be used
     * to import WSDL that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(String, Map, String)}). It
     * supports absolute imports. If you try to register the same wsdl
     * (canonically equals without comments) twice, it will be skipped and a
     * null array is returned.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link String}
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(String wsdlContent) throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link String}
     * @param imports
     *            a {@link Map} (key = import location --> value = import
     *            content as {@link String}) of all imported WSDL. Could be null
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam String wsdlContent, Map<String, String> imports,
            String wsdlFileName) throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link String}
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam String wsdlContent, String wsdlFileName)
            throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. This methods mustn't be used
     * to import WSDL that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(InputSource, Map, String)}). It
     * supports absolute imports. If you try to register the same wsdl
     * (canonically equals without comments) twice, it will be skipped and a
     * null array is returned.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link InputSource}
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(InputSource wsdlContent) throws WSDLServiceException,
            TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. This methods mustn't be used
     * to import WSDL that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(InputSource, Map, String)}). If
     * you try to register the same wsdl (canonically equals without comments)
     * twice, it will be skipped and a null array is returned. This methods must
     * be used to import WSDL that uses relative imports of other WSDL.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link InputSource}
     * @param imports
     *            a {@link Map} (key = import location --> value = import
     *            content as {@link InputSource}) of all imported WSDL. Could be
     *            null
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam InputSource wsdlContent,
            Map<String, InputSource> imports, String wsdlFileName) throws WSDLServiceException,
            TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. If you try to register the
     * same wsdl (canonically equals without comments) twice, it will be skipped
     * and a null array is returned. This methods must be used to import WSDL
     * that uses relative imports of other WSDL.
     * 
     * @param wsdlContent
     *            the WSDL description content as {@link InputSource}
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @return IDs of all registered tech service as result of the WSDL import,
     *         could be null (if no services registered)
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    @Transactional(readOnly = false)
    String[] importServiceDefFile(@NotNullParam InputSource wsdlContent, String wsdlFileName)
            throws WSDLServiceException, TimeoutException;

    /**
     * Read a WSDL description and add all described entities (Services,
     * Interfaces, Endpoints etc.) in the registry. This methods mustn't be used
     * to import WSDL that uses relative imports of other WSDL (see
     * {@link WSDLManager#importServiceDefFile(InputSource, Map, String)}). This
     * methods must be used to import WSDL that uses relative imports of other
     * WSDL. If "storeEntities" is set to "true", all described entities will be
     * stored. "isImport" specify if you are trying to import a wsdl file alone
     * or to add it to an existing service
     * 
     * @param wsdlSource
     *            the WSDL description content as {@link InputSource}
     * @param imports
     *            a {@link Map} (key = import location --> value = import
     *            content as {@link InputSource}) of all imported WSDL. Could be
     *            null
     * @param wsdlFileName
     *            the wsdl file name. Can be null
     * @param storeEntities
     *            boolean that specifies if described entities must be
     *            registered or not
     * @param isImport
     *            boolean that specifies if you are trying to import a wsdl file
     *            alone or to add it to an existing service
     * @return
     * @throws WSDLServiceException
     *             if the wsdl file can't be read or registered, if the wsdl
     *             file signature can't be extracted or if the wsdl file
     *             contains inconsistent entities (ex: same interface names but
     *             with different operations...)
     * @throws TimeoutException
     *             if a timeout occured during wsdl parsing
     */
    WsdlEntitiesImportReport registerServiceDefFile(InputSource wsdlSource,
            Map<String, InputSource> imports, String wsdlFileName, boolean storeEntities,
            boolean isImport) throws WSDLServiceException, TimeoutException;
}
