/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UniversalORMDAO.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.persistence.dao;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import org.hibernate.NonUniqueResultException;

import com.trg.search.IMutableSearch;
import com.trg.search.ISearch;
import com.trg.search.SearchResult;

/**
 * Data Access Object (DAO) interface.
 * 
 * The only real difference between this DAO and {@link GenericORMDAO} is that
 * instances of java.lang.Class are passed into the methods in this class, and
 * they are part of the constructor in the GenericDao, hence you'll have to do
 * some casting if you use this one.
 * 
 * @see org.appfuse.dao.GenericDao
 */
public interface UniversalORMDAO {

    /**
     * Generic method used to get all objects of a particular type. This is the
     * same as lookup up all rows in a table.
     * 
     * @param clazz
     *            the type of objects (a.k.a. while table) to get data from
     * @return List of populated objects
     */
    List getAll(Class clazz);

    /**
     * Generic method to get an object based on class and identifier.
     * 
     * @param clazz
     *            model class to lookup
     * @param id
     *            the identifier (primary key) of the class
     * @return a populated object or null if not found
     */
    Object get(Class clazz, Serializable id);

    /**
     * Generic method to save an object - handles both update and insert.
     * 
     * @param o
     *            the object to save
     * @return a populated object
     */
    Object save(Object o);

    /**
     * Checks for existence of an object of type T using the id arg.
     * 
     * @param id
     *            the id of the entity
     * @return - true if it exists, false if it doesn't
     */
    boolean exists(Class clazz, Serializable id);

    /**
     * Generic method to delete an object based on class and id
     * 
     * @param clazz
     *            model class to lookup
     * @param id
     *            the identifier (primary key) of the class
     */
    void remove(Class clazz, Serializable id);

    /**
     * Generic method to delete an object
     * 
     * @param object
     *            the object to remove
     */
    void remove(Object object);

    /**
     * Generic method to delete all object based on class and ids list
     * 
     * @param clazz
     *            model class to lookup
     * @param ids
     *            the identifiers (primary key) of object to delete
     */
    void removeAll(Class clazz, List ids);

    void removeAll(List objects);

    /**
     * Retrieve a {@link List} of entities matching given ids
     * 
     * @param clazz
     *            model class to lookup
     * @param ids
     *            a {@link List} of ids
     * @return a {@link List} of entities matching ids, must be non null, could
     *         be empty
     */
    List getAll(Class clazz, List ids);

    /**
     * Retrieve a {@link List} of entities matching given ids, sorted and
     * paginated according to the given request options
     * 
     * @param clazz
     *            model class to lookup
     * @param ids
     *            a {@link List} of ids
     * @param requestOptions
     *            include sort order and pagination information
     * @return a sorted/paginated {@link List} of entities matching ids, must be
     *         non null, could be empty
     */
    List getAll(Class clazz, List ids, RequestOptions requestOptions);

    /**
     * Generic method used to get all objects of a particular type, sorted and
     * paginated according to the given request options. This is the same as
     * lookup up all rows in a table.
     * 
     * @param clazz
     *            model class to lookup
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return List of populated objects
     */
    List getAll(Class clazz, RequestOptions requestOptionsTO);

    /**
     * This method allows to search Objects on String properties, fitting search
     * criteria. Properties can be direct object fields or fields of included
     * objects (in this case, fields must be referenced by the doted notation
     * like "foo.bar"). Use a like statement for each criteria on each property.
     * Results are sorted and paginated in respect of the given request options.
     * 
     * @param clazz
     *            model class to lookup
     * @param criteria
     *            the search criteria
     * @param properties
     *            the searched properties
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return a list of object matching the different criteria sorted and
     *         paginated
     */
    List searchLike(Class clazz, String[] criteria, String[] properties,
            RequestOptions requestOptionsTO);

    /**
     * This method allows to search Objects on String properties, fitting search
     * criteria. Properties can be direct object fields or fields of included
     * objects (in this case, fields must be referenced by the doted notation
     * like "foo.bar"). Use a equality statement for each criteria on each
     * property. Results are sorted and paginated in respect of the given
     * request options.
     * 
     * @param clazz
     *            model class to lookup
     * @param criteria
     *            the search criteria
     * @param properties
     *            the searched properties
     * @param requestOptionsTO
     *            include sort order and pagination information
     * @return a list of object matching the different criteria sorted and
     *         paginated
     */
    List searchEquals(Class clazz, String[] criteria, String[] properties,
            RequestOptions requestOptionsTO);

    /**
     * Search for objects based on the search parameters in the specified
     * <code>ISearch</code> object.
     * 
     * @see ISearch
     */
    @SuppressWarnings("unchecked")
    public List search(Class clazz, IMutableSearch search);

    /**
     * Returns the total number of results that would be returned using the
     * given <code>ISearch</code> if there were no paging or maxResult limits.
     * 
     * @see ISearch
     */
    public int count(Class clazz, IMutableSearch search);

    /**
     * Returns a <code>SearchResult</code> object that includes the list of
     * results like <code>search()</code> and the total length like
     * <code>searchLength</code>.
     * 
     * @see ISearch
     */
    @SuppressWarnings("unchecked")
    public SearchResult searchAndCount(Class clazz, IMutableSearch search);

    /**
     * Search for a single result using the given parameters.
     */
    public Object searchUnique(Class clazz, IMutableSearch search) throws NonUniqueResultException;

    /**
     * Find a list of records by using a named query
     * 
     * @param queryName
     *            query name of the named query
     * @param queryParams
     *            a map of the query names and the values
     * @return a list of the records found
     */
    List findByNamedQuery(String queryName, Map<String, Object> queryParams);

    Object merge(Object object);

    Object persist(Object object);

    Object saveOnly(Object object);

    Object updateOnly(Object object);
}
