/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDIIdentifierGenerator.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.persistence.util;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.log4j.Logger;
import org.hibernate.HibernateException;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.exception.JDBCExceptionHelper;
import org.hibernate.id.IdentifierGenerator;
import org.ow2.dragon.util.RegistryConfig;
import org.ow2.dragon.util.StringHelper;
import org.ow2.dragon.util.UDDIStandardTModelKeys;

/**
 * The UDDI key generator. Generates a key like this:
 * 
 * uddiScheme : RootDomain : UUID
 * 
 * @author ofabre
 * 
 */
public class UDDIIdentifierGenerator implements IdentifierGenerator {

    public static final String UDDI_V3_SCHEME = "uddi";

    public static final String UDDI_V2_SCHEME = "uuid";

    private static final String PARTITION_SEPARATOR = ":";

    private static final String ROOT_DOMAIN = "root.domain";

    private Logger log = Logger.getLogger(UDDIIdentifierGenerator.class);

    /*
     * (non-Javadoc)
     * 
     * @seeorg.hibernate.id.IdentifierGenerator#generate(org.hibernate.engine.
     * SessionImplementor, java.lang.Object)
     */
    public Serializable generate(SessionImplementor session, Object object)
            throws HibernateException {

        Serializable id = session.getEntityPersister(null, object)
        // TODO: cache the persister, this shows up in yourkit
                .getIdentifier(object, session.getEntityMode());

        // If the id isn't specified by the application, generate it
        if (id == null || (id instanceof String && StringHelper.isNullOrEmpty((String) id))) {
            id = getUddiV3KeyPrefix() + getUUID();
        }

        // Based on JAVA UUID
        return id;

        // Based on SQL GUID
        // return UDDI_SCHEME + PARTITION_SEPARATOR + rootDomain +
        // PARTITION_SEPARATOR
        // + getGUID(session);
    }

    private synchronized String getUUID() {
        return UUID.randomUUID().toString();
    }

    private String getGUID(SessionImplementor session) {
        final String sql = session.getFactory().getDialect().getSelectGUIDString();
        try {
            PreparedStatement st = session.getBatcher().prepareSelectStatement(sql);
            try {
                ResultSet rs = st.executeQuery();
                final String result;
                try {
                    rs.next();
                    result = rs.getString(1);
                } finally {
                    rs.close();
                }
                log.debug("GUID identifier generated: " + result);
                return result;
            } finally {
                session.getBatcher().closeStatement(st);
            }
        } catch (SQLException sqle) {
            throw JDBCExceptionHelper.convert(session.getFactory().getSQLExceptionConverter(),
                    sqle, "could not retrieve GUID", sql);
        }
    }

    public static String toUddiV2Id(String uddiV3Id) {
        String v2Key = null;
        if (!StringHelper.isNullOrEmpty(uddiV3Id)) {
            uddiV3Id = uddiV3Id.toLowerCase();
            // Try to find it in conversion map
            v2Key = (String) UDDIStandardTModelKeys.UDDIV2TOV3KEY.getKey(uddiV3Id);
            if (v2Key == null) {
                // Uses a simple algorithm to convert it to uddi v3 key
                if (!uddiV3Id.startsWith(getUddiV3KeyPrefix())) {
                    throw new HibernateException("Trying to use an invalid uddi v3 key ('"
                            + uddiV3Id + "'). Valid v3 key for this registry are: '"
                            + getUddiV3KeyPrefix() + "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'");
                } else {
                    v2Key = getUddiV2KeyPrefix()
                            + uddiV3Id.substring(getUddiV3KeyPrefix().length());
                }
            }
        }
        return v2Key;
    }

    private static String getUddiV3KeyPrefix() {
        String rootDomain = "";
        try {
            rootDomain = RegistryConfig.getConfiguration().getString(ROOT_DOMAIN);
        } catch (ConfigurationException ce) {
            throw new HibernateException(
                    "Can't retrieve root.domain property from registry config file", ce);
        }
        return UDDI_V3_SCHEME + PARTITION_SEPARATOR + rootDomain + PARTITION_SEPARATOR;
    }

    private static String getUddiV2KeyPrefix() {
        return UDDI_V2_SCHEME + PARTITION_SEPARATOR;
    }

    public static String toUddiV3Id(String uddiV2Id) {
        String v3Key = null;
        if (!StringHelper.isNullOrEmpty(uddiV2Id)) {
            uddiV2Id = uddiV2Id.toLowerCase();
            // Try to find it in conversion map
            v3Key = (String) UDDIStandardTModelKeys.UDDIV2TOV3KEY.get(uddiV2Id);
            if (v3Key == null) {
                // Uses a simple algorithm to convert it to uddi v3 key
                if (!uddiV2Id.startsWith(getUddiV2KeyPrefix())) {
                    throw new HibernateException("Trying to use an invalid uddi v2 key ('"
                            + uddiV2Id + "'). Valid v2 key are: '" + getUddiV2KeyPrefix()
                            + "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx'");
                } else {
                    v3Key = getUddiV3KeyPrefix()
                            + uddiV2Id.substring(getUddiV2KeyPrefix().length());
                }
            }
        }

        return v3Key;
    }

    /**
     * A valid id is an Id starting with "uddi:'root domain'" where root domain
     * is the "root.domain" property defined in dragon.properties file or an id
     * registered in the uddiV2toV3 map
     * 
     * @param uddiV3Id
     *            a v3 id
     * @return true if it's a valid Id false otherwise
     */
    public static boolean isUddiV3ValidId(String uddiV3Id) {
        boolean result = UDDIStandardTModelKeys.UDDIV2TOV3KEY.containsValue(uddiV3Id);
        if (!result) {
            result = uddiV3Id.startsWith(getUddiV3KeyPrefix());
        }
        return result;
    }

}
