/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * RoleGroupManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.administration;

import java.util.ArrayList;
import java.util.List;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.api.service.administration.RoleGroupException;
import org.ow2.dragon.api.service.administration.RoleGroupManager;
import org.ow2.dragon.api.to.administration.RoleGroupTO;
import org.ow2.dragon.persistence.bo.administration.Role;
import org.ow2.dragon.persistence.bo.administration.RoleGroup;
import org.ow2.dragon.persistence.bo.administration.User;
import org.ow2.dragon.persistence.dao.administration.RoleDAO;
import org.ow2.dragon.persistence.dao.administration.RoleGroupDAO;
import org.ow2.dragon.persistence.dao.administration.UserDAO;
import org.ow2.dragon.util.StringHelper;

/**
 * @author strino - eBM WebSourcing
 * 
 */

public class RoleGroupManagerImpl implements RoleGroupManager {
    private RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler;

    private RoleGroupDAO roleGroupDAO;

    private RoleDAO roleDAO;

    private UserDAO userDAO;

    // create the roles group admin if not exists in database
    public void init() {

        // retrieve if the role group already exist
        RoleGroup roleGroupBO = roleGroupDAO.getRoleGroupByName("admin");

        if (roleGroupBO == null) {
            RoleGroup roleGroup = new RoleGroup();
            roleGroup.setName("admin");
            List<Role> listRoles = roleDAO.getAll();
            roleGroup.setListRoles(listRoles);

            // save the role group in database
            roleGroupDAO.save(roleGroup);
        }

    }

    @CheckAllArgumentsNotNull
    public String createRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException {

        RoleGroup roleGroup = new RoleGroup();

        this.validateBeforeSaveOrUpdate(roleGroupTO, null);

        this.rolesAndUsersTransfertObjectAssembler.toRoleGroupBO(roleGroupTO, roleGroup);

        List<Role> listR = new ArrayList<Role>();
        List<Role> listRole = roleGroup.getListRoles();

        for (Role ro : listRole) {

            // retrieve the id role
            ro = this.roleDAO.getRoleByName(ro.getName());
            listR.add(ro);
        }

        roleGroup.setListRoles(listR);

        // create role group
        return this.roleGroupDAO.save(roleGroup).getId();

    }

    public List<RoleGroupTO> getAllRoleGroup() throws RoleGroupException {

        // retrieve all role group
        List<RoleGroup> listAllGroupe = this.roleGroupDAO.getAll();

        // transform role groups
        List<RoleGroupTO> listRole = this.rolesAndUsersTransfertObjectAssembler
                .toAllGroupes(listAllGroupe);

        return listRole;
    }

    @CheckAllArgumentsNotNull
    public List<RoleGroupTO> getRoleGroupNotInUser(String idUser) throws RoleGroupException {

        List<RoleGroupTO> listRoleGroupTO = new ArrayList<RoleGroupTO>();

        // retrieve user with its id
        User user = this.userDAO.get(idUser);

        List<RoleGroup> listRoleGroupInUser = new ArrayList<RoleGroup>();

        if (user != null) {
            listRoleGroupInUser = user.getListRoleGroup();
        } else {
            throw new RoleGroupException("A user with that name doesn't exist ! ");
        }

        // retrieve all role groups
        List<RoleGroup> allRoleGroup = this.roleGroupDAO.getAll();

        if (allRoleGroup != null) {

            if (!listRoleGroupInUser.isEmpty()) {

                // remove user role's groups from list all role groups
                allRoleGroup.removeAll(listRoleGroupInUser);

            }
            for (RoleGroup rG : allRoleGroup) {

                // transform role groups
                listRoleGroupTO.add(this.rolesAndUsersTransfertObjectAssembler.toRoleGroupTO(rG));
            }
        } else {
            throw new RoleGroupException("no role groups in database ! ");
        }

        return listRoleGroupTO;
    }

    @CheckAllArgumentsNotNull
    public RoleGroupTO getRoleGroup(String idGroup) throws RoleGroupException {

        // retrieve role group
        RoleGroup roleGroup = this.roleGroupDAO.get(idGroup);

        if (roleGroup == null) {
            throw new RoleGroupException("no role groups whith this id in database ! ");
        }

        // transform role group
        RoleGroupTO role = this.rolesAndUsersTransfertObjectAssembler.toRoleGroupTO(roleGroup);

        return role;
    }

    @CheckAllArgumentsNotNull
    public void removeRoleGroup(String idGroup) throws RoleGroupException {

        // retrieve role group by its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idGroup);

        if (roleGroup == null) {
            throw new RoleGroupException("no role groups whith this id in database ! ");
        }

        // remove role group
        this.roleGroupDAO.remove(roleGroup);
    }

    @CheckAllArgumentsNotNull
    public String updateRoleGroup(RoleGroupTO roleGroupTO) throws RoleGroupException {

        String id = roleGroupTO.getId();

        if (id == null) {
            throw new RoleGroupException("this role doesn't already exist in database : ");
        }

        // retrieve role group by its id
        RoleGroup group = this.roleGroupDAO.get(roleGroupTO.getId());

        if (group == null) {
            throw new RoleGroupException("this role doesn't already exist in database : " + id);
        }

        this.validateBeforeSaveOrUpdate(roleGroupTO, group.getName());

        this.rolesAndUsersTransfertObjectAssembler.toRoleGroupBO(roleGroupTO, group);

        // transform List<String> to List<Role>
        List<Role> listR = new ArrayList<Role>();
        List<Role> listRole = group.getListRoles();

        for (Role ro : listRole) {
            // retrieve the id role
            ro = this.roleDAO.getRoleByName(ro.getName());
            listR.add(ro);
        }

        group.setListRoles(listR);

        this.roleGroupDAO.save(group);

        return group.getId();
    }

    @CheckAllArgumentsNotNull
    public void removeUser(String idUser, String idRoleGroup) throws RoleGroupException {
        // retrieve roleGroup by its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);
        if (roleGroup == null) {
            throw new RoleGroupException(
                    "You are trying to remove a user to a non existing role whit id : "
                            + idRoleGroup);
        }

        // retrieve user by its id
        User user = this.userDAO.get(idUser);
        if (user == null) {
            throw new RoleGroupException(
                    "You are trying to remove a non existing user to a role whith id : "
                            + idRoleGroup);
        }

        roleGroup.removeUser(user);

        this.roleGroupDAO.save(roleGroup);
    }

    @CheckAllArgumentsNotNull
    public void addUser(String idUser, String idRoleGroup) throws RoleGroupException {

        // retrieve role group with its id
        RoleGroup roleGroup = this.roleGroupDAO.get(idRoleGroup);

        if (roleGroup == null) {
            throw new RoleGroupException(
                    "You are trying to add a user to a non existing role whit id : " + idRoleGroup);
        }

        // retrieve user with its id
        User user = this.userDAO.get(idUser);

        if (user == null) {
            throw new RoleGroupException(
                    "You are trying to add a non existing user to a role whith id : " + idRoleGroup);
        }
        // insert the new user in the RoleGroup list<User>
        roleGroup.addUser(user);

        // save role Group
        this.roleGroupDAO.save(roleGroup);

    }

    private void validateBeforeSaveOrUpdate(RoleGroupTO roleGroup, String nameRoleGroup)
            throws RoleGroupException {
        String roleGroupNameTO = roleGroup.getName();
        if (StringHelper.isNullOrEmpty(roleGroupNameTO)) {
            throw new RoleGroupException("Role Name must be speficied.");
        }

        if (!roleGroupNameTO.equalsIgnoreCase(nameRoleGroup)) {
            RoleGroup group = this.roleGroupDAO.getRoleGroupByName(roleGroupNameTO);
            if (group != null) {
                throw new RoleGroupException("A role with that name already exists : "
                        + roleGroupNameTO);
            }
        }

    }

    public void setRolesAndUsersTransfertObjectAssembler(
            RolesAndUsersTransfertObjectAssembler rolesAndUsersTransfertObjectAssembler) {
        this.rolesAndUsersTransfertObjectAssembler = rolesAndUsersTransfertObjectAssembler;
    }

    public void setRoleGroupDAO(RoleGroupDAO roleGroupDAO) {
        this.roleGroupDAO = roleGroupDAO;
    }

    public void setRoleDAO(RoleDAO roleDAO) {
        this.roleDAO = roleDAO;
    }

    public void setUserDAO(UserDAO userDAO) {
        this.userDAO = userDAO;
    }

}
