/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * EndpointManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.deployment;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeoutException;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.deployment.EndpointManager;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.wsdl.WSDLManager;
import org.ow2.dragon.api.service.wsdl.WSDLServiceException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Comment;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Rating;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.dao.RequestOptions;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.dao.deployment.CommentDAO;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.deployment.TechnicalServiceDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceSpecificationDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.service.wsdl.importreport.WsdlEntitiesImportReport;
import org.ow2.dragon.util.StringHelper;
import org.ow2.dragon.util.XMLUtil;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class EndpointManagerImpl implements EndpointManager {

    private EndpointDAO endpointDAO;

    private TechnicalServiceDAO technicalServiceDAO;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private ServiceSpecificationDAO serviceSpecificationDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    private MetadataService metadataService;

    private WSDLManager wsdlManager;

    // Added By TIAR Abderrahmane
    private CommentDAO commentDAO;

    public void setCommentDAO(final CommentDAO commentDAO) {
        this.commentDAO = commentDAO;
    }

    // End

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.deployment.EndpointManager#getAllEndpoints
     * (org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<EndpointTO> getAllEndpoints(final RequestOptionsTO requestOptionsTO)
            throws DeploymentException {
        final List<EndpointTO> result = new ArrayList<EndpointTO>();
        final List<Endpoint> endpoints = this.endpointDAO.getAll(this.transfertObjectAssembler
                .toEndpointRequestOptions(requestOptionsTO));
        this.toEndpointsTO(result, endpoints);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.deployment.EndpointManager#getEndpoint
     * (java.lang.String)
     */
    @CheckArgumentsNotNull
    public EndpointTO getEndpoint(final String endpointId, final String user)
            throws DeploymentException {
        // retrieve person bo
        final Endpoint endpointBO = this.endpointDAO.get(endpointId);
        if (endpointBO == null) {
            throw new DeploymentException("No endpoint found matching the given id: " + endpointId);
        }
        // create person to from bo
        final EndpointTO endpointTO = this.toEndpointTO(endpointBO, user);
        return endpointTO;
    }

    @CheckArgumentsNotNull
    public String createEndpoint(final String epName, final String epAddress, final String servId)
            throws DeploymentException {
        // Validate endpoint before creation
        validateEndpointBeforeCreation(epName, epAddress, servId);

        // Retrieve tech service to link
        TechnicalService technicalService = this.technicalServiceDAO.get(servId);
        if (technicalService == null) {
            throw new DeploymentException("The specified service doesn't exist: " + servId);
        }

        // create endpoint bo from to
        final Endpoint endpointBO = new Endpoint();
        endpointBO.setName(epName);
        endpointBO.setNetworkAddress(epAddress);

        // then add it to the given service
        technicalService.addEndpoint(endpointBO);

        // Persist new endpoint
        this.endpointDAO.save(endpointBO);
        return endpointBO.getId();
    }

    /**
     * @param endpointTO
     * @throws DeploymentException
     */
    private void validateEndpointBeforeCreation(final String epName, final String epAddress,
            final String servId) throws DeploymentException {
        // Check if the endpoint name isn't null or empty
        if (StringHelper.isNullOrEmpty(epName)) {
            throw new DeploymentException("Endpoint name must be specified.");
        }

        // Check if the parent service isn't null and if the parent service
        // id isn't null or empty
        if (StringHelper.isNullOrEmpty(servId)) {
            throw new DeploymentException(
                    "Endpoint must have a parent Service with a specified id.");
        }

        // check if the endpoint isn't already created : (endpoint name +
        // service id) must be unique
        Endpoint endpoint = this.endpointDAO.getByNameAndServiceId(servId, epName);
        if (endpoint != null) {
            throw new DeploymentException("Endpoint already created with the same name (" + epName
                    + ") in the same service (" + servId + ")");
        }
    }

    /**
     * @param endpointTO
     * @throws DeploymentException
     */
    private void validateEndpointBeforeUpdate(EndpointTO endpointTO) throws DeploymentException {
        // Check if the endpoint name isn't null or empty
        String epName = endpointTO.getName();
        if (StringHelper.isNullOrEmpty(epName)) {
            throw new DeploymentException("Endpoint name must be specified.");
        }

        // Check if the parent service isn't null and if the parent service
        // id isn't null or empty
        TechnicalServiceTO techService = endpointTO.getTechService();
        if (techService == null) {
            throw new DeploymentException("Endpoint must have a parent Service.");
        }
        String serviceId = techService.getId();
        if (StringHelper.isNullOrEmpty(serviceId)) {
            throw new DeploymentException(
                    "Endpoint must have a parent Service with a specified id.");
        }

        // check if the endpoint isn't already created : (endpoint name +
        // service id) must be unique
        Endpoint endpoint = this.endpointDAO.getByNameAndServiceId(serviceId, epName);
        if (endpoint != null && !endpointTO.getId().equals(endpoint.getId())) {
            throw new DeploymentException("Endpoint already created with the same name (" + epName
                    + ") in the same service (" + serviceId + ")");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.EndpointManager#
     * getEndpointsByBinding(java.lang.String,
     * org.ow2.dragon.ui.model.to.RequestOptions)
     */
    @CheckArgumentsNotNull
    public List<EndpointTO> getEndpointsByBinding(final String bindingId,
            final RequestOptionsTO requestOptionsTO) throws DeploymentException {
        throw new RuntimeException("not implemented method");
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.EndpointManager#
     * getEndpointsByTechService(java.lang.String,
     * org.ow2.dragon.ui.model.to.RequestOptions)
     */
    @CheckArgumentsNotNull
    public List<EndpointTO> getEndpointsByTechService(final String techServiceId,
            final RequestOptionsTO requestOptionsTO) {
        final List<EndpointTO> result = new ArrayList<EndpointTO>();

        // Search endpoints
        RequestOptions requestOptions = this.transfertObjectAssembler
                .toEndpointRequestOptions(requestOptionsTO);
        if (requestOptions != null) {
            requestOptions.setCaseSensitive(true);
        }
        final List<Endpoint> endpoints = this.endpointDAO.searchEquals(
                new String[] { techServiceId }, new String[] { "technicalService.id" },
                requestOptions);
        this.toEndpointsTO(result, endpoints);

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.ui.businessdelegate.deployment.EndpointManager#removeEndpoint
     * (java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public void removeEndpoint(final String endpointId) throws DeploymentException {
        final Endpoint endpoint = this.endpointDAO.get(endpointId);
        if (endpoint != null) {
            // Remove binding if this endpoint is the last one related to it
            Binding binding = endpoint.getBinding();
            if (binding != null) {
                Set<Endpoint> endpoints = binding.getEndpoints();
                endpoints.remove(endpoint);
                if (endpoints.size() == 0) {
                    this.removeBinding(binding);
                }
            }
            // Remove service spec if this endpoint is the last one related to
            // it and if no tech serv is related to it
            Set<ServiceSpecification> serviceSpecifications = endpoint.getServiceSpecifications();
            if (serviceSpecifications != null) {
                for (ServiceSpecification serviceSpecification : serviceSpecifications) {
                    serviceSpecification.getEndpoints().remove(endpoint);
                    if ((serviceSpecification.getTechnicalServices() == null || serviceSpecification
                            .getTechnicalServices().isEmpty())
                            && serviceSpecification.getEndpoints().size() == 0) {
                        this.removeServiceSpec(serviceSpecification);
                    }
                }
            }

            // Then remove endpoint
            deleteEndpoint(endpoint);
        } else {
            throw new DeploymentException("No endpoint found for the given id: " + endpointId);
        }
    }

    /**
     * @param endpointId
     */
    private void deleteEndpoint(Endpoint endpoint) {
        // Remove endpoint from service eps list to prevent resave by cascade
        endpoint.getTechnicalService().getEndpoints().remove(endpoint);

        // Then remove ep
        this.endpointDAO.remove(endpoint);
    }

    /**
     * @param serviceSpecification
     */
    private void removeServiceSpec(ServiceSpecification serviceSpecification) {
        // Delete overview docs
        deleteOverviewDocs(serviceSpecification);

        // Then delete service spec
        this.universalUnifiedDAO.remove(serviceSpecification);
    }

    private void deleteOverviewDocs(ServiceSpecification serviceSpecification) {
        List<SimpleFile> overviewDocs = serviceSpecification.getOverviewDocs();
        if (overviewDocs != null) {
            for (SimpleFile simpleFile : overviewDocs) {
                metadataService.deleteMetadata(simpleFile.getId());
            }
        }

    }

    /**
     * @param binding
     */
    private void removeBinding(Binding binding) {
        // Remove binding from service specs
        Set<ServiceSpecification> serviceSpecifications = binding.getServiceSpecifications();
        for (ServiceSpecification serviceSpecification : serviceSpecifications) {
            serviceSpecification.getBindings().remove(binding);
        }

        // Remove binding
        this.universalUnifiedDAO.remove(binding);

        // Remove interface if this binding is the last one related to it
        ServiceInterface serviceInterface = binding.getServiceInterface();
        if (serviceInterface != null) {
            Set<Binding> bindings = serviceInterface.getBindings();
            bindings.remove(binding);
            if (bindings.size() == 0) {
                this.removeInterface(serviceInterface);
            }
        }
    }

    /**
     * @param serviceInterface
     * 
     */
    private void removeInterface(ServiceInterface serviceInterface) {
        // Remove interface from service specs
        Set<ServiceSpecification> serviceSpecifications = serviceInterface.getServiceSpecs();
        for (ServiceSpecification serviceSpecification : serviceSpecifications) {
            serviceSpecification.getServiceInterfaces().remove(serviceInterface);
        }

        // Remove interface
        this.universalUnifiedDAO.remove(serviceInterface);

    }

    public void setEndpointDAO(final EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    private void toEndpointsTO(final List<EndpointTO> result, final Collection<Endpoint> endpoints) {
        if ((endpoints != null) && !endpoints.isEmpty()) {
            for (final Endpoint endpoint : endpoints) {
                EndpointTO endpointTO = toEndpointTO(endpoint, null);
                endpointTO.setGlobalRating(updateGlobalRating(endpointTO.getId()));
                result.add(endpointTO);
            }
        }
    }

    private EndpointTO toEndpointTO(final Endpoint endpoint, final String user) {
        return this.transfertObjectAssembler.toEndpointTO(endpoint, user);
    }

    @CheckAllArgumentsNotNull
    public String updateEndpoint(final EndpointTO endpointTO) throws DeploymentException {
        // Retrieve endpoint
        final Endpoint endpoint = this.endpointDAO.get(endpointTO.getId());
        if (endpoint == null) {
            throw new DeploymentException(
                    "Your are trying to update a non existing Endpoint with id: "
                            + endpointTO.getId());
        }

        // Validate
        this.validateEndpointBeforeUpdate(endpointTO);

        // update fields
        this.transfertObjectAssembler.toEndpointBO(endpointTO, endpoint);
        return this.endpointDAO.save(endpoint).getId();
    }

    public float updateGlobalRating(String endpointId) {
        Endpoint endpoint = endpointDAO.get(endpointId);
        float globalRating = 0f;
        if (!endpoint.getRatings().isEmpty()) {
            Iterator<Rating> iter = endpoint.getRatings().iterator();
            int size = endpoint.getRatings().size();
            while (iter.hasNext()) {
                globalRating = globalRating + iter.next().getMyRating();
            }
            globalRating = globalRating / size;
        }
        return globalRating;
    }

    public void updateComments(String endpointId, CommentTO comment) {
        Endpoint endpoint = endpointDAO.get(endpointId);

        if (endpoint != null) {
            final Comment commentTemp = new Comment();

            commentTemp.setContent(comment.getContent());
            commentTemp.setDate(comment.getDate());
            commentTemp.setUserName(comment.getUserName());
            commentTemp.setId(comment.getId());

            endpoint.addComment(commentTemp);
        }
    }

    public List<EndpointTO> getEndpointsByTag(String tag) {
        List<EndpointTO> result = new ArrayList<EndpointTO>();
        final List<Endpoint> allEndpoints = this.endpointDAO.getAll();
        final List<Endpoint> endpoints = new ArrayList<Endpoint>();

        Iterator<Endpoint> iter = allEndpoints.iterator();

        while (iter.hasNext()) {
            final Endpoint endpointTemp = iter.next();
            if (endpointTemp.isTagged(tag)) {
                endpoints.add(endpointTemp);
            }
        }
        toEndpointsTO(result, endpoints);
        return result;
    }

    public void removeTag(String tag, String endpointId) {
        Endpoint endpoint = this.endpointDAO.get(endpointId);
        Set<String> tags = endpoint.getTags();
        tags.remove(tag);
        endpoint.setTags(tags);
    }

    public void removeComment(String endpointId, String commentId) {
        Endpoint endpoint = this.endpointDAO.get(endpointId);
        Comment comment = this.commentDAO.get(commentId);
        List<Comment> comments = endpoint.getComments();

        comments.remove(comment);
        endpoint.setComments(comments);
    }

    @CheckAllArgumentsNotNull
    public List<KeyedRefTO> getCategoriesForEndpoint(String endpointId) throws DeploymentException {
        Endpoint endpoint = this.endpointDAO.get(endpointId);
        if (endpoint == null) {
            throw new DeploymentException("No endpoint found for the given id: " + endpointId);
        }

        return transfertObjectAssembler.toCategoriesTO(endpoint.getCategoryBag());
    }

    @SuppressWarnings("unchecked")
    @CheckAllArgumentsNotNull
    public void removeCategories(String endpointId, List<String> categoryIds)
            throws DeploymentException {
        // Retrieve endpoint
        Endpoint endpoint = endpointDAO.get(endpointId);
        if (endpoint == null) {
            throw new DeploymentException("No endpoint found for the given id: " + endpointId);
        }

        // Unlink all categories
        List<KeyedReference> keyRefs = universalUnifiedDAO
                .getAll(KeyedReference.class, categoryIds);
        // Check if category ids exist
        List<KeyedReference> keyedReferences = endpoint.getCategoryBag().getKeyedReferences();
        for (KeyedReference keyRef : keyRefs) {
            if (!keyedReferences.contains(keyRef)) {
                throw new DeploymentException("No category found for the given id: "
                        + keyRef.getId());
            }
        }
        keyedReferences.removeAll(keyRefs);

        // Delete categories
        universalUnifiedDAO.removeAll(keyRefs);
    }

    @CheckAllArgumentsNotNull
    public void addCategory(String endpointId, String categoryId, String categoryValueId)
            throws DeploymentException {
        // Retrieve endpoint
        Endpoint endpoint = endpointDAO.get(endpointId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        // Retrieve category value
        CategoryValue value = (CategoryValue) universalUnifiedDAO.get(CategoryValue.class,
                categoryValueId);

        if (endpoint != null && category != null && value != null) {
            // add category to endpoint
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(value.getDescription());
            keyedReference.setKeyValue(value.getValue());
            this.validateCategory(endpoint, keyedReference);

            CategoryBag categoryBag = endpoint.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                endpoint.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown endpoint, category system or category value");
        }

        endpointDAO.save(endpoint);
    }

    private void validateCategory(Endpoint endpoint, KeyedReference keyedReference)
            throws DeploymentException {
        CategoryBag categoryBag = endpoint.getCategoryBag();
        if (categoryBag != null) {
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            if (keyedReferences != null) {
                if (keyedReferences.contains(keyedReference)) {
                    throw new DeploymentException("Category already added to this endpoint.");
                }
            }
        }
    }

    @CheckArgumentsNotNull
    public void addCategory(String endpointId, String categoryId, String categoryValue,
            String categoryDesc) throws DeploymentException {
        // Retrieve endpoint
        Endpoint endpoint = endpointDAO.get(endpointId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        if (endpoint != null && category != null && !StringHelper.isNullOrEmpty(categoryValue)) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(categoryDesc);
            keyedReference.setKeyValue(categoryValue);
            this.validateCategory(endpoint, keyedReference);

            CategoryBag categoryBag = endpoint.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                endpoint.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown endpoint, category system or category value");
        }

        endpointDAO.save(endpoint);
    }

    private String addServiceDefFile(InputSource wsdlSource, Map<String, InputSource> imports,
            String wsdlFileName, String endpointId, boolean storeEntities)
            throws DeploymentException, TimeoutException {
        /*
         * Register WSDL file
         */
        WsdlEntitiesImportReport registeredWsdlEntities;
        try {
            registeredWsdlEntities = this.wsdlManager.registerServiceDefFile(wsdlSource, imports,
                    wsdlFileName, storeEntities, false);
        } catch (WSDLServiceException e) {
            throw new DeploymentException("Can't register the given wsdl file", e);
        }

        /*
         * Then link service spec to endpoint
         */
        ServiceSpecification serviceSpecification = this.serviceSpecificationDAO
                .get(registeredWsdlEntities.getServiceSpecId());
        Endpoint endpoint = this.endpointDAO.get(endpointId);
        serviceSpecification.addEndpoint(endpoint);
        this.serviceSpecificationDAO.save(serviceSpecification);
        return serviceSpecification.getId();
    }

    @CheckArgumentsNotNull
    public String addServiceDefFile(URI wsdlURI, String wsdlFileName, String endpointId,
            boolean storeEntities) throws DeploymentException, TimeoutException {
        InputSource inputSource;
        try {
            inputSource = XMLUtil.createWsdlSource(wsdlURI);
        } catch (Exception e) {
            throw new DeploymentException("Can't read wsdl file", e);
        }
        return this.addServiceDefFile(inputSource, null, wsdlFileName, endpointId, storeEntities);
    }

    @CheckArgumentsNotNull
    public String addServiceDefFile(Document wsdlDoc, String wsdlFileName, String endpointId,
            boolean storeEntities) throws DeploymentException, TimeoutException {
        // Create the source wsdl InputSource
        InputSource wsdlSource;
        try {
            wsdlSource = XMLUtil.createWsdlSource(wsdlDoc);
        } catch (Exception e) {
            throw new DeploymentException("Can't read wsdl file", e);
        }
        return this.addServiceDefFile(wsdlSource, null, wsdlFileName, endpointId, storeEntities);
    }

    @CheckAllArgumentsNotNull
    public void removeServiceSpec(String endpointId, String serviceSpecificationId)
            throws DeploymentException {
        // Retrieve serviceSpec
        ServiceSpecification serviceSpecification = this.serviceSpecificationDAO
                .get(serviceSpecificationId);
        if (serviceSpecification == null) {
            throw new DeploymentException(
                    "You're trying to remove a non registered service description");
        }
        // Retrieve endpoint
        Endpoint endpoint = this.endpointDAO.get(endpointId);
        if (endpoint == null) {
            throw new DeploymentException(
                    "You're trying to remove a service description from a non registered endpoint");
        }

        // unlink it from service
        serviceSpecification.removeEndpoint(endpoint);
        if (serviceSpecification.getTechnicalServices().size() == 0
                && serviceSpecification.getEndpoints().size() == 0) {
            // then if not connected to any service or ep, remove it totaly
            this.removeServiceSpec(serviceSpecification);
        }

    }

    public void setUniversalUnifiedDAO(UniversalUnifiedDAO universalUnifiedDAO) {
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

    /**
     * @param metadataService
     *            the metadataService to set
     */
    public void setMetadataService(MetadataService metadataService) {
        this.metadataService = metadataService;
    }

    /**
     * @param technicalServiceDAO
     *            the technicalServiceDAO to set
     */
    public void setTechnicalServiceDAO(TechnicalServiceDAO technicalServiceDAO) {
        this.technicalServiceDAO = technicalServiceDAO;
    }

    /**
     * @param serviceSpecificationDAO
     *            the serviceSpecificationDAO to set
     */
    public void setServiceSpecificationDAO(ServiceSpecificationDAO serviceSpecificationDAO) {
        this.serviceSpecificationDAO = serviceSpecificationDAO;
    }

    /**
     * @param wsdlManager
     *            the wsdlManager to set
     */
    public void setWsdlManager(WSDLManager wsdlManager) {
        this.wsdlManager = wsdlManager;
    }
}
