/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * TechServiceManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.deployment;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeoutException;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.deployment.EndpointManager;
import org.ow2.dragon.api.service.deployment.TechServiceManager;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.api.service.wsdl.WSDLManager;
import org.ow2.dragon.api.service.wsdl.WSDLServiceException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.deployment.CommentTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.deployment.ServiceSearchProperties;
import org.ow2.dragon.api.to.deployment.TechnicalServiceTO;
import org.ow2.dragon.api.to.metadata.SimpleFileTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Comment;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Rating;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.dao.deployment.CommentDAO;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.deployment.TechnicalServiceUnifiedDAO;
import org.ow2.dragon.persistence.dao.specification.ServiceSpecificationDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.service.wsdl.importreport.WsdlEntitiesImportReport;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.StringHelper;
import org.ow2.dragon.util.XMLUtil;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class TechServiceManagerImpl implements TechServiceManager {

    private TechnicalServiceUnifiedDAO technicalServiceUnifiedDAO;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    private MetadataService metadataService;

    private ServiceSpecificationDAO serviceSpecificationDAO;

    private EndpointDAO endpointDAO;

    private EndpointManager endpointManager;

    private WSDLManager wsdlManager;

    // Added By TIAR Abderrahmane
    private CommentDAO commentDAO;

    public void setCommentDAO(final CommentDAO commentDAO) {
        this.commentDAO = commentDAO;
    }

    // End

    private String[] createSearchProperties(List<ServiceSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(ServiceSearchProperties.NAME)) {
                propertiesList.add("fullName");
            }
            if (searchedProperties.contains(ServiceSearchProperties.CATEGORY)) {
                propertiesList.add("categoryBag.keyedReferences.keyValue");
                propertiesList.add("categoryBag.keyedReferences.keyName");
            }
            if (searchedProperties.contains(ServiceSearchProperties.PURPOSE)) {
                propertiesList.add("descriptions.description");
            }
            if (searchedProperties.contains(ServiceSearchProperties.OPERATIONS)) {
                propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.name");
                propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.purpose");
            }
            if (searchedProperties.contains(ServiceSearchProperties.PROTOCOL)) {
                propertiesList.add("endpoints.binding.protocols.name.name");
                propertiesList.add("endpoints.binding.transports.name.name");
                propertiesList.add("endpoints.binding.protocols.descriptions.description");
                propertiesList.add("endpoints.binding.transports.descriptions.description");
            }
            if (searchedProperties.contains(ServiceSearchProperties.ORGANIZATION)) {
                propertiesList.add("toLinks.type");
                propertiesList.add("toLinks.from.names.name");
            }
            if (searchedProperties.contains(ServiceSearchProperties.RELATED_DOCS)) {
                propertiesList.add("relatedDocs.author");
                propertiesList.add("relatedDocs.title");
                propertiesList.add("relatedDocs.extractedContent");
            }
        } else {
            propertiesList.add("fullName");
            propertiesList.add("categoryBag.keyedReferences.keyValue");
            propertiesList.add("categoryBag.keyedReferences.keyName");
            propertiesList.add("descriptions.description");
            propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.name");
            propertiesList.add("endpoints.binding.serviceInterface.specifiedOps.purpose");
            propertiesList.add("endpoints.binding.protocols.name.name");
            propertiesList.add("endpoints.binding.transports.name.name");
            propertiesList.add("endpoints.binding.protocols.descriptions.description");
            propertiesList.add("endpoints.binding.transports.descriptions.description");
            propertiesList.add("toLinks.type");
            propertiesList.add("toLinks.from.names.name");
            propertiesList.add("relatedDocs.author");
            propertiesList.add("relatedDocs.title");
            propertiesList.add("relatedDocs.extractedContent");
        }
        // TODO add search on person
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * createTechService
     * (org.ow2.dragon.ui.model.to.deployment.TechnicalServiceTO)
     */
    @CheckAllArgumentsNotNull
    public String createTechService(final TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        // Validate tech service before creation
        validateServiceBeforeCreation(techServiceTO);
        // create tech service bo from to
        final TechnicalService technicalServiceBO = new TechnicalService();
        this.transfertObjectAssembler.toTechServiceBO(techServiceTO, technicalServiceBO);

        // Persist new tech service
        return this.technicalServiceUnifiedDAO.save(technicalServiceBO).getId();
    }

    private void validateServiceBeforeCreation(TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        // Check if the name isn't null or empty
        if (StringHelper.isNullOrEmpty(techServiceTO.getName())) {
            throw new DeploymentException("Service name must be specified.");
        }

        // check if the service isn't already created : name must be
        // unique
        TechnicalService technicalService = this.technicalServiceUnifiedDAO
                .getTechServiceByName(techServiceTO.getName());
        if (technicalService != null) {
            throw new DeploymentException("Service already created with the same name ("
                    + techServiceTO.getName() + ")");
        }

    }

    private void validateServiceBeforeUpdate(TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        // Check if the name isn't null or empty
        if (StringHelper.isNullOrEmpty(techServiceTO.getName())) {
            throw new DeploymentException("Service name must be specified.");
        }

        // check if the service isn't already created : name must be
        // unique
        TechnicalService technicalService = this.technicalServiceUnifiedDAO
                .getTechServiceByName(techServiceTO.getName());
        if (technicalService != null && !techServiceTO.getId().equals(technicalService.getId())) {
            throw new DeploymentException("Service already created with the same name ("
                    + techServiceTO.getName() + ")");
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * getAllTechServices(org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<TechnicalServiceTO> getAllTechServices(final RequestOptionsTO requestOptionsTO) {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();
        final List<TechnicalService> techServices = this.technicalServiceUnifiedDAO
                .getAll(this.transfertObjectAssembler.toServiceRequestOptions(requestOptionsTO));
        toTechServicesTO(result, techServices);
        return result;
    }

    private void toTechServicesTO(List<TechnicalServiceTO> result,
            final Collection<TechnicalService> techServices) {
        if ((techServices != null) && !techServices.isEmpty()) {
            for (final TechnicalService techServ : techServices) {
                TechnicalServiceTO techTO = toTechServiceTO(techServ, null);
                techTO.setGlobalRating(updateGlobalRating(techTO.getId()));
                result.add(techTO);
            }
        }
    }

    private TechnicalServiceTO toTechServiceTO(final TechnicalService techServ, String user) {
        return this.transfertObjectAssembler.toTechServiceTO(techServ, user);
    }

    @CheckAllArgumentsNotNull
    public EndpointTO getEndpoint(final String techServiceName, final String endpointName)
            throws DeploymentException {

        Endpoint endpointBO = endpointDAO
                .getByNameAndServiceFullName(techServiceName, endpointName);
        if (endpointBO == null) {
            throw new DeploymentException("Can't retrieve the endpoint '" + endpointName
                    + "' which is part of technical service '" + techServiceName + "'");
        }

        return toEndpointTO(endpointBO);
    }

    private EndpointTO toEndpointTO(final Endpoint endpoint) {
        return this.transfertObjectAssembler.toEndpointTO(endpoint, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * getTechService(java.lang.String)
     */
    @CheckArgumentsNotNull
    public TechnicalServiceTO getTechService(final String techServiceId, String user)
            throws DeploymentException {
        // retrieve tech serv bo
        final TechnicalService technicalServiceBO = this.technicalServiceUnifiedDAO
                .get(techServiceId);
        if (technicalServiceBO == null) {
            throw new DeploymentException("No tech service found for the given id: "
                    + techServiceId);
        }
        // create tech serv to from bo
        final TechnicalServiceTO techServiceTO = toTechServiceTO(technicalServiceBO, user);
        return techServiceTO;
    }

    public List<TechnicalServiceTO> loadServicesFromWSDL(final String[] servicesId,
            final RequestOptionsTO requestOptionsTO) throws DeploymentException {
        // Doesn't use requestOption for the moment
        final List<TechnicalServiceTO> techServs = new ArrayList<TechnicalServiceTO>();
        if (servicesId != null) {
            for (final String servId : servicesId) {
                final TechnicalServiceTO serviceTO = this.getTechService(servId, null);
                if (serviceTO != null) {
                    techServs.add(serviceTO);
                }
            }
        }
        return techServs;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * removeTechService(java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public void removeTechService(final String techServiceId) throws DeploymentException {
        final TechnicalService technicalService = this.technicalServiceUnifiedDAO
                .get(techServiceId);
        if (technicalService != null) {
            // Remove related endpoints
            List<Endpoint> endpoints = new ArrayList<Endpoint>();
            // List copy could be removed if cascade from service to endpoint is
            // removed. Don't forget to remove the corresponding line of code in
            // EndpointManagerImpl.deleteEndpoint()
            endpoints.addAll(technicalService.getEndpoints());
            if (endpoints != null) {
                for (Endpoint endpoint : endpoints) {
                    endpointManager.removeEndpoint(endpoint.getId());
                }
            }
            // Remove service spec if this service is the last one related to it
            final Set<ServiceSpecification> serviceSpecifications = technicalService
                    .getServiceSpecifications();
            if (serviceSpecifications != null) {
                for (ServiceSpecification serviceSpecification : serviceSpecifications) {
                    serviceSpecification.getTechnicalServices().remove(technicalService);
                    if (serviceSpecification.getTechnicalServices().size() == 0) {
                        this.removeServiceSpec(serviceSpecification);
                    }
                }
            }
            // Remove related docs
            Set<SimpleFile> relatedDocs = technicalService.getRelatedDocs();
            if (relatedDocs != null) {
                for (SimpleFile simpleFile : relatedDocs) {
                    metadataService.deleteMetadata(simpleFile.getId());
                }
            }
            // then remove tech serv
            this.technicalServiceUnifiedDAO.remove(technicalService);
        } else {
            throw new DeploymentException("No tech service found fro the given id: "
                    + techServiceId);
        }
    }

    /**
     * @param serviceSpecification
     * @throws DeploymentException
     */
    @CheckAllArgumentsNotNull
    public void removeServiceSpec(String serviceId, String serviceSpecificationId)
            throws DeploymentException {
        // Retrieve serviceSpec
        ServiceSpecification serviceSpecification = this.serviceSpecificationDAO
                .get(serviceSpecificationId);
        if (serviceSpecification == null) {
            throw new DeploymentException(
                    "You're trying to remove a non registered service description");
        }
        // Retrieve service
        TechnicalService service = this.technicalServiceUnifiedDAO.get(serviceId);
        if (service == null) {
            throw new DeploymentException(
                    "You're trying to remove a service description from a non registered service");
        }

        // unlink it from service
        serviceSpecification.removeTechnicalService(service);
        if (serviceSpecification.getTechnicalServices().size() == 0
                && serviceSpecification.getEndpoints().size() == 0) {
            // then if not connected to any service or endpoint, remove it
            // totaly
            this.removeServiceSpec(serviceSpecification);
        }
    }

    /**
     * @param serviceSpecification
     */
    private void removeServiceSpec(ServiceSpecification serviceSpecification) {
        // Delete overview docs
        deleteOverviewDocs(serviceSpecification);

        // Then delete service spec
        this.serviceSpecificationDAO.remove(serviceSpecification);
    }

    private void deleteOverviewDocs(ServiceSpecification serviceSpecification) {
        List<SimpleFile> overviewDocs = serviceSpecification.getOverviewDocs();
        if (overviewDocs != null) {
            for (SimpleFile simpleFile : overviewDocs) {
                metadataService.deleteMetadata(simpleFile.getId());
            }
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * searchTechService(java.lang.String, boolean, boolean, boolean, boolean,
     * boolean, boolean, org.ow2.dragon.ui.model.to.RequestOptions)
     */
    @CheckArgumentsNotNull
    public List<TechnicalServiceTO> searchTechService(final String searchCriteria,
            final List<ServiceSearchProperties> searchedProperties, final RequestOptionsTO options)
            throws DeploymentException {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<TechnicalService> techServices;
        try {
            techServices = this.technicalServiceUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toServiceRequestOptions(options));
        } catch (DAOLayerException e) {
            throw new DeploymentException(
                    "You must specified non empty search criteria and properties.", e);
        }

        toTechServicesTO(result, techServices);
        return result;
    }

    public void setMetadataService(final MetadataService metadataService) {
        this.metadataService = metadataService;
    }

    public void setServiceSpecificationDAO(final ServiceSpecificationDAO serviceSpecificationDAO) {
        this.serviceSpecificationDAO = serviceSpecificationDAO;
    }

    public void setTechnicalServiceUnifiedDAO(
            final TechnicalServiceUnifiedDAO technicalServiceUnifiedDAO) {
        this.technicalServiceUnifiedDAO = technicalServiceUnifiedDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager#
     * updateTechService
     * (org.ow2.dragon.ui.model.to.deployment.TechnicalServiceTO)
     */
    @CheckAllArgumentsNotNull
    public String updateTechService(final TechnicalServiceTO techServiceTO)
            throws DeploymentException {
        // Retrieve tech serv
        final TechnicalService technicalService = this.technicalServiceUnifiedDAO.get(techServiceTO
                .getId());
        if (technicalService == null) {
            throw new DeploymentException(
                    "Your are trying to update a non existing Technical Service with id: "
                            + techServiceTO.getId());
        }

        // Validate
        this.validateServiceBeforeUpdate(techServiceTO);

        // update fields
        this.transfertObjectAssembler.toTechServiceBO(techServiceTO, technicalService);
        this.technicalServiceUnifiedDAO.save(technicalService);

        return technicalService.getId();
    }

    public String getRelatedDocContentAsString(String relatedDocId) throws DeploymentException {
        String result = null;
        try {
            InputStream inputStream = this.metadataService
                    .loadMetadataContentAsInputStream(relatedDocId);
            result = StringHelper.toString(inputStream);
        } catch (final MetadataServiceException e) {
            throw new DeploymentException("Can't load related doc content", e);
        } catch (final IOException e) {
            throw new DeploymentException("Can't convert related doc content to String", e);
        }
        return result;
    }

    @CheckAllArgumentsNotNull
    public InputStream getRelatedDocContent(String relatedDocId) throws DeploymentException {
        InputStream result = null;
        try {
            result = this.metadataService.loadMetadataContentAsInputStream(relatedDocId);
        } catch (final MetadataServiceException e) {
            throw new DeploymentException("Can't load related doc content", e);
        }
        return result;
    }

    public List<SimpleFileTO> getRelatedDocsForTechServ(String techServiceId)
            throws DeploymentException {
        List<SimpleFileTO> result = new ArrayList<SimpleFileTO>();

        // retrieve tech serv bo
        final TechnicalService technicalServiceBO = this.technicalServiceUnifiedDAO
                .get(techServiceId);

        Set<SimpleFile> relatedDocs = technicalServiceBO.getRelatedDocs();
        for (SimpleFile simpleFile : relatedDocs) {
            result.add(transfertObjectAssembler.toSimpleFileTO(simpleFile));
        }

        return result;
    }

    private void addRelatedDocs(String serviceId, Set<SimpleFile> relatedDocs)
            throws DeploymentException {
        if (!relatedDocs.isEmpty()) {
            TechnicalService service = technicalServiceUnifiedDAO.get(serviceId);
            if (service == null) {
                throw new DeploymentException("No tech service found for the given id : "
                        + serviceId);
            }
            for (SimpleFile relatedDoc : relatedDocs) {
                service.addRelatedDoc(relatedDoc);
            }
            technicalServiceUnifiedDAO.save(service);
        }
    }

    @CheckArgumentsNotNull
    public String registerRelatedDoc(String serviceId, String mimetype, byte[] docContent,
            String fileName) throws DeploymentException {
        SimpleFile doc = null;
        try {
            doc = metadataService.storeMetadataAndIndexContent(mimetype, docContent, fileName);
        } catch (MetadataServiceException e) {
            throw new DeploymentException("Can't store document in registry", e);
        }
        Set<SimpleFile> docs = new HashSet<SimpleFile>();
        docs.add(doc);
        addRelatedDocs(serviceId, docs);
        return doc.getId();
    }

    @CheckAllArgumentsNotNull
    public void removeRelatedDoc(String serviceId, String documentId) throws DeploymentException {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);
        SimpleFile simpleFile = metadataService.loadMetadata(documentId);
        if (technicalService == null || simpleFile == null) {
            throw new DeploymentException("Service (id:" + serviceId + ") or document (id:"
                    + documentId + ") not found");
        }
        technicalService.getRelatedDocs().remove(simpleFile);
        metadataService.deleteMetadata(documentId);
    }

    @CheckAllArgumentsNotNull
    public void addEndpoint(String endpointId, String serviceId) throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(serviceId);
        if (service == null) {
            throw new DeploymentException("Service doesn't exist for id: " + serviceId);
        }

        // Retrieve endpoint
        Endpoint endpoint = endpointDAO.get(endpointId);
        if (endpoint == null) {
            throw new DeploymentException("Endpoint doesn't exist for id: " + endpointId);
        }

        // Check if an endpoint with the same name isn't already linked to the
        // given service
        Endpoint endpoint2 = endpointDAO.getByNameAndServiceId(serviceId, endpoint.getName());
        if (endpoint2 != null) {
            throw new DeploymentException(
                    "Endpoint with the same name already linked to the given service.");
        }

        service.addEndpoint(endpoint);
        technicalServiceUnifiedDAO.save(service);
    }

    @CheckArgumentsNotNull
    public void addCategory(String techServiceId, String categoryId, String categoryValue,
            String categoryDesc) throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        if (service != null && category != null && !StringHelper.isNullOrEmpty(categoryValue)) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(categoryDesc);
            keyedReference.setKeyValue(categoryValue);
            this.validateCategory(service, keyedReference);

            CategoryBag categoryBag = service.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                service.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown service, category system or category value");
        }

        technicalServiceUnifiedDAO.save(service);
    }

    @CheckAllArgumentsNotNull
    public void addCategory(String techServiceId, String categoryId, String categoryValueId)
            throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        // Retrieve category value
        CategoryValue value = (CategoryValue) universalUnifiedDAO.get(CategoryValue.class,
                categoryValueId);

        if (service != null && category != null) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(value.getDescription());
            keyedReference.setKeyValue(value.getValue());
            this.validateCategory(service, keyedReference);

            CategoryBag categoryBag = service.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                service.setCategoryBag(categoryBag);
            }
        } else {
            throw new DeploymentException(
                    "You have specified unknown service, category system or category value");
        }

        technicalServiceUnifiedDAO.save(service);
    }

    private void validateCategory(TechnicalService techServ, KeyedReference keyedReference)
            throws DeploymentException {
        CategoryBag categoryBag = techServ.getCategoryBag();
        if (categoryBag != null) {
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            if (keyedReferences != null) {
                if (keyedReferences.contains(keyedReference)) {
                    throw new DeploymentException("Category already added to this service.");
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    @CheckAllArgumentsNotNull
    public void removeCategories(String techServiceId, List<String> categoryIds)
            throws DeploymentException {
        // Retrieve service
        TechnicalService service = technicalServiceUnifiedDAO.get(techServiceId);
        if (service == null) {
            throw new DeploymentException("No service found for the given id: " + techServiceId);
        }

        // Unlink all categories
        List<KeyedReference> keyRefs = universalUnifiedDAO
                .getAll(KeyedReference.class, categoryIds);
        // Check if category ids exist
        List<KeyedReference> keyedReferences = service.getCategoryBag().getKeyedReferences();
        for (KeyedReference keyRef : keyRefs) {
            if (!keyedReferences.contains(keyRef)) {
                throw new DeploymentException("No category found for the given id: "
                        + keyRef.getId());
            }
        }
        keyedReferences.removeAll(keyRefs);

        // Delete categories
        universalUnifiedDAO.removeAll(keyRefs);
    }

    private String addServiceDefFile(InputSource wsdlSource, Map<String, InputSource> imports,
            String wsdlFileName, String techServiceId, boolean storeEntities)
            throws DeploymentException, TimeoutException {
        /*
         * Register WSDL file
         */
        WsdlEntitiesImportReport registeredWsdlEntities;
        try {
            registeredWsdlEntities = this.wsdlManager.registerServiceDefFile(wsdlSource, imports,
                    wsdlFileName, storeEntities, false);
        } catch (WSDLServiceException e) {
            throw new DeploymentException("Can't register the given wsdl file", e);
        }

        /*
         * Then link service spec to service
         */
        ServiceSpecification serviceSpecification = this.serviceSpecificationDAO
                .get(registeredWsdlEntities.getServiceSpecId());
        TechnicalService technicalService = this.technicalServiceUnifiedDAO.get(techServiceId);
        serviceSpecification.addTechnicalService(technicalService);
        this.serviceSpecificationDAO.save(serviceSpecification);
        return serviceSpecification.getId();
    }

    @CheckArgumentsNotNull
    public String addServiceDefFile(URI wsdlURI, String wsdlFileName, String techServiceId,
            boolean storeEntities) throws DeploymentException, TimeoutException {
        InputSource inputSource;
        try {
            inputSource = XMLUtil.createWsdlSource(wsdlURI);
        } catch (Exception e) {
            throw new DeploymentException("Can't read wsdl file", e);
        }
        return this
                .addServiceDefFile(inputSource, null, wsdlFileName, techServiceId, storeEntities);
    }

    @CheckArgumentsNotNull
    public String addServiceDefFile(Document wsdlDoc, String wsdlFileName, String techServiceId,
            boolean storeEntities) throws DeploymentException, TimeoutException {
        // Create the source wsdl InputSource
        InputSource wsdlSource;
        try {
            wsdlSource = XMLUtil.createWsdlSource(wsdlDoc);
        } catch (Exception e) {
            throw new DeploymentException("Can't read wsdl file", e);
        }
        return this.addServiceDefFile(wsdlSource, null, wsdlFileName, techServiceId, storeEntities);
    }

    // Added By TIAR Abderrahmane
    public float updateGlobalRating(String serviceId) {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);
        float globalRating = 0f;
        if (!technicalService.getRatings().isEmpty()) {
            Iterator<Rating> iter = technicalService.getRatings().iterator();
            int size = technicalService.getRatings().size();
            while (iter.hasNext()) {
                globalRating = globalRating + iter.next().getMyRating();
            }
            globalRating = globalRating / size;
        }
        return globalRating;
    }

    public void updateComments(String serviceId, CommentTO comment) {
        TechnicalService technicalService = technicalServiceUnifiedDAO.get(serviceId);

        if (technicalService != null) {
            final Comment commentTemp = new Comment();

            commentTemp.setContent(comment.getContent());
            commentTemp.setDate(comment.getDate());
            commentTemp.setUserName(comment.getUserName());
            // commentTemp.setId(comment.getId()) ;

            technicalService.addComment(commentTemp);
        }
    }

    public List<TechnicalServiceTO> getTechServicesByTag(String tag) {
        List<TechnicalServiceTO> result = new ArrayList<TechnicalServiceTO>();
        final List<TechnicalService> allTechServices = this.technicalServiceUnifiedDAO.getAll();
        final List<TechnicalService> techServices = new ArrayList<TechnicalService>();

        Iterator<TechnicalService> iter = allTechServices.iterator();

        while (iter.hasNext()) {
            final TechnicalService techServiceTemp = iter.next();
            if (techServiceTemp.isTagged(tag)) {
                techServices.add(techServiceTemp);
            }
        }
        toTechServicesTO(result, techServices);
        return result;
    }

    public void removeTag(String tag, String serviceId) {
        TechnicalService techService = this.technicalServiceUnifiedDAO.get(serviceId);
        Set<String> tags = techService.getTags();
        tags.remove(tag);
        techService.setTags(tags);
    }

    public void removeComment(String serviceId, String commentId) {
        TechnicalService techService = this.technicalServiceUnifiedDAO.get(serviceId);
        Comment comment = this.commentDAO.get(commentId);
        List<Comment> comments = techService.getComments();

        comments.remove(comment);
        techService.setComments(comments);
    }

    public void setUniversalUnifiedDAO(UniversalUnifiedDAO universalUnifiedDAO) {
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

    @CheckAllArgumentsNotNull
    public List<KeyedRefTO> getCategoriesForTechServ(String techServiceId)
            throws DeploymentException {
        TechnicalService service = this.technicalServiceUnifiedDAO.get(techServiceId);
        if (service == null) {
            throw new DeploymentException("No service found for the given id: " + techServiceId);
        }

        return transfertObjectAssembler.toCategoriesTO(service.getCategoryBag());
    }

    public void setEndpointDAO(EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

    /**
     * @param endpointManager
     *            the endpointManager to set
     */
    public void setEndpointManager(EndpointManager endpointManager) {
        this.endpointManager = endpointManager;
    }

    /**
     * @param wsdlManager
     *            the wsdlManager to set
     */
    public void setWsdlManager(WSDLManager wsdlManager) {
        this.wsdlManager = wsdlManager;
    }

}
