/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * OrganizationManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.organization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.aop.annotation.CheckArgumentsNotNull;
import org.ow2.dragon.api.service.organization.OrganizationException;
import org.ow2.dragon.api.service.organization.OrganizationManager;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.SortCriteria;
import org.ow2.dragon.api.to.common.KeyedRefTO;
import org.ow2.dragon.api.to.organization.OrganizationSearchProperties;
import org.ow2.dragon.api.to.organization.OrganizationUnitTO;
import org.ow2.dragon.api.to.organization.PostTO;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.Identifier;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.ow2.dragon.persistence.dao.DAOLayerException;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.dao.organization.OrganizationUnitUnifiedDAO;
import org.ow2.dragon.persistence.dao.organization.PostDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.StringHelper;

/**
 * @author ofabre - eBM WebSourcing
 * 
 */
public class OrganizationManagerImpl implements OrganizationManager {

    private TransfertObjectAssembler transfertObjectAssembler;

    private OrganizationUnitUnifiedDAO organizationUnitUnifiedDAO;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private PostDAO postDAO;

    @CheckAllArgumentsNotNull
    public void addPost(final String organizationId, final String postId)
            throws OrganizationException {
        // Retrieve org
        final OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO
                .get(organizationId);
        if (organizationUnit == null) {
            throw new OrganizationException(
                    "You are trying to add a post to a non existing org with id: " + organizationId);
        }

        // Retrieve post
        final Post post = this.postDAO.get(postId);
        if (post == null) {
            throw new OrganizationException(
                    "You are trying to add a non existing post to an org. Post id: " + postId);
        }

        organizationUnit.addPost(post);

        this.organizationUnitUnifiedDAO.save(organizationUnit);
    }

    @CheckAllArgumentsNotNull
    public void addCategory(String orgId, String categoryId, String categoryValueId)
            throws OrganizationException {

        // Retrieve org
        OrganizationUnit org = organizationUnitUnifiedDAO.get(orgId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        // Retrieve category value
        CategoryValue value = (CategoryValue) universalUnifiedDAO.get(CategoryValue.class,
                categoryValueId);

        if (org != null && category != null && value != null) {
            // Validate category to add
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(value.getDescription());
            keyedReference.setKeyValue(value.getValue());
            this.validateCategory(org, keyedReference);

            // add category to organization
            CategoryBag categoryBag = org.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                org.setCategoryBag(categoryBag);
            }
        } else {
            throw new OrganizationException(
                    "You have specified unknown organization, category system or category value.");
        }

        organizationUnitUnifiedDAO.save(org);
    }

    private void validateCategory(OrganizationUnit org, KeyedReference keyedReference)
            throws OrganizationException {
        CategoryBag categoryBag = org.getCategoryBag();
        if (categoryBag != null) {
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            if (keyedReferences != null) {
                if (keyedReferences.contains(keyedReference)) {
                    throw new OrganizationException("Category already added to this organization.");
                }
            }
        }
    }

    @CheckArgumentsNotNull
    public void addIdentifier(String orgId, String identifierId, String identifierValue,
            String identifierDesc) throws OrganizationException {
        // Retrieve org
        OrganizationUnit org = organizationUnitUnifiedDAO.get(orgId);

        // Retrieve category
        Identifier identifier = (Identifier) universalUnifiedDAO
                .get(Identifier.class, identifierId);

        if (org != null && identifier != null && !StringHelper.isNullOrEmpty(identifierValue)) {
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(identifier);
            keyedReference.setKeyName(identifierDesc);
            keyedReference.setKeyValue(identifierValue);
            // Validate identifier to add
            this.validateIdentifier(org, keyedReference);

            // add category to service
            List<KeyedReference> identifierBag = org.getIdentifierBag();
            if (identifierBag != null) {
                identifierBag.add(keyedReference);
            } else {
                identifierBag = new ArrayList<KeyedReference>();
                identifierBag.add(keyedReference);
                org.setIdentifierBag(identifierBag);
            }
        } else {
            throw new OrganizationException(
                    "You have specified unknown organization, identifier system or identifier value");
        }

        organizationUnitUnifiedDAO.save(org);
    }

    private void validateIdentifier(OrganizationUnit org, KeyedReference keyedReference)
            throws OrganizationException {
        List<KeyedReference> identifierBag = org.getIdentifierBag();
        if (identifierBag != null) {
            if (identifierBag.contains(keyedReference)) {
                throw new OrganizationException("Identifier already added to this organization.");
            }
        }
    }

    @CheckArgumentsNotNull
    public void addCategory(String orgId, String categoryId, String categoryValue,
            String categoryDesc) throws OrganizationException {
        // Retrieve org
        OrganizationUnit org = organizationUnitUnifiedDAO.get(orgId);

        // Retrieve category
        Category category = (Category) universalUnifiedDAO.get(Category.class, categoryId);

        if (org != null && category != null && !StringHelper.isNullOrEmpty(categoryValue)) {
            // add category to service
            KeyedReference keyedReference = new KeyedReference();
            keyedReference.setTmodel(category);
            keyedReference.setKeyName(categoryDesc);
            keyedReference.setKeyValue(categoryValue);
            this.validateCategory(org, keyedReference);

            CategoryBag categoryBag = org.getCategoryBag();
            if (categoryBag != null) {
                categoryBag.addKeyedReference(keyedReference);
            } else {
                categoryBag = new CategoryBag();
                categoryBag.addKeyedReference(keyedReference);
                org.setCategoryBag(categoryBag);
            }
        } else {
            throw new OrganizationException(
                    "You have specified unknown organization, category system or category value");
        }

        organizationUnitUnifiedDAO.save(org);
    }

    @CheckAllArgumentsNotNull
    public String createOrganization(final OrganizationUnitTO organizationUnitTO)
            throws OrganizationException {
        // Validate Organization before creation
        validateOrganizationBeforeCreation(organizationUnitTO);
        // create organization unit bo from to
        final OrganizationUnit organizationUnitBO = new OrganizationUnit();
        this.transfertObjectAssembler.toOrganizationUnitBO(organizationUnitTO, organizationUnitBO);
        // Persist new Organization Unit
        return this.organizationUnitUnifiedDAO.save(organizationUnitBO).getId();
    }

    private void validateOrganizationBeforeCreation(OrganizationUnitTO organizationUnitTO)
            throws OrganizationException {
        // Check if the name and city aren't null or empty
        if (StringHelper.isNullOrEmpty(organizationUnitTO.getName())
                || StringHelper.isNullOrEmpty(organizationUnitTO.getCity())) {
            throw new OrganizationException("Organization Unit name and city must be specified.");
        }

        // check if the organization isn't already created : name/city must be
        // unique
        OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO
                .getOrgUnitByNameAndCity(organizationUnitTO.getName(), organizationUnitTO.getCity());
        if (organizationUnit != null) {
            throw new OrganizationException(
                    "Organization Unit already created with the same name ("
                            + organizationUnitTO.getName() + ") and city ("
                            + organizationUnitTO.getCity() + ")");
        }

    }

    private String[] createSearchProperties(List<OrganizationSearchProperties> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains(OrganizationSearchProperties.NAME)) {
                propertiesList.add("names.name");
            }
            if (searchedProperties.contains(OrganizationSearchProperties.TYPE)) {
                propertiesList.add("type");
            }
            if (searchedProperties.contains(OrganizationSearchProperties.LOCALIZATION)) {
                propertiesList.add("addresses.addressLines.addressLine");
            }
            if (searchedProperties.contains(OrganizationSearchProperties.CATEGORY)) {
                propertiesList.add("categoryBag.keyedReferences.keyValue");
                propertiesList.add("categoryBag.keyedReferences.keyName");
            }
        } else {
            // Search on all properties
            propertiesList.add("names.name");
            propertiesList.add("type");
            propertiesList.add("addresses.addressLines.addressLine");
            propertiesList.add("categoryBag.keyedReferences.keyValue");
            propertiesList.add("categoryBag.keyedReferences.keyName");
        }
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.OrganizationManager#
     * getAllOrganizations(org.ow2.dragon.ui.model.dataaccess.RequestOptions)
     */
    public List<OrganizationUnitTO> getAllOrganizations(final RequestOptionsTO requestOptionsTO) {
        final List<OrganizationUnitTO> result = new ArrayList<OrganizationUnitTO>();
        final List<OrganizationUnit> organizations = this.organizationUnitUnifiedDAO
                .getAll(this.transfertObjectAssembler.toPartyRequestOptions(requestOptionsTO));
        this.toOrgUnitsTO(result, organizations);
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.OrganizationManager#
     * getOrganization(java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public OrganizationUnitTO getOrganization(final String organizationId)
            throws OrganizationException {
        // retrieve organization unit bo
        final OrganizationUnit organizationUnitBO = this.organizationUnitUnifiedDAO
                .get(organizationId);
        if (organizationUnitBO == null) {
            throw new OrganizationException("No organization found for the given id: "
                    + organizationId);
        }
        // create organization unit to from bo
        final OrganizationUnitTO organizationUnitTO = this.toOrgUnitTO(organizationUnitBO);
        return organizationUnitTO;
    }

    @CheckAllArgumentsNotNull
    public List<OrganizationUnitTO> getAllOrgsWithoutMeAndMyChildren(final String organizationId)
            throws OrganizationException {
        final List<OrganizationUnitTO> result = new ArrayList<OrganizationUnitTO>();
        List<OrganizationUnit> organizations;
        try {
            organizations = this.organizationUnitUnifiedDAO
                    .getAllWithoutMeAndMyChildren(organizationId);
        } catch (DAOLayerException e) {
            throw new OrganizationException("No organization found for the given id: "
                    + organizationId, e);
        }
        this.toOrgUnitsTO(result, organizations);
        return result;
    }

    public GenericUnifiedDAO<OrganizationUnit, String> getOrganizationUnitDAO() {
        return this.organizationUnitUnifiedDAO;
    }

    @CheckAllArgumentsNotNull
    public List<KeyedRefTO> getCategoriesForOrg(String orgId) throws OrganizationException {
        OrganizationUnit org = this.organizationUnitUnifiedDAO.get(orgId);
        if (org == null) {
            throw new OrganizationException("No organization found for the given id: " + orgId);
        }

        return transfertObjectAssembler.toCategoriesTO(org.getCategoryBag());
    }

    @CheckAllArgumentsNotNull
    @SuppressWarnings("unchecked")
    public void removeCategories(String orgId, List<String> categoryIds)
            throws OrganizationException {
        if (categoryIds != null) {
            // Retrieve org
            OrganizationUnit org = organizationUnitUnifiedDAO.get(orgId);
            if (org == null) {
                throw new OrganizationException("No organization found for the given id: " + orgId);
            }

            // Unlink all categories
            List<KeyedReference> keyRefs = universalUnifiedDAO.getAll(KeyedReference.class,
                    categoryIds);
            org.getCategoryBag().getKeyedReferences().removeAll(keyRefs);

            // Delete categories
            universalUnifiedDAO.removeAll(keyRefs);
        }
    }

    @CheckAllArgumentsNotNull
    public List<KeyedRefTO> getIdentifiersForOrg(String orgId) throws OrganizationException {
        OrganizationUnit org = this.organizationUnitUnifiedDAO.get(orgId);
        if (org == null) {
            throw new OrganizationException("No organization found for the given id: " + orgId);
        }

        return transfertObjectAssembler.toIdentifiersTO(org.getIdentifierBag());
    }

    @CheckAllArgumentsNotNull
    @SuppressWarnings("unchecked")
    public void removeIdentifiers(String orgId, List<String> identifierIds)
            throws OrganizationException {
        if (identifierIds != null) {
            // Retrieve org
            OrganizationUnit org = organizationUnitUnifiedDAO.get(orgId);
            if (org == null) {
                throw new OrganizationException("No organization found for the given id: " + orgId);
            }

            // Unlink all identifiers
            List<KeyedReference> keyRefs = universalUnifiedDAO.getAll(KeyedReference.class,
                    identifierIds);
            org.getIdentifierBag().removeAll(keyRefs);

            // Delete categories
            universalUnifiedDAO.removeAll(keyRefs);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.OrganizationManager#
     * removeOrganization(java.lang.String)
     */
    @CheckAllArgumentsNotNull
    public void removeOrganization(final String organizationId) {
        this.organizationUnitUnifiedDAO.remove(organizationId);
    }

    @CheckAllArgumentsNotNull
    public void removePost(final String organizationId, final String postId)
            throws OrganizationException {
        final OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO
                .get(organizationId);
        final Post post = this.postDAO.get(postId);
        if (organizationUnit == null) {
            throw new OrganizationException("Organization doesn't exist for the given id: "
                    + organizationId);
        }
        organizationUnit.removePost(post);
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.ow2.dragon.ui.businessdelegate.organization.OrganizationManager#
     * searchOrganization(java.lang.String, boolean, boolean, boolean, boolean,
     * org.ow2.dragon.ui.model.dataaccess.RequestOptions)
     */
    @CheckArgumentsNotNull
    public List<OrganizationUnitTO> searchOrganization(final String searchCriteria,
            List<OrganizationSearchProperties> searchedProperties,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        final List<OrganizationUnitTO> result = new ArrayList<OrganizationUnitTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<OrganizationUnit> organizations;
        try {
            organizations = this.organizationUnitUnifiedDAO.searchORMResult(criteria, properties,
                    transfertObjectAssembler.toPartyRequestOptions(requestOptionsTO));
        } catch (DAOLayerException e) {
            throw new OrganizationException(
                    "You must specified non empty search criteria and properties.", e);
        }

        // Create result array
        this.toOrgUnitsTO(result, organizations);
        return result;
    }

    public void setOrganizationUnitUnifiedDAO(
            final OrganizationUnitUnifiedDAO organizationUnitUnifiedDAO) {
        this.organizationUnitUnifiedDAO = organizationUnitUnifiedDAO;
    }

    public void setPostDAO(final PostDAO postDAO) {
        this.postDAO = postDAO;
    }

    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    private void toOrgUnitsTO(final List<OrganizationUnitTO> result,
            final Collection<OrganizationUnit> organizations) {
        if ((organizations != null) && !organizations.isEmpty()) {
            for (final OrganizationUnit organizationUnit : organizations) {
                result.add(this.toOrgUnitTO(organizationUnit));
            }
        }
    }

    private OrganizationUnitTO toOrgUnitTO(final OrganizationUnit organizationUnit) {
        return this.transfertObjectAssembler.toOrganizationUnitTO(organizationUnit);
    }

    @CheckArgumentsNotNull
    public List<PostTO> getPostsByOrganization(final String orgId,
            final RequestOptionsTO requestOptionsTO) throws OrganizationException {
        final List<PostTO> result = new ArrayList<PostTO>();

        // search org
        final OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO.get(orgId);
        if (organizationUnit == null) {
            throw new OrganizationException("No organization found for the given id: " + orgId);
        }

        // Search for post in org
        Set<Post> posts = organizationUnit.getPosts();

        List<Post> postList = new ArrayList<Post>(posts);
        if (requestOptionsTO != null && requestOptionsTO.hasSortOption()) {
            final SortCriteria sortProp = requestOptionsTO.getSortCriteria();
            Collections.sort(postList, new Comparator<Post>() {
                public int compare(Post o1, Post o2) {
                    int result = 0;
                    if (SortCriteria.POST_NAME.equals(sortProp)) {
                        result = o1.getName().compareTo(o2.getName());
                    }
                    // TODO implement for other post properties
                    // they are not mandatory so be careful of nullity
                    return result;
                }
            });
            if (!requestOptionsTO.isSortAscendingly()) {
                Collections.reverse(postList);
            }
        }
        this.toPostsTO(result, postList);
        return result;
    }

    private PostTO toPostTO(final Post post) {
        return this.transfertObjectAssembler.toPostTO(post);
    }

    private void toPostsTO(final List<PostTO> result, final Collection<Post> posts) {
        if ((posts != null) && !posts.isEmpty()) {
            for (final Post post : posts) {
                result.add(this.toPostTO(post));
            }
        }
    }

    @CheckAllArgumentsNotNull
    public String updateOrganization(final OrganizationUnitTO organizationUnitTO)
            throws OrganizationException {
        // Retrieve organization unit
        String orgId = organizationUnitTO.getId();
        if (orgId == null) {
            throw new NullPointerException("Organization to update must have an id.");
        }
        final OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO.get(orgId);
        if (organizationUnit == null) {
            throw new OrganizationException(
                    "Your are trying to update a non existing Organization with id: " + orgId);
        }

        // Validate Organization before update
        this.validateOrganizationBeforeUpdate(organizationUnitTO);

        // update fields
        this.transfertObjectAssembler.toOrganizationUnitBO(organizationUnitTO, organizationUnit);
        this.organizationUnitUnifiedDAO.save(organizationUnit);

        return organizationUnit.getId();
    }

    private void validateOrganizationBeforeUpdate(OrganizationUnitTO organizationUnitTO)
            throws OrganizationException {
        // Check if the name and city aren't null or empty
        if (StringHelper.isNullOrEmpty(organizationUnitTO.getName())
                || StringHelper.isNullOrEmpty(organizationUnitTO.getCity())) {
            throw new OrganizationException("Organization Unit name and city must be specified.");
        }

        // check if the organization isn't already created : name/city must be
        // unique
        OrganizationUnit organizationUnit = this.organizationUnitUnifiedDAO
                .getOrgUnitByNameAndCity(organizationUnitTO.getName(), organizationUnitTO.getCity());
        if (organizationUnit != null
                && !organizationUnit.getId().equals(organizationUnitTO.getId())) {
            throw new OrganizationException("Organization Unit already exists with the same name ("
                    + organizationUnitTO.getName() + ") and city (" + organizationUnitTO.getCity()
                    + ")");
        }

        // Check if the selected mother org isn't me or one of my children
        if (organizationUnitTO.getMotherOrganization() != null) {
            if (organizationUnitTO.getMotherOrganization().getId() != null) {
                List<OrganizationUnit> meAndMyChildren = this.organizationUnitUnifiedDAO
                        .getOrgAndChildren(organizationUnitTO.getId());
                List<String> meAndMyChildrenIds = new ArrayList<String>();
                for (OrganizationUnit organizationUnit2 : meAndMyChildren) {
                    meAndMyChildrenIds.add(organizationUnit2.getId());
                }
                if (meAndMyChildrenIds.contains(organizationUnitTO.getMotherOrganization().getId())) {
                    throw new OrganizationException("You are not allowed to choose you or "
                            + "one of your sub organization as mother organization");
                }
            }
        }
    }

    public void setUniversalUnifiedDAO(UniversalUnifiedDAO universalUnifiedDAO) {
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

}
