/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * SLAImporterImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.sla;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.ow2.dragon.aop.annotation.CheckAllArgumentsNotNull;
import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.deployment.TechServiceManager;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.api.service.organization.RoleOfPartyManager;
import org.ow2.dragon.api.service.sla.SLAManager;
import org.ow2.dragon.api.service.sla.SLAServiceException;
import org.ow2.dragon.api.to.RequestOptionsTO;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.organization.PersonToEndpointTO;
import org.ow2.dragon.api.to.sla.ManagedAgreementTO;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.organization.Party;
import org.ow2.dragon.persistence.bo.sla.ManagedAgreement;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.organization.PartyDAO;
import org.ow2.dragon.persistence.dao.sla.ManagedAgreementDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.dragon.util.ContentType;
import org.ow2.dragon.util.InputStreamUtil;
import org.ow2.dragon.util.SearchHelper;
import org.ow2.dragon.util.XMLUtil;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.ebmwebsourcing.addressing.addressing4agreement.monitoring.api.EndpointReferenceType;
import com.ebmwebsourcing.agreement.definition.WSAgreementFactory;
import com.ebmwebsourcing.agreement.definition.api.Agreement;
import com.ebmwebsourcing.agreement.definition.api.AgreementLifeCycle;
import com.ebmwebsourcing.agreement.definition.api.WSAgreementException;
import com.ebmwebsourcing.wsstar.addressing.definition.api.WSAddressingException;
import com.ebmwebsourcing.wsstar.dm.api.WSDMException;

/**
 * @author nsalatge - eBM WebSourcing
 */
public class SLAManagerImpl implements SLAManager {

    private MetadataService metadataService;

    private ManagedAgreementDAO managedAgreementDAO;

    private SLATransfertObjectAssembler slatransfertObjectAssembler;

    private TransfertObjectAssembler transfertObjectAssembler;

    private TechServiceManager techServiceManager;

    private RoleOfPartyManager roleOfPartyManager;

    private PartyDAO partyDAO;

    private EndpointDAO endpointDAO;

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.ui.businessdelegate.sla.SLAManager#createAgreement(org .ow2.dragon.ui.model.to.sla.AgreementTO)
     */
    public String createManagedAgreement(final ManagedAgreementTO managedAgreementTO) throws SLAServiceException {
        String res = null;
        try {
            final ManagedAgreement a = this.slatransfertObjectAssembler.toManagedAgreementBO(managedAgreementTO);

            final String buffer = WSAgreementFactory.newInstance().newWSAgreementWriter().write(managedAgreementTO.getAgreement());

            final SimpleFile agreementFile = this.metadataService.storeMetadata(ContentType.XML, buffer.getBytes());
            a.setAgreementFile(agreementFile);

            res = this.managedAgreementDAO.save(a).getId();

        } catch (final WSAgreementException e) {
            throw new SLAServiceException(e);
        } catch (final MetadataServiceException e) {
            throw new SLAServiceException(e);
        }
        return res;
    }


    private String[] createSearchProperties(final List<String> searchedProperties) {
        final List<String> propertiesList = new ArrayList<String>();
        if (searchedProperties != null && !searchedProperties.isEmpty()) {
            if (searchedProperties.contains("name")) {
                propertiesList.add("name");
            }
            if (searchedProperties.contains("initiator")) {
                propertiesList.add("initiator");
            }
            if (searchedProperties.contains("responder")) {
                propertiesList.add("responder");
            }
        } else {
            // search on all properties
            propertiesList.add("name");
            propertiesList.add("initiator");
            propertiesList.add("responder");
        }
        return propertiesList.toArray(new String[0]);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.ui.businessdelegate.sla.SLAManager#getAllAgreements(org .ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<ManagedAgreementTO> getAllManagedAgreements(final RequestOptionsTO requestOptionsTO) throws SLAServiceException {
        final List<ManagedAgreementTO> result = new ArrayList<ManagedAgreementTO>();
        final List<ManagedAgreement> agreements = this.managedAgreementDAO.getAll(this.transfertObjectAssembler.toAgreementRequestOptions(requestOptionsTO));
        try {
            if (agreements != null && !agreements.isEmpty()) {
                for (final ManagedAgreement agreement : agreements) {
                    final ManagedAgreementTO maTO = this.slatransfertObjectAssembler.toManagedAgreementTO(agreement);

                    // load file
                    final InputStream inputStream = this.metadataService.loadMetadataContentAsInputStream(agreement.getAgreementFile().getId());
                    final Agreement a = WSAgreementFactory.newInstance().newWSAgreementReader().read(new InputSource(inputStream));
                    maTO.setAgreement(a);

                    if (a instanceof AgreementLifeCycle) {
                        ((AgreementLifeCycle) a).setState(AgreementLifeCycle.State.fromValue(maTO.getState()));
                    }

                    result.add(maTO);
                }
            }
        } catch (final WSAgreementException e) {
            throw new SLAServiceException("Cannot convert agreement to agreement TO: " + e.getMessage(), e);
        } catch (final MetadataServiceException e) {
            throw new SLAServiceException("Cannot convert agreement to agreement TO: " + e.getMessage(), e);
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.ui.businessdelegate.sla.SLAManager#getAgreement(java.lang .Long)
     */
    @CheckAllArgumentsNotNull
    public ManagedAgreementTO getManagedAgreement(final String id) throws SLAServiceException {
        ManagedAgreementTO agreementTO = null;
        final ManagedAgreement ag = this.managedAgreementDAO.get(id);

        if (ag == null) {
            throw new SLAServiceException("Impossible to find agreement with id: " + id);
        }

        try {

            // create tech serv to from bo
            agreementTO = this.slatransfertObjectAssembler.toManagedAgreementTO(ag);


            if (agreementTO.getAgreement() instanceof AgreementLifeCycle) {
                ((AgreementLifeCycle) agreementTO.getAgreement()).setState(AgreementLifeCycle.State.fromValue(agreementTO.getState()));
            }

        } catch (final WSAgreementException e) {
            throw new SLAServiceException("Can't parse sla agreement: " + e.getMessage(), e);
        } 

        return agreementTO;
    }

    /**
     * @return the managedAgreementDAO
     */
    public ManagedAgreementDAO getManagedAgreementDAO() {
        return this.managedAgreementDAO;
    }

    public SLATransfertObjectAssembler getSlatransfertObjectAssembler() {
        return this.slatransfertObjectAssembler;
    }

    /**
     * @return the techServiceManager
     */
    public TechServiceManager getTechServiceManager() {
        return this.techServiceManager;
    }

    /**
     * @return the transfertObjectAssembler
     */
    public TransfertObjectAssembler getTransfertObjectAssembler() {
        return this.transfertObjectAssembler;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.service.sla.SLAManager#importServiceDefFile(org.w3c.dom .Document)
     */
    public String importAgreementDefFile(final Document slaContent) throws SLAServiceException {
        String storedSLAId = null;
        try {

            ManagedAgreement managedAgreement = null;
            final Agreement agreement = WSAgreementFactory.newInstance().newWSAgreementReader().read(slaContent);

            if (this.isEditable(agreement)) {
                managedAgreement = new ManagedAgreement(true);
            } else {
                managedAgreement = new ManagedAgreement(false);
            }
            if(agreement.getContext() != null && agreement.getContext().getResponder() != null) {
                String endpointRef = agreement.getContext().getResponder().getAddress();
                String[] addr = endpointRef.split("::");
                if(addr == null || addr.length != 2) {
                    throw new WSDMException("Endpoint address is not correctly formatted => {ns}ServiceName::EndpointName and not: " + endpointRef);
                }
                QName serviceName = QName.valueOf(addr[0].trim());
                String endpointName = addr[1].trim();

                final EndpointTO endpoint = techServiceManager.getEndpoint(serviceName.toString(), endpointName);     
                if(endpoint == null) {
                    throw new WSDMException("Impossible to find the endpoint responder corresponding to " + endpointRef);
                }

                List<PersonToEndpointTO> persons = this.roleOfPartyManager.getPersonRolesByEndpoint(endpoint.getId(), null);
                List<PersonToEndpointTO> managers = new ArrayList<PersonToEndpointTO>();

                for(PersonToEndpointTO person: persons) {
                    if(person.getType().equals("Manager")) {
                        managers.add(person);
                    }
                }

                if(managers.size() > 1) {
                    throw new WSDMException("Impossible to affect agreement because there are several managers for this endpoint: " + endpointRef);
                }

                if(managers.size() == 1) {
                    Party party = this.partyDAO.get(managers.get(0).getPartyTO().getId());
                    if(party == null) {
                        throw new WSDMException("Impossible to find the party corresponding to this id:  " + managers.get(0).getPartyTO().getId());
                    }
                    managedAgreement.setProvider(party);
                }

                managedAgreement.setInitiator(agreement.getContext().getInitiator().getAddress());
                managedAgreement.setResponder(agreement.getContext().getResponder().getAddress());

                Endpoint ep = this.endpointDAO.get(endpoint.getId());
                if(ep == null) {
                    throw new WSDMException("Impossible to find the endpoint corresponding to this id:  " + endpoint.getId());
                }
                managedAgreement.setEndpoint(ep);
            }



            /*
             * Store SLA file in persistent storage
             */
            InputSource agreementSource;
            try {
                agreementSource = XMLUtil.createWsdlSource(slaContent);
            } catch (Exception e1) {
                throw new SLAServiceException("Can't read wsdl file", e1);
            }
            final SimpleFile agreementFile = this.metadataService.storeMetadata(ContentType.XML, InputStreamUtil
                    .getBytes(agreementSource.getByteStream()), slaContent.getDocumentURI());


            managedAgreement.setAgreementFile(agreementFile);

            /*
             * persist agreement
             */
            storedSLAId = this.managedAgreementDAO.save(managedAgreement).getId();


        } catch (final MetadataServiceException e) {
            throw new SLAServiceException("Can't register sla file to persistent storage: " + e.getMessage(), e);
        } catch (final WSAgreementException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (WSAddressingException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (WSDMException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (DeploymentException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (IOException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } 

        return storedSLAId;
    }


    public void setPartyDAO(PartyDAO partyDAO) {
        this.partyDAO = partyDAO;
    }


    /*
     * (non-Javadoc) see org.ow2.dragon.service.sla.SLAManager#importServiceDefFile(java.lang.String )
     */
    public String importAgreementDefFile(final URI slaURI) throws SLAServiceException {
        String storedSLAId = null;
        Document slaContent = null;
        try {
            slaContent = XMLUtil.createDocument(InputStreamUtil
                    .getInputStream(slaURI));
            slaContent.setDocumentURI(slaURI.toString());
        } catch (SAXException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (IOException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        } catch (URISyntaxException e) {
            throw new SLAServiceException("Can't create sla agreement: " + e.getMessage(), e);
        }
        return this.importAgreementDefFile(slaContent);
    }

    /**
     * @param ag
     * @throws AddressingException
     * @throws SLAServiceException
     */
    private boolean isEditable(final Agreement ag) throws SLAServiceException {
        boolean res = true;
        try {
            if (ag.getContext() != null) {

                if (!(ag.getContext().getInitiator() instanceof EndpointReferenceType)) {
                    res = false;
                }

                if (!(ag.getContext().getResponder() instanceof EndpointReferenceType)) {
                    res = false;
                }
            }
        } catch (final WSAddressingException e) {
            throw new SLAServiceException(e);
        }

        if (res) {
            // for (final ExactlyOneOrOneOrMoreOrAllItem item :
            // mag.getAgreement().getTerms().getAll()
            // .getExactlyOneOrOneOrMoreOrAllItems()) {
            // if (item.getItemServiceDescriptionTerm() != null) {
            // if ((!(item.getItemServiceDescriptionTerm().getAny() instanceof
            // ReferenceProperties))
            // && (!(item.getItemServiceDescriptionTerm().getAny() instanceof
            // JAXBElement) && (((JAXBElement) item
            // .getItemServiceDescriptionTerm().getAny()).getValue() instanceof
            // ReferenceProperties))) {
            // return false;
            // }
            // }
            // if (item.getItemServiceReference() != null) {
            // if ((!(item.getItemServiceReference().getAny() instanceof
            // EndpointReferenceType))
            // && (!(item.getItemServiceReference().getAny() instanceof
            // JAXBElement) && (((JAXBElement) item
            // .getItemServiceReference().getAny()).getValue() instanceof
            // EndpointReferenceType))) {
            // {
            // return false;
            // }
            // }
            // if (item.getItemGuaranteeTerm() != null) {
            // if (!(item.getItemGuaranteeTerm().getQualifyingCondition()
            // instanceof TExpression)) {
            // return false;
            // }
            // if
            // (!(item.getItemGuaranteeTerm().getServiceLevelObjective().getKPITarget()
            // .getTarget() instanceof TExpression)) {
            // return false;
            // }
            // for (final CompensationType c : item.getItemGuaranteeTerm()
            // .getBusinessValueList().getPenalty()) {
            // if (!(c.getValueExpression() instanceof TExpression)) {
            // return false;
            // }
            // }
            // for (final CompensationType c : item.getItemGuaranteeTerm()
            // .getBusinessValueList().getReward()) {
            // if (!(c.getValueExpression() instanceof TExpression)) {
            // return false;
            // }
            // }
            // }
            // }
            // }
        }
        return res;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.ui.businessdelegate.sla.SLAManager#removeAgreement(java .lang.Long)
     */
    public void removeManagedAgreement(final String id) throws SLAServiceException {
        this.managedAgreementDAO.remove(id);
    }

    /*
     * (non-Javadoc)
     * @seeorg.ow2.dragon.ui.businessdelegate.deployment.TechServiceManager# searchTechService(java.lang.String, boolean, boolean, boolean, boolean, boolean, boolean, org.ow2.dragon.ui.model.to.RequestOptions)
     */
    public List<ManagedAgreementTO> searchManagedAgreement(final String searchCriteria, final List<String> searchedProperties, final RequestOptionsTO options) throws SLAServiceException {
        final List<ManagedAgreementTO> result = new ArrayList<ManagedAgreementTO>();

        // Split searchCriteria
        final String[] criteria = SearchHelper.splitSearchCriteria(searchCriteria);

        // Create search properties
        final String[] properties = this.createSearchProperties(searchedProperties);

        // Search for bo
        List<ManagedAgreement> agreementsBO;
        agreementsBO = this.managedAgreementDAO.searchEquals(criteria, properties, this.transfertObjectAssembler.toAgreementRequestOptions(options));

        try {
            // Create result array
            if (agreementsBO != null && !agreementsBO.isEmpty()) {
                for (final ManagedAgreement managedAgreement : agreementsBO) {
                    final ManagedAgreementTO mabTO = this.slatransfertObjectAssembler.toManagedAgreementTO(managedAgreement);

                    // load file
                    final InputStream inputStream = this.metadataService.loadMetadataContentAsInputStream(managedAgreement.getAgreementFile().getId());
                    final Agreement a = WSAgreementFactory.newInstance().newWSAgreementReader().read(new InputSource(inputStream));
                    mabTO.setAgreement(a);

                    if (a instanceof AgreementLifeCycle) {
                        ((AgreementLifeCycle) a).setState(AgreementLifeCycle.State.fromValue(mabTO.getState()));
                    }

                    result.add(mabTO);
                }
            }
        } catch (final WSAgreementException e) {
            throw new SLAServiceException(e);
        } catch (final MetadataServiceException e) {
            throw new SLAServiceException(e);
        }
        return result;
    }

    /**
     * @param managedAgreementDAO
     *            the managedAgreementDAO to set
     */
    public void setManagedAgreementDAO(final ManagedAgreementDAO managedAgreementDAO) {
        this.managedAgreementDAO = managedAgreementDAO;
    }

    public void setMetadataService(final MetadataService metadataService) {
        this.metadataService = metadataService;
    }

    public void setSlatransfertObjectAssembler(final SLATransfertObjectAssembler slatransfertObjectAssembler) {
        this.slatransfertObjectAssembler = slatransfertObjectAssembler;
    }

    /**
     * @param techServiceManager
     *            the techServiceManager to set
     */
    public void setTechServiceManager(final TechServiceManager techServiceManager) {
        this.techServiceManager = techServiceManager;
    }

    /**
     * @param transfertObjectAssembler
     *            the transfertObjectAssembler to set
     */
    public void setTransfertObjectAssembler(final TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    public void setRoleOfPartyManager(RoleOfPartyManager roleOfPartyManager) {
        this.roleOfPartyManager = roleOfPartyManager;
    }

    public void setEndpointDAO(EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.dragon.ui.businessdelegate.sla.SLAManager#updateAgreement(org .ow2.dragon.ui.model.to.agreement.AgreementTO)
     */
    public String updateManagedAgreement(final ManagedAgreementTO managedAgreementTO) throws SLAServiceException {
        // Retrieve agreeement
        final ManagedAgreement managedAgreement = this.managedAgreementDAO.get(managedAgreementTO.getId());
        if (managedAgreement == null) {
            throw new SLAServiceException("Your are trying to update a non existing agreement with id: " + managedAgreementTO.getId());
        }

        try {
            // update fields
            this.slatransfertObjectAssembler.toManagedAgreementBO(managedAgreementTO, managedAgreement);
            this.managedAgreementDAO.save(managedAgreement);

            // save file
            final String buffer = WSAgreementFactory.newInstance().newWSAgreementWriter().write(managedAgreementTO.getAgreement());

            // Remove old file
            this.metadataService.deleteMetadata(managedAgreement.getAgreementFile().getId());

            // Write new one
            final SimpleFile agreementFile = this.metadataService.storeMetadata(ContentType.XML, buffer.getBytes());
            managedAgreement.setAgreementFile(agreementFile);

        } catch (final WSAgreementException e) {
            throw new SLAServiceException(e);
        } catch (final MetadataServiceException e) {
            throw new SLAServiceException(e);
        }

        return managedAgreement.getId();
    }

    public MetadataService getMetadataService() {
        return this.metadataService;
    }


    @Override
    public List<ManagedAgreementTO> getAllManagedAgreementsFromServiceId(String serviceId)
    throws SLAServiceException {
        List<ManagedAgreementTO> res = new ArrayList<ManagedAgreementTO>();
        List<Endpoint> endpoints = this.endpointDAO.getAllEndpoints(serviceId);
        for(Endpoint ep: endpoints) {
            res.addAll(getAllManagedAgreementsFromEndpointId(ep.getId()));
        }
        return res;
    }


    @Override
    public List<ManagedAgreementTO> getAllManagedAgreementsFromEndpointId(String endpointId)
    throws SLAServiceException {
        List<ManagedAgreementTO> res = new ArrayList<ManagedAgreementTO>();
        try {
            List<ManagedAgreement> managedAgreements = new ArrayList<ManagedAgreement>();
            managedAgreements.addAll(this.managedAgreementDAO.getAllAgreementsFromEndpoint(endpointId));
            for(ManagedAgreement ag: managedAgreements) {
                res.add(this.slatransfertObjectAssembler.toManagedAgreementTO(ag));
            }
        } catch (WSAgreementException e) {
            throw new SLAServiceException(e);
        }
        return res;
    }

}
