/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * NotificationManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.wsdm;

import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.namespace.QName;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.wsdm.NotificationManager;
import org.ow2.dragon.api.to.wsdm.QoSMetricTO;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.BindingOperation;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.wsdm.QoSMetric;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.wsdm.QoSMetricDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;

import com.ebmwebsourcing.wsstar.dm.api.QoSMetrics;
import com.ebmwebsourcing.wsstar.dm.api.WSDMException;

/**
 * @author aruffie - eBM Websourcing
 * 
 */
public class NotificationManagerImpl implements NotificationManager {

    private Logger logger = Logger.getLogger(this.getClass());


    private QoSMetricDAO qoSMetricDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    private EndpointDAO endpointDAO;



    public NotificationManagerImpl() {

    }

    public void handleNotification(final QoSMetrics metric, final String endpointRef) {
        QoSMetricTO metricTO =  this.transformQoSMetrics(metric, endpointRef);
        this.storeQoSMetrics(metricTO);
    }

    /**
     * Runnable threatment, allow to handle a stored notification in queue
     * 
     * @author aruffie
     * 
     */
    private void storeQoSMetrics(QoSMetricTO metricTO) {

        final QoSMetric metricBO = new QoSMetric();

        // Transform and store metric
        transfertObjectAssembler.toQoSMetricBO(metricTO, metricBO);
        qoSMetricDAO.save(metricBO);

    }

    /**
     * allow to store a incoming QoSMetrics in the queue
     * 
     * @author aruffie
     * 
     */
    private QoSMetricTO transformQoSMetrics(final QoSMetrics metric, final String endpointRef) {
        QoSMetricTO metricTO = null;
        try {

            /*
             * Convert in metric TO
             */
            metricTO = convertQoSMetrics(metric, endpointRef);


        } catch (WSDMException e) {
            logger.error(e.getMessage(), e);
        }
        return metricTO;
    }

    /**
     * Convert a WSSTAR QoS Metrics in QoSMetricTO. The endpoint reference
     * is use to recover the binding operation in database
     * 
     * @param QoSMetrics
     *            metric
     * @param String
     *            endpointRef
     * @return QoSMetricTO
     * @throws WSDMException 
     */
    private QoSMetricTO convertQoSMetrics(final QoSMetrics metric, final String endpointRef) throws WSDMException {
        final Calendar calendar = GregorianCalendar.getInstance();
        final QoSMetricTO metricTO = new QoSMetricTO();

        if (metric.getLastRequestSize() != null
                && metric.getLastRequestSize().getValue() != null) {
            metricTO.setLastRequestSize(metric.getLastRequestSize().getValue());
        }

        if (metric.getLastResponseSize() != null
                && metric.getLastResponseSize().getValue() != null) {
            metricTO.setLastResponseSize(metric.getLastResponseSize().getValue());
        }

        if (metric.getMaxRequestSize() != null && metric.getMaxRequestSize().getValue() != null) {
            metricTO.setMaxRequestSize(metric.getMaxRequestSize().getValue());
        }

        if (metric.getMaxResponseSize() != null
                && metric.getMaxResponseSize().getValue() != null) {
            metricTO.setMaxResponseSize(metric.getMaxResponseSize().getValue());
        }

        if (metric.getNumberOfFailedRequests() != null
                && metric.getNumberOfFailedRequests().getValue() != null) {
            metricTO.setNumberOfFailedRequest(metric.getNumberOfFailedRequests().getValue());
        }

        if (metric.getNumberOfRequests() != null
                && metric.getNumberOfRequests().getValue() != null) {
            metricTO.setNumberOfRequests(metric.getNumberOfRequests().getValue());
        }

        if (metric.getNumberOfSuccessfulRequests() != null
                && metric.getNumberOfSuccessfulRequests().getValue() != null) {
            metricTO.setNumberOfSuccessfulRequest(metric.getNumberOfSuccessfulRequests()
                    .getValue());
        }

        if (metric.getLastResponseTime() != null
                && metric.getLastResponseTime().getValue() != null) {
            metricTO.setLastResponseTime(metric.getLastResponseTime().getValue()
                    .getTimeInMillis(calendar));
        }

        if (metric.getMaxResponseTime() != null
                && metric.getMaxResponseTime().getValue() != null) {
            metricTO.setMaxResponseTime(metric.getMaxResponseTime().getValue().getTimeInMillis(
                    calendar));
        }
        if (metric.getServiceTime() != null && metric.getServiceTime().getValue() != null) {
            metricTO.setServiceTime(metric.getServiceTime().getValue()
                    .getTimeInMillis(calendar));
        }

        if(metric.getOperation() == null) {
            throw new WSDMException("The operation in metric cannot be null!!!");
        }


        /*
         * Retrieve linked endpoints to the provided reference name
         */
        String[] addr = endpointRef.split("::");
        if(addr == null || addr.length != 2) {
            throw new WSDMException("Endpoint address is not correctly formatted => {ns}ServiceName::EndpointName and not: " + endpointRef);
        }
        QName serviceName = QName.valueOf(addr[0].trim());
        String endpointName = addr[1].trim();

        final Endpoint endpoint = endpointDAO.getByNameAndServiceFullName(serviceName.toString(), endpointName);     

        if(endpoint == null) {
            throw new WSDMException("Impossible to find endpoint: " + endpointRef);
        }

        final Binding binding = endpoint.getBinding();
        if (binding != null) {
            final Set<BindingOperation> set = binding.getBindingOps();
            if (set != null && set.size() > 0) {
                Iterator<BindingOperation> it = set.iterator();
                while (it.hasNext()) {
                    final BindingOperation bo = it.next();
                    if (bo.getSpecifiedOp() != null) {
                        if (bo.getSpecifiedOp().getName() != null) {

                            /*
                             * If the current binding operation is
                             * the researched, set it into the qos
                             * metric object. After, break the
                             * research loop for optimization
                             */
                            final QName boQName = QName.valueOf(bo.getSpecifiedOp()
                                    .getName());
                            final QName metricQName = QName.valueOf(metric
                                    .getOperation());
                            if (boQName.toString().equals(metricQName.toString())
                                    || boQName.getLocalPart().equals(
                                            metricQName.getLocalPart())) {
                                metricTO.setBindingOperation(transfertObjectAssembler
                                        .toBindingOpTO(bo));
                                break;
                            }
                        }
                    }
                }
            }
        }

        if(metricTO.getBindingOperation() == null) {
            throw new WSDMException("Impossible to find the binding operation corresponding to " + metric.getOperation());
        }

        return metricTO;
    }


    public QoSMetricDAO getQoSMetricDAO() {
        return qoSMetricDAO;
    }

    public void setQoSMetricDAO(QoSMetricDAO qoSMetricDAO) {
        this.qoSMetricDAO = qoSMetricDAO;
    }

    public TransfertObjectAssembler getTransfertObjectAssembler() {
        return transfertObjectAssembler;
    }

    public void setTransfertObjectAssembler(TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    /**
     * @return the endpointDAO
     */
    public EndpointDAO getEndpointDAO() {
        return endpointDAO;
    }

    /**
     * @param endpointDAO
     *            the endpointDAO to set
     */
    public void setEndpointDAO(EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

}
