/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2008 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * QoSMetricManagerImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.wsdm;

import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPConnectionFactory;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import org.ow2.dragon.api.service.deployment.DeploymentException;
import org.ow2.dragon.api.service.wsdm.QoSMetricManager;
import org.ow2.dragon.api.service.wsdm.QoSMetricServiceException;
import org.ow2.dragon.api.to.deployment.EndpointTO;
import org.ow2.dragon.api.to.wsdm.QoSMetricTO;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.BindingOperation;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.wsdm.QoSMetric;
import org.ow2.dragon.persistence.dao.GenericUnifiedDAO;
import org.ow2.dragon.persistence.dao.deployment.BindingDAO;
import org.ow2.dragon.persistence.dao.deployment.BindingOperationDAO;
import org.ow2.dragon.persistence.dao.deployment.EndpointDAO;
import org.ow2.dragon.persistence.dao.wsdm.QoSMetricDAO;
import org.ow2.dragon.service.TransfertObjectAssembler;
import org.ow2.easywsdl.schema.api.XmlException;

import com.ebmwebsourcing.wsstar.addressing.definition.api.WSAddressingException;
import com.ebmwebsourcing.wsstar.notification.definition.utils.WSNotificationException;
import com.ebmwebsourcing.wsstar.notification.extension.utils.WSNotificationExtensionException;

/**
 * @author aruffie - eBM Websourcing
 * 
 */
public class QoSMetricManagerImpl implements QoSMetricManager {

    private QoSMetricDAO qoSMetricDAO;

    private BindingOperationDAO bindingOperationDAO;

    private EndpointDAO endpointDAO;

    private BindingDAO bindingDAO;

    private GenericUnifiedDAO<TechnicalService, String> technicalServiceUnifiedDAO;

    private TransfertObjectAssembler transfertObjectAssembler;

    /**
     * Allow to save in database new QoSMetric object, by providing a
     * QoSMetricTO
     * 
     * @param qoSMetricTO
     * @throws DeploymentException
     */
    public void saveQoSMetric(final QoSMetricTO qoSMetricTO) throws DeploymentException {
        this.qoSMetricDAO.save(this.toQoSMetricBO(qoSMetricTO));
    }

    /**
     * Allow to remove a QoSMetric by providing the linked binding operation
     * identifier
     * 
     * @param String
     *            bindingOperationId
     * @throws DeploymentException
     */
    public void removeQoSMetric(final String qoSMetricId) throws DeploymentException {
        this.qoSMetricDAO.remove(qoSMetricId);
    }

    /**
     * Allow to get a specific QoSMetricTO by providing the binding operation
     * identifier
     * 
     * @param String
     *            bindingOperationId
     * @return
     * @throws DeploymentException
     */
    public QoSMetricTO getQoSMetric(final String bindingOperationId) throws DeploymentException {
        final List<QoSMetric> qoSMetricBOs = this.qoSMetricDAO.searchEquals(
                new String[] { bindingOperationId }, new String[] { "bindingOperation.id" }, null);

        if (!qoSMetricBOs.isEmpty()) {
            return this.toQoSMetricTO(qoSMetricBOs.get(0));
        }
        return null;
    }

    /**
     * Allow to recover all QoSMetrics stored in database
     * 
     * @return List<QoSMetricTO
     * @throws DeploymentException
     */
    public List<QoSMetricTO> getAllQoSMetrics() throws DeploymentException {
        final List<QoSMetricTO> result = new ArrayList<QoSMetricTO>();
        final List<QoSMetric> qoSMetrics = this.qoSMetricDAO.getAll();
        this.toQoSMetricsTO(result, qoSMetrics);
        return result;
    }

    /**
     * Allow to update a QoSMetricBO by providing a QoSMetricTO
     * 
     * @param QoSMetricTO
     *            qoSMetricTO
     * @throws DeploymentException
     */
    public void updateQoSMetric(final QoSMetricTO qoSMetricTO) throws DeploymentException {
        final QoSMetric qoSMetric = this.qoSMetricDAO.get(qoSMetricTO.getId());
        if (qoSMetric != null) {
            qoSMetric.setLastRequestSize(qoSMetricTO.getLastRequestSize());
            qoSMetric.setLastResponseSize(qoSMetricTO.getLastResponseSize());
            qoSMetric.setLastResponseTime(qoSMetricTO.getLastResponseTime());
            qoSMetric.setMaxRequestSize(qoSMetricTO.getMaxRequestSize());
            qoSMetric.setMaxResponseSize(qoSMetricTO.getMaxResponseSize());
            qoSMetric.setMaxResponseTime(qoSMetricTO.getMaxResponseTime());
            qoSMetric.setNumberOfFailedRequest(qoSMetricTO.getNumberOfFailedRequest());
            qoSMetric.setNumberOfRequests(qoSMetricTO.getNumberOfRequests());
            qoSMetric.setNumberOfSuccessfulRequest(qoSMetric.getNumberOfSuccessfulRequest());
            qoSMetric.setServiceTime(qoSMetricTO.getServiceTime());
        }
    }

    /**
     * Allow to calculate QoSMetric for a given endpoint
     * 
     * @param EndpointTO
     *            endpointTO
     * @return QoSMetricTO
     * @throws DeploymentException
     */
    public QoSMetricTO getQoSMetricForEndpoint(final String endpointId) throws DeploymentException {

        // Retrieve Endpoint
        final EndpointTO endpointTO = this.transfertObjectAssembler.toEndpointTO(this.endpointDAO
                .get(endpointId), null);
        if (endpointTO != null && endpointTO.getBinding() != null) {
            return this.getQoSMetricForBinding(endpointTO.getBinding().getId());
        }
        return null;
    }

    /**
     * Allow to calculate QoSMetric for a given technical service
     * 
     * @param techServiceTO
     * @return
     * @throws DeploymentException
     */
    public QoSMetricTO getQoSMetricForTechnicalService(final String techServiceId)
            throws DeploymentException {

        // retrieve tech serv bo
        final TechnicalService technicalServiceBO = this.technicalServiceUnifiedDAO
                .get(techServiceId);

        final List<QoSMetricTO> qosList = new ArrayList<QoSMetricTO>();
        for (final Endpoint endpoint : technicalServiceBO.getEndpoints()) {
            final QoSMetricTO qoSMetric = this.getQoSMetricForEndpoint(endpoint.getId());
            if (qoSMetric != null) {
                qosList.add(qoSMetric);
            }
        }
        return aggregateQoSMetric(qosList);
    }

    /**
     * Allow to calculate QoSMetric for a given binding
     * 
     * @param String
     *            bindingId
     * @return QoSMetricTO
     * @throws DeploymentException
     */

    public QoSMetricTO getQoSMetricForBinding(final String bindingId) throws DeploymentException {
        final Binding bindingBO = this.bindingDAO.get(bindingId);
        final Set<BindingOperation> bindingOpsList = bindingBO.getBindingOps();
        final List<QoSMetricTO> qosList = new ArrayList<QoSMetricTO>();
        for (final BindingOperation bdoTO : bindingOpsList) {
            final QoSMetricTO qoSMetric = this.getQoSMetric(bdoTO.getId());
            if (qoSMetric != null) {
                qosList.add(qoSMetric);
            }
        }
        return aggregateQoSMetric(qosList);
    }

    /**
     * Allow to aggregate all QoSMetrics in one (by averaging)
     * 
     * @param List
     *            <QoSMetricTO> qosList
     * @return QoSMetricTO
     */
    private QoSMetricTO aggregateQoSMetric(final List<QoSMetricTO> qosList) {
        if (!qosList.isEmpty()) {
            final QoSMetricTO result = new QoSMetricTO();

            // Add all values
            for (final QoSMetricTO to : qosList) {
                result.setLastRequestSize(result.getLastRequestSize() + to.getLastRequestSize());
                result.setLastResponseSize(result.getLastResponseSize() + to.getLastResponseSize());
                result.setLastResponseTime(result.getLastResponseTime() + to.getLastResponseTime());
                result.setMaxRequestSize(result.getMaxRequestSize() + to.getMaxRequestSize());
                result.setMaxResponseSize(result.getMaxResponseSize() + to.getMaxResponseSize());
                result.setMaxResponseTime(result.getMaxResponseTime() + to.getMaxResponseTime());
                result.setNumberOfFailedRequest(result.getNumberOfFailedRequest()
                        + to.getNumberOfFailedRequest());
                result.setNumberOfRequests(result.getNumberOfRequests() + to.getNumberOfRequests());
                result.setNumberOfSuccessfulRequest(result.getNumberOfSuccessfulRequest()
                        + to.getNumberOfSuccessfulRequest());
                result.setServiceTime(result.getServiceTime() + to.getServiceTime());
            }

            // Perform all averages
            result.setLastRequestSize(result.getLastRequestSize() / qosList.size());
            result.setLastResponseSize(result.getLastResponseSize() / qosList.size());
            result.setLastResponseTime(result.getLastResponseTime() / qosList.size());
            result.setMaxRequestSize(result.getMaxRequestSize() / qosList.size());
            result.setMaxResponseSize(result.getMaxResponseSize() / qosList.size());
            result.setMaxResponseTime(result.getMaxResponseTime() / qosList.size());
            result.setNumberOfFailedRequest(result.getNumberOfFailedRequest() / qosList.size());
            result.setNumberOfRequests(result.getNumberOfRequests() / qosList.size());
            result.setNumberOfSuccessfulRequest(result.getNumberOfSuccessfulRequest()
                    / qosList.size());
            result.setServiceTime(result.getServiceTime() / qosList.size());
            return result;
        }
        return null;
    }

    private void toQoSMetricsTO(final List<QoSMetricTO> result, final List<QoSMetric> qoSMetrics) {
        if ((qoSMetrics != null) && !qoSMetrics.isEmpty()) {
            for (final QoSMetric qoSMetric : qoSMetrics) {
                result.add(this.toQoSMetricTO(qoSMetric));
            }
        }
    }

    private QoSMetricTO toQoSMetricTO(final QoSMetric qoSMetric) {
        return this.transfertObjectAssembler.toQoSMetricTO(qoSMetric);
    }

    private QoSMetric toQoSMetricBO(final QoSMetricTO qoSMetricTO) {
        final QoSMetric qoSMetricBO = new QoSMetric();
        this.transfertObjectAssembler.toQoSMetricBO(qoSMetricTO, qoSMetricBO);
        return qoSMetricBO;
    }

    public QoSMetricDAO getQoSMetricDAO() {
        return qoSMetricDAO;
    }

    public void setQoSMetricDAO(QoSMetricDAO qoSMetricDAO) {
        this.qoSMetricDAO = qoSMetricDAO;
    }

    public BindingOperationDAO getBindingOperationDAO() {
        return bindingOperationDAO;
    }

    public void setBindingOperationDAO(BindingOperationDAO bindingOperationDAO) {
        this.bindingOperationDAO = bindingOperationDAO;
    }

    public TransfertObjectAssembler getTransfertObjectAssembler() {
        return transfertObjectAssembler;
    }

    public void setTransfertObjectAssembler(TransfertObjectAssembler transfertObjectAssembler) {
        this.transfertObjectAssembler = transfertObjectAssembler;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.ow2.dragon.api.service.deployment.QoSMetricManager#getCurrentMessage
     * (java.lang.String)
     */
    public Map<String, String> getCurrentMessage(final String address)
            throws QoSMetricServiceException {
        
        try{
            
            final SOAPConnectionFactory soapConnectionFactory = SOAPConnectionFactory.newInstance();
            final SOAPConnection soapConnection = soapConnectionFactory.createConnection();
            final URL endpointURL = new URL(address);
            
            /*
             * Create SOAP GetCurrentMessage message thank to SAAJ
             * in order to get several metrics returned by the non
             * functional targeted endpoint 
             */
            final SOAPMessage message = SAAJHelper.createGetCurrentMessageSOAPMessage();
            final MimeHeaders headers = message.getMimeHeaders();

            /*
             * Add SOAPAction in the message in order to specify to the
             * monitoring ESB how to handler this SOAP message
             */
            headers.addHeader("SOAPAction", "http://petals.ow2.org/wsdm/GetCurrentMessage");

            final SOAPMessage response = soapConnection.call(message, endpointURL);
            
        } catch (final MalformedURLException e) {
            throw new QoSMetricServiceException(e);
        } catch (final SOAPException e) {
            throw new QoSMetricServiceException(e);
        } 
                return null;
        //TODO: refactor getCurrentMessage and use SAAJ like Subscribe/Unsubscribe request
        /*final QName service_name = new QName("http://petals.ow2.org/wsdm", "WSDMService");
        final URL url = this.getClass().getClassLoader().getResource("wsdm-server.wsdl");
        WSDMService ss = new WSDMService(url, service_name);
        final QName portName = new QName("http://petals.ow2.org/wsdm", "WSDMServicePort");

        ss.addPort(portName, "WSDMBinding", address);

        ClientQoSMetricManager port = ss.getPort(portName, ClientQoSMetricManager.class);

        try {
            final GetCurrentMessage message = new GetCurrentMessage();
            final TopicExpressionType tType = new TopicExpressionType();
            tType.setDialect("http://www.w3.org/TR/1999/REC-xpath-19991116\\");
            tType.getContent().add("mows-ev:MetricsCapability");
            message.setTopic(tType);
            final GetCurrentMessageResponse response = port.getCurrentMessage(message);
            return formatResponse(response);
        } catch (final InvalidTopicExpressionFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final ResourceUnknownFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final TopicExpressionDialectUnknownFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final MultipleTopicsSpecifiedFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final TopicNotSupportedFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final NoCurrentMessageOnTopicFault e) {
            throw new QoSMetricServiceException(e);
        } catch (final javax.xml.ws.soap.SOAPFaultException e) {
            throw new QoSMetricServiceException(e);
        }*/
    }

    /**
     * Allow to format the received response by return a Metric/Value Map
     * 
     * @param GetCurrentMessageResponse
     *            response
     * @return Map<String,String>
     */
    /*private Map<String, String> formatResponse(final GetCurrentMessageResponse response) {
        final List<Object> any = response.getAny();
        final Map<String, String> result = new HashMap<String, String>();
        for (final Object obj : any) {
            final ElementNSImpl element = (ElementNSImpl) obj;
            result.put(element.getNodeName(), element.getTextContent());
        }
        return result;
    }*/

    /**
     * Allow to subscribe to a non function endpoint in order to receive several
     * notifications
     * 
     * @param String
     *            endpointId
     * @param String
     *            responseAddress
     * @return EndpointTO
     */
    public EndpointTO subscribe(final String endpointId, final String responseAddress)
            throws QoSMetricServiceException {

        try {
            final Endpoint endpoint = endpointDAO.get(endpointId);
            final SOAPConnectionFactory soapConnectionFactory = SOAPConnectionFactory.newInstance();
            final SOAPConnection soapConnection = soapConnectionFactory.createConnection();
            final URL endpointURL = new URL(endpoint.getSubscriptionAddr());

            /*
             * Create SOAP subscription message thank to SAAJ. the response
             * address is the dragon endpoint address where notifications will
             * be sent
             */
            final SOAPMessage message = SAAJHelper.createSubscribeSOAPMessage(responseAddress);
            final MimeHeaders headers = message.getMimeHeaders();

            /*
             * Add SOAPAction in the message in order to specify to the
             * monitoring ESB how to handler this SOAP message
             */
            headers.addHeader("SOAPAction", "http://petals.ow2.org/wsdm/Subscribe");

            final SOAPMessage response = soapConnection.call(message, endpointURL);

            // Retrieve the returned subscription ID
            endpoint.setResourcesUuidType(SAAJHelper.extractResourcesUuidType(response).trim());
            endpoint.setSubscription(true);
            endpointDAO.save(endpoint);
            soapConnection.close();
            return this.transfertObjectAssembler.toEndpointTO(endpoint, null);

        } catch (final UnsupportedOperationException e) {
            throw new QoSMetricServiceException(e);
        } catch (final SOAPException e) {
            throw new QoSMetricServiceException(e);
        } catch (final MalformedURLException e) {
            throw new QoSMetricServiceException(e);
        } catch (final WSAddressingException e) {
            throw new QoSMetricServiceException(e);
        } catch (final XmlException e) {
            throw new QoSMetricServiceException(e);
        }
    }

    /**
     * Allow to unsubscribe to a non function endpoint in order to stop
     * notification shipments
     * 
     * @param String
     *            endpointId
     * @param String
     *            responseAddress
     * @return EndpointTO
     */
    public EndpointTO unSubscribe(final String endpointId, final String responseAddress)
            throws QoSMetricServiceException {

        final Endpoint endpoint = this.endpointDAO.get(endpointId);

        try {

            final SOAPConnectionFactory soapConnectionFactory = SOAPConnectionFactory.newInstance();
            final SOAPConnection soapConnection = soapConnectionFactory.createConnection();
            final URL endpointURL = new URL(endpoint.getSubscriptionAddr());

            /*
             * Create SOAP unsubscription message thank to SAAJ. The response
             * address is the dragon endpoint address where notifications was
             * sent. The Uuid is the id returned in response of subscribed
             * request
             */
            final SOAPMessage message = SAAJHelper.createUnSubscribeSOAPMessage(responseAddress,
                    endpoint.getResourcesUuidType());

            /*
             * Add SOAPAction in the message in order to specify to the
             * monitoring ESB how to handler this SOAP message
             */
            final MimeHeaders headers = message.getMimeHeaders();
            headers.addHeader("SOAPAction", "http://petals.ow2.org/wsdm/Unsubscribe");
            final SOAPMessage response = soapConnection.call(message, endpointURL);
            endpoint.setResourcesUuidType(null);
            endpoint.setSubscription(false);
            this.endpointDAO.save(endpoint);
            soapConnection.close();
            return this.transfertObjectAssembler.toEndpointTO(endpoint, null);
        } catch (final UnsupportedOperationException e) {
            throw new QoSMetricServiceException(e);
        } catch (final SOAPException e) {
            if (e.getCause().equals(ConnectException.class)) {
                throw new QoSMetricServiceException(e.getMessage() + " because, "
                        + e.getCause().getMessage() + ". Check if the address: "
                        + endpoint.getSubscriptionAddr() + "  is reachable.");
            }
            throw new QoSMetricServiceException(e);
        } catch (final MalformedURLException e) {
            throw new QoSMetricServiceException(e);
        } catch (final WSAddressingException e) {
            throw new QoSMetricServiceException(e);
        } catch (WSNotificationException e) {
            throw new QoSMetricServiceException(e);
        } catch (WSNotificationExtensionException e) {
            throw new QoSMetricServiceException(e);
        } 
    }

    public EndpointDAO getEndpointDAO() {
        return endpointDAO;
    }

    public void setEndpointDAO(EndpointDAO endpointDAO) {
        this.endpointDAO = endpointDAO;
    }

    public BindingDAO getBindingDAO() {
        return bindingDAO;
    }

    public void setBindingDAO(BindingDAO bindingDAO) {
        this.bindingDAO = bindingDAO;
    }

    public GenericUnifiedDAO<TechnicalService, String> getTechnicalServiceUnifiedDAO() {
        return technicalServiceUnifiedDAO;
    }

    public void setTechnicalServiceUnifiedDAO(
            GenericUnifiedDAO<TechnicalService, String> technicalServiceUnifiedDAO) {
        this.technicalServiceUnifiedDAO = technicalServiceUnifiedDAO;
    }
}
