/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * CommonQueryHelper.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.query;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import org.ow2.dragon.persistence.bo.common.CategoryBag;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.KeyedReferenceGroup;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.service.uddi.FindQualifiers;
import org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3;
import org.ow2.dragon.util.StringHelper;

import com.trg.search.Filter;
import com.trg.search.Search;
import com.trg.search.Sort;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class CommonQueryHelper {

    protected static Search createDistinctSearch(Class<?> searchedType) {
        Search searchQuery = new Search();
        searchQuery.setSearchClass(searchedType);
        searchQuery.setDistinct(true);
        return searchQuery;
    }

    protected static void addPagingOptions(Search searchQuery, Paging paging) {
        searchQuery.setFirstResult(paging.getFirstResult());
        searchQuery.setMaxResults(paging.getMaxResult());
    }

    protected static void addSortOptions(Class<?> searchedType, Search searchQuery,
            FindQualifiers findQualifiers) {
        // Sort on name by default
        Sort sortName = null;
        Sort sortDate = new Sort("lastUpdated");

        if (searchedType.equals(TModel.class)) {
            sortName = new Sort("name.name");
            // Need to fetch names and date to be able to do a "select distinct"
            // with an "order by" clause
            searchQuery.addFetch("name");
        } else if (searchedType.equals(OrganizationUnit.class)
                || searchedType.equals(TechnicalService.class)) {
            sortName = new Sort("names.name");
            // Need to fetch names and date to be able to do a "select distinct"
            // with an "order by" clause
            searchQuery.addFetch("names");
        }

        if (sortName != null) {
            // Set case sensitivity
            // TODO replace it by a more elegant solution based on heritage
            if (findQualifiers instanceof FindQualifiersV3) {
                FindQualifiersV3 findQualifiersV3 = (FindQualifiersV3) findQualifiers;
                if (findQualifiersV3.isCaseInsensitiveSort()) {
                    sortName.setIgnoreCase(true);
                } else if (findQualifiersV3.isCaseSensitiveSort()) {
                    sortName.setIgnoreCase(false);
                }
            }

            if (findQualifiers.isSortByNameAsc()) {
                if (findQualifiers.isSortByDateAsc()) {
                    sortDate.setDesc(false);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                } else if (findQualifiers.isSortByDateDesc()) {
                    sortDate.setDesc(true);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                } else {
                    sortDate.setDesc(false);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                }
            } else if (findQualifiers.isSortByNameDesc()) {
                if (findQualifiers.isSortByDateAsc()) {
                    sortDate.setDesc(false);
                    sortName.setDesc(true);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                } else if (findQualifiers.isSortByDateDesc()) {
                    sortDate.setDesc(true);
                    sortName.setDesc(true);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                } else {
                    sortDate.setDesc(false);
                    sortName.setDesc(true);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                }
            } else {
                if (findQualifiers.isSortByDateAsc()) {
                    sortDate.setDesc(false);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortDate);
                    searchQuery.addSort(sortName);
                } else if (findQualifiers.isSortByDateDesc()) {
                    sortDate.setDesc(true);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortDate);
                    searchQuery.addSort(sortName);
                } else {
                    sortDate.setDesc(false);
                    sortName.setDesc(false);
                    searchQuery.addSort(sortName);
                    searchQuery.addSort(sortDate);
                }
            }
        } else {
            if (findQualifiers.isSortByDateAsc()) {
                sortDate.setDesc(false);
            } else if (findQualifiers.isSortByDateDesc()) {
                sortDate.setDesc(true);
            } else {
                sortDate.setDesc(false);
            }
            searchQuery.addSort(sortDate);
        }
    }

    protected static Filter createNamesFilter(Class<?> searchedType, FindQualifiers findQualifiers,
            List<Name> names) {
        Filter result = null;
        if (names != null && !names.isEmpty()) {
            result = Filter.or();
            for (Name name : names) {
                result.add(Filter.some("names",
                        createNameFilter(searchedType, findQualifiers, name)));
            }
        }
        return result;
    }

    protected static Filter createNameFilter(Class<?> searchedType, FindQualifiers findQualifiers,
            Name name) {
        Filter result = null;
        String langCodePropertyKey = null;
        String namePropertyKey = null;
        if (searchedType.equals(TModel.class)) {
            langCodePropertyKey = "name.langCode";
            namePropertyKey = "name.name";
        } else if (searchedType.equals(TechnicalService.class)
                || searchedType.equals(OrganizationUnit.class)) {
            langCodePropertyKey = "langCode";
            namePropertyKey = "name";
        }
        if (name != null) {
            if (findQualifiers.isExactMatch()) {
                if (findQualifiers.isCaseSensitiveMatch()) {
                    if (!StringHelper.isNullOrEmpty(name.getLangCode())) {
                        result = Filter.and(Filter.iequal(langCodePropertyKey, name.getLangCode()),
                                Filter.equal(namePropertyKey, name.getName()));
                    } else {
                        result = Filter.equal(namePropertyKey, name.getName());
                    }
                } else {
                    if (!StringHelper.isNullOrEmpty(name.getLangCode())) {
                        result = Filter.and(Filter.iequal(langCodePropertyKey, name.getLangCode()),
                                Filter.iequal(namePropertyKey, name.getName()));
                    } else {
                        result = Filter.iequal(namePropertyKey, name.getName());
                    }
                }
            } else {
                if (findQualifiers.isCaseSensitiveMatch()) {
                    if (!StringHelper.isNullOrEmpty(name.getLangCode())) {
                        result = Filter.and(Filter.iequal(langCodePropertyKey, name.getLangCode()),
                                Filter.like(namePropertyKey, name.getName()));
                    } else {
                        result = Filter.like(namePropertyKey, name.getName());
                    }
                } else {
                    if (!StringHelper.isNullOrEmpty(name.getLangCode())) {
                        result = Filter.and(Filter.iequal(langCodePropertyKey, name.getLangCode()),
                                Filter.ilike(namePropertyKey, name.getName()));
                    } else {
                        result = Filter.ilike(namePropertyKey, name.getName());
                    }
                }
            }
        }
        return result;
    }

    protected static Filter createIdentifierBagFilter(FindQualifiers findQualifiers,
            List<KeyedReference> dragonIdentifierBag) {
        Filter identifierBagFilter = null;
        if (findQualifiers.isAndAllKeys() || findQualifiers.isOrLikeKeys()) {
            identifierBagFilter = Filter.and();
        } else {
            // Default value
            identifierBagFilter = Filter.or();
        }

        // Add category KeyedReferences filters
        createKeyedReferencesFilter(findQualifiers, dragonIdentifierBag, identifierBagFilter,
                "identifierBag");

        return identifierBagFilter;
    }

    private static void createKeyedReferencesFilter(FindQualifiers findQualifiers,
            List<KeyedReference> krs, Filter parentFilter, String krsPropertyKey) {
        if (krs != null) {
            // Add category KeyedReferences filters
            if (findQualifiers.isOrLikeKeys()) {
                // Create a "or" filter between KeyedReference in the same
                // namespace (e.g having the same tmodelKey)
                Collections.sort(krs, new Comparator<KeyedReference>() {
                    public int compare(KeyedReference o1, KeyedReference o2) {
                        return o1.getTmodel().getId().compareTo(o2.getTmodel().getId());
                    }
                });
                String currentKRTModelKey = null;
                Filter sameNamespaceFilter = null;
                for (KeyedReference keyedReference : krs) {
                    if (!keyedReference.getTmodel().getId().equals(currentKRTModelKey)) {
                        sameNamespaceFilter = Filter.or();
                        parentFilter.add(sameNamespaceFilter);
                        currentKRTModelKey = keyedReference.getTmodel().getId();
                    }
                    sameNamespaceFilter.add(Filter.some(krsPropertyKey, createKeyedReferenceFilter(
                            findQualifiers, keyedReference)));
                }
            } else {
                for (KeyedReference keyedReference : krs) {
                    parentFilter.add(Filter.some(krsPropertyKey, createKeyedReferenceFilter(
                            findQualifiers, keyedReference)));
                }
            }
        }
    }

    private static Filter createKeyedReferenceFilter(FindQualifiers findQualifiers,
            KeyedReference keyedReference) {
        Filter result = null;
        String tModelProperty = "tmodel.id";
        String keyValueProperty = "keyValue";
        String keyNameProperty = "keyName";
        if (findQualifiers.isExactMatch()) {
            if (findQualifiers.isCaseSensitiveMatch()) {
                result = Filter.and(Filter
                        .equal(tModelProperty, keyedReference.getTmodel().getId()), Filter.equal(
                        keyValueProperty, keyedReference.getKeyValue()));
                if (!StringHelper.isNullOrEmpty(keyedReference.getKeyName())) {
                    result.add(Filter.equal(keyNameProperty, keyedReference.getKeyName()));
                }
            } else {
                result = Filter.and(Filter
                        .equal(tModelProperty, keyedReference.getTmodel().getId()), Filter.iequal(
                        keyValueProperty, keyedReference.getKeyValue()));
                if (!StringHelper.isNullOrEmpty(keyedReference.getKeyName())) {
                    result.add(Filter.iequal(keyNameProperty, keyedReference.getKeyName()));
                }
            }
        } else {
            if (findQualifiers.isCaseSensitiveMatch()) {
                result = Filter.and(Filter
                        .equal(tModelProperty, keyedReference.getTmodel().getId()), Filter.like(
                        keyValueProperty, keyedReference.getKeyValue()));
                if (!StringHelper.isNullOrEmpty(keyedReference.getKeyName())) {
                    result.add(Filter.like(keyNameProperty, keyedReference.getKeyName()));
                }
            } else {
                result = Filter.and(Filter
                        .equal(tModelProperty, keyedReference.getTmodel().getId()), Filter.ilike(
                        keyValueProperty, keyedReference.getKeyValue()));
                if (!StringHelper.isNullOrEmpty(keyedReference.getKeyName())) {
                    result.add(Filter.ilike(keyNameProperty, keyedReference.getKeyName()));
                }
            }
        }
        return result;
    }

    protected static Filter createCategoryBagFilter(FindQualifiers findQualifiers,
            CategoryBag dragonCategoryBag) {
        Filter filterCategoryBag = null;
        if (findQualifiers.isOrAllKeys()) {
            filterCategoryBag = Filter.or();
        } else {
            // Default value
            filterCategoryBag = Filter.and();
        }
        if (dragonCategoryBag != null) {
            createKeyedReferencesFilter(findQualifiers, dragonCategoryBag.getKeyedReferences(),
                    filterCategoryBag, "categoryBag.keyedReferences");
            createKeyedReferenceGroupsFilter(findQualifiers, dragonCategoryBag, filterCategoryBag);
        }
        return filterCategoryBag;
    }

    private static void createKeyedReferenceGroupsFilter(FindQualifiers findQualifiers,
            CategoryBag dragonCategoryBag, Filter filterCategoryBag) {
        List<KeyedReferenceGroup> keyedReferenceGroups = dragonCategoryBag
                .getKeyedReferenceGroups();
        if (keyedReferenceGroups != null) {
            // Add category KeyedReferenceGroups filters
            String catBagKeyedRefGroupsProperty = "categoryBag.keyedReferenceGroups";
            if (findQualifiers.isOrLikeKeys()) {
                // Create a "or" filter between KeyedReferenceGroup in the same
                // namespace (e.g having the same tmodelKey)
                Collections.sort(keyedReferenceGroups, new Comparator<KeyedReferenceGroup>() {
                    public int compare(KeyedReferenceGroup o1, KeyedReferenceGroup o2) {
                        return o1.getTmodel().getId().compareTo(o2.getTmodel().getId());
                    }
                });
                String currentKRGTModelKey = null;
                Filter sameNamespaceFilter = null;
                for (KeyedReferenceGroup keyedReferenceGroup : keyedReferenceGroups) {
                    if (!keyedReferenceGroup.getTmodel().getId().equals(currentKRGTModelKey)) {
                        sameNamespaceFilter = Filter.or();
                        filterCategoryBag.add(sameNamespaceFilter);
                        currentKRGTModelKey = keyedReferenceGroup.getTmodel().getId();
                    }
                    sameNamespaceFilter.add(Filter.some(catBagKeyedRefGroupsProperty,
                            createKeyedReferenceGroupFilter(findQualifiers, keyedReferenceGroup)));
                }
            } else {
                for (KeyedReferenceGroup keyedReferenceGroup : keyedReferenceGroups) {
                    filterCategoryBag.add(Filter.some(catBagKeyedRefGroupsProperty,
                            createKeyedReferenceGroupFilter(findQualifiers, keyedReferenceGroup)));
                }
            }
        }
    }

    private static Filter createKeyedReferenceGroupFilter(FindQualifiers findQualifiers,
            KeyedReferenceGroup keyedReferenceGroup) {
        Filter keyedReferenceGroupFilter = Filter.and(Filter.equal("tmodel.id", keyedReferenceGroup
                .getTmodel().getId()));
        List<KeyedReference> keyedReferences = keyedReferenceGroup.getKeyedReferences();
        if (keyedReferences != null) {
            Filter keyedRefInGroupFilter = null;
            if (findQualifiers.isOrAllKeys()) {
                keyedRefInGroupFilter = Filter.or();
            } else {
                // Default value
                keyedRefInGroupFilter = Filter.and();
            }
            createKeyedReferencesFilter(findQualifiers, keyedReferences, keyedRefInGroupFilter,
                    "keyedReferences");

            keyedReferenceGroupFilter.add(keyedRefInGroupFilter);
        }
        return keyedReferenceGroupFilter;
    }

    protected static Filter createTModelKeysFilter(FindQualifiers findQualifiers,
            Set<String> modelKeys, String[] tModelBagSingleProperties,
            String[] tModelBagMultipleProperties) {
        Filter filterTModelKeys = null;
        if (findQualifiers.isOrAllKeys()) {
            filterTModelKeys = Filter.or();
        } else {
            // Default value
            filterTModelKeys = Filter.and();
        }
        if (modelKeys != null) {
            for (String modelKey : modelKeys) {
                Filter keyFilter = Filter.or();
                filterTModelKeys.add(keyFilter);
                // Try to find key in single related properties (like protocol)
                if (tModelBagSingleProperties != null) {
                    for (String tModelBagSingleProperty : tModelBagSingleProperties) {
                        keyFilter.add(Filter.equal(tModelBagSingleProperty + ".id", modelKey));
                    }
                }
                // Try to find key in multiple related properties (like service
                // specs)
                if (tModelBagMultipleProperties != null) {
                    for (String tModelBagMultipleProperty : tModelBagMultipleProperties) {
                        keyFilter.add(Filter.some(tModelBagMultipleProperty, Filter.equal("id",
                                modelKey)));
                    }
                }
            }
        }
        return filterTModelKeys;
    }
}
