/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDITransfertObjectAssembler.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v2.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.ow2.dragon.persistence.bo.common.Description;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.Link;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.organization.Address;
import org.ow2.dragon.persistence.bo.organization.AddressLine;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.Email;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Party;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.bo.organization.PhoneNumber;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.persistence.util.UDDIIdentifierGenerator;
import org.ow2.dragon.service.uddi.query.FindServiceQueryHelper;
import org.ow2.dragon.service.uddi.v2.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v2.error.InvalidKeyPassedException;
import org.ow2.dragon.util.DragonRepoFileLocator;
import org.uddi.api_v2.AccessPoint;
import org.uddi.api_v2.BindingTemplate;
import org.uddi.api_v2.BindingTemplates;
import org.uddi.api_v2.BusinessEntity;
import org.uddi.api_v2.BusinessInfo;
import org.uddi.api_v2.BusinessInfos;
import org.uddi.api_v2.BusinessService;
import org.uddi.api_v2.BusinessServices;
import org.uddi.api_v2.CategoryBag;
import org.uddi.api_v2.Contact;
import org.uddi.api_v2.Contacts;
import org.uddi.api_v2.DiscoveryURL;
import org.uddi.api_v2.DiscoveryURLs;
import org.uddi.api_v2.IdentifierBag;
import org.uddi.api_v2.InstanceDetails;
import org.uddi.api_v2.OverviewDoc;
import org.uddi.api_v2.Phone;
import org.uddi.api_v2.RelatedBusinessInfo;
import org.uddi.api_v2.RelatedBusinessInfos;
import org.uddi.api_v2.ServiceInfo;
import org.uddi.api_v2.ServiceInfos;
import org.uddi.api_v2.TModelInfo;
import org.uddi.api_v2.TModelInfos;
import org.uddi.api_v2.TModelInstanceDetails;
import org.uddi.api_v2.TModelInstanceInfo;
import org.uddi.api_v2.URLType;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class UDDITransferObjectAssembler {

    private Logger logger = Logger.getLogger(this.getClass());

    private HttpServletRequest request;

    public UDDITransferObjectAssembler(UniversalUnifiedDAO universalUnifiedDAO) {
        super();
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

    private UniversalUnifiedDAO universalUnifiedDAO;

    public BusinessEntity toUDDIBusiness(OrganizationUnit dragonBusiness) {
        BusinessEntity apiBusiness = new BusinessEntity();
        apiBusiness.setBusinessKey(UDDIIdentifierGenerator.toUddiV2Id(dragonBusiness.getId()));
        apiBusiness.setCategoryBag(toUDDICategoryBag(dragonBusiness.getCategoryBag()));
        apiBusiness.setIdentifierBag(toUDDIIdentifierBag(dragonBusiness.getIdentifierBag()));

        List<Name> names = dragonBusiness.getNames();
        if (names != null) {
            List<org.uddi.api_v2.Name> apiNames = apiBusiness.getName();
            for (Name name : names) {
                apiNames.add(toUDDIName(name));
            }
        }

        List<Description> descriptions = dragonBusiness.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v2.Description> apiDescriptions = apiBusiness.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        List<DiscoveryUrl> dragonDiscoveryUrls = dragonBusiness.getDiscoveryUrls();
        if (dragonDiscoveryUrls != null) {
            DiscoveryURLs apiDiscoveryURLs = new DiscoveryURLs();
            List<DiscoveryURL> apiDiscoveryURLList = apiDiscoveryURLs.getDiscoveryURL();
            for (DiscoveryUrl discoveryUrl : dragonDiscoveryUrls) {
                apiDiscoveryURLList.add(toUDDIDiscoveryUrl(discoveryUrl));
            }
            apiBusiness.setDiscoveryURLs(apiDiscoveryURLs);
        }

        apiBusiness.setContacts(retrieveOrganizationContacts(dragonBusiness));

        apiBusiness.setBusinessServices(retrieveLinkedBusinessServices(dragonBusiness));

        return apiBusiness;
    }

    private BusinessServices retrieveLinkedBusinessServices(OrganizationUnit dragonBusiness) {
        BusinessServices businessServices = new BusinessServices();

        Set<Link> roles = dragonBusiness.getFromLinks();
        if (roles != null) {
            List<BusinessService> apiServices = businessServices.getBusinessService();
            for (Link role : roles) {
                if (role.getTo() instanceof TechnicalService
                        && role.getType().equalsIgnoreCase(FindServiceQueryHelper.PROVIDER)) {
                    TechnicalService techServ = (TechnicalService) role.getTo();
                    apiServices.add(toUDDIService(techServ));
                }
            }
        }

        return businessServices;
    }

    /**
     * Retrieve a list of child parties that are instances of {@link Person} and
     * are marked as contact or have a post marked as contact <br/>
     * TODO if optimization is needed, could use a DAO search instead of a java
     * search
     * 
     * @param dragonBusiness
     * @return a {@link Contacts} element
     */
    private Contacts retrieveOrganizationContacts(OrganizationUnit dragonBusiness) {
        Contacts contacts = new Contacts();
        List<Contact> apiContacts = contacts.getContact();
        // Retrieve contact of this organization
        Set<Party> parties = dragonBusiness.getParties();
        if (parties != null) {
            for (Party party : parties) {
                if (party instanceof Person) {
                    Person person = (Person) party;
                    if (person.isContact()
                            || (person.getPost() != null && person.getPost().isContact())) {
                        apiContacts.add(toUDDIContact(person));
                    }
                }
            }
        }
        return contacts;
    }

    private Contact toUDDIContact(Person person) {
        Contact contact = new Contact();
        if (person.getPost() != null) {
            contact.setUseType(person.getPost().getNature());
        }

        // Add first person names as personname
        List<Name> names = person.getNames();
        if (names != null && !names.isEmpty()) {
            contact.setPersonName(names.get(0).getName());
        }

        List<Description> descriptions = person.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v2.Description> apiDescriptions = contact.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
            // Add role description to contact description
            apiDescriptions.add(toUDDIDescription(person.getPost()));
        }

        List<PhoneNumber> dragonPhones = person.getPhoneNumbers();
        if (dragonPhones != null) {
            List<Phone> apiPhones = contact.getPhone();
            for (PhoneNumber phone : dragonPhones) {
                apiPhones.add(toUDDIPhone(phone));
            }
        }

        List<Email> dragonEmails = person.getEmailAddresses();
        if (dragonEmails != null) {
            List<org.uddi.api_v2.Email> apiEmails = contact.getEmail();
            for (Email email : dragonEmails) {
                apiEmails.add(toUDDIEmail(email));
            }
        }

        // Add person addresses to contact addresses
        List<Address> dragonAddresses = person.getAddresses();
        List<org.uddi.api_v2.Address> apiAddresses = contact.getAddress();
        if (dragonAddresses != null) {
            for (Address address : dragonAddresses) {
                apiAddresses.add(toUDDIAddress(address));
            }
        }
        // Add person parent org addresses to contact addresses
        if (person.getParentOrganization() != null) {
            List<Address> orgAddresses = person.getParentOrganization().getAddresses();
            if (orgAddresses != null) {
                for (Address address : orgAddresses) {
                    apiAddresses.add(toUDDIAddress(address));
                }
            }
        }

        return contact;
    }

    private org.uddi.api_v2.Description toUDDIDescription(Post post) {
        org.uddi.api_v2.Description apiDescription = null;
        if (post != null) {
            apiDescription = new org.uddi.api_v2.Description();
            apiDescription.setLang("en");
            apiDescription.setValue(post.getDescription());
        }
        return apiDescription;
    }

    private org.uddi.api_v2.Address toUDDIAddress(Address address) {
        org.uddi.api_v2.Address apiAddress = null;
        if (address != null) {
            apiAddress = new org.uddi.api_v2.Address();
            // TODO apiAddress.setSortCode(value);
            apiAddress.setUseType(address.getUseType());
            if (address.getTModel() != null) {
                apiAddress.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(address.getTModel().getId()));
            }
            List<AddressLine> dragonLines = address.getAddressLines();
            if (dragonLines != null) {
                List<org.uddi.api_v2.AddressLine> apiLines = apiAddress.getAddressLine();
                for (AddressLine addressLine : dragonLines) {
                    apiLines.add(toUDDIAddressLine(addressLine));
                }
            }
        }
        return apiAddress;
    }

    private org.uddi.api_v2.AddressLine toUDDIAddressLine(AddressLine addressLine) {
        org.uddi.api_v2.AddressLine apiAddressLine = null;
        if (addressLine != null) {
            apiAddressLine = new org.uddi.api_v2.AddressLine();
            apiAddressLine.setKeyName(addressLine.getKeyName());
            apiAddressLine.setKeyValue(addressLine.getKeyValue());
            apiAddressLine.setValue(addressLine.getAddressLine());
        }
        return apiAddressLine;
    }

    private org.uddi.api_v2.Email toUDDIEmail(Email email) {
        org.uddi.api_v2.Email apiEmail = null;
        if (email != null) {
            apiEmail = new org.uddi.api_v2.Email();
            apiEmail.setUseType(email.getUseType());
            apiEmail.setValue(email.getEmail());
        }
        return apiEmail;
    }

    private Phone toUDDIPhone(PhoneNumber phone) {
        Phone apiPhone = null;
        if (phone != null) {
            apiPhone = new Phone();
            apiPhone.setUseType(phone.getUseType());
            apiPhone.setValue(phone.getPhoneNumber());
        }
        return apiPhone;
    }

    private DiscoveryURL toUDDIDiscoveryUrl(DiscoveryUrl discoveryUrl) {
        DiscoveryURL apiUrl = null;
        if (discoveryUrl != null) {
            apiUrl = new DiscoveryURL();
            apiUrl.setUseType(discoveryUrl.getUseType());
            apiUrl.setValue(discoveryUrl.getUrl());
        }
        return apiUrl;
    }

    private DiscoveryUrl toDragonDiscoveryUrl(DiscoveryURL discoveryUrl) {
        DiscoveryUrl dragonUrl = null;
        if (discoveryUrl != null) {
            dragonUrl = new DiscoveryUrl();
            dragonUrl.setUseType(discoveryUrl.getUseType());
            dragonUrl.setUrl(discoveryUrl.getValue());
        }
        return dragonUrl;
    }

    public List<DiscoveryUrl> toDragonDiscoveryUrls(DiscoveryURLs discoveryURLs) {
        List<DiscoveryUrl> dragonDiscoveryUrls = new ArrayList<DiscoveryUrl>();
        if (discoveryURLs != null) {
            List<DiscoveryURL> apiDiscoveryURLs = discoveryURLs.getDiscoveryURL();
            if (apiDiscoveryURLs != null) {
                for (DiscoveryURL discoveryUrl : apiDiscoveryURLs) {
                    dragonDiscoveryUrls.add(toDragonDiscoveryUrl(discoveryUrl));
                }
            }
        }
        return dragonDiscoveryUrls;
    }

    public BusinessService toUDDIService(TechnicalService dragonService) {
        BusinessService apiService = new BusinessService();
        apiService.setServiceKey(UDDIIdentifierGenerator.toUddiV2Id(dragonService.getId()));

        List<Name> names = dragonService.getNames();
        if (names != null) {
            List<org.uddi.api_v2.Name> apiNames = apiService.getName();
            for (Name name : names) {
                apiNames.add(toUDDIName(name));
            }
        }

        List<Description> descriptions = dragonService.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v2.Description> apiDescriptions = apiService.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        apiService.setCategoryBag(toUDDICategoryBag(dragonService.getCategoryBag()));

        apiService.setBusinessKey(UDDIIdentifierGenerator.toUddiV2Id(retrieveBusinessKeyFormTechServ(dragonService)));

        Set<Endpoint> eps = dragonService.getEndpoints();
        if (eps != null) {
            BindingTemplates bindingTemplates = new BindingTemplates();
            List<BindingTemplate> apiBindings = bindingTemplates.getBindingTemplate();
            for (Endpoint ep : eps) {
                apiBindings.add(toUDDIBindingTemplate(ep));
            }
            apiService.setBindingTemplates(bindingTemplates);
        }

        return apiService;
    }

    /**
     * Try to identify an organization unit which is "Provider" of the service
     * and use the first one id as businessKey <br/>
     * TODO replace role strings by Constants or better by TModel describing
     * possible roles
     * 
     * @param dragonService
     * @return the businessKey
     */
    private String retrieveBusinessKeyFormTechServ(TechnicalService dragonService) {
        String bKey = null;
        Set<Link> roles = dragonService.getToLinks();
        if (roles != null) {
            for (Link partyToTechService : roles) {
                if (FindServiceQueryHelper.PROVIDER.equalsIgnoreCase(partyToTechService.getType())) {
                    bKey = partyToTechService.getFrom().getId();
                    break;
                }
            }
        }
        return bKey;
    }

    public BindingTemplate toUDDIBindingTemplate(Endpoint dragonEP) {
        BindingTemplate apiBinding = new BindingTemplate();
        apiBinding.setBindingKey(UDDIIdentifierGenerator.toUddiV2Id(dragonEP.getId()));
        apiBinding.setServiceKey(UDDIIdentifierGenerator.toUddiV2Id(dragonEP.getTechnicalService().getId()));

        AccessPoint accessPoint = new AccessPoint();
        accessPoint.setURLType(URLType.HTTP);
        accessPoint.setValue(dragonEP.getNetworkAddress());
        apiBinding.setAccessPoint(accessPoint);

        List<Description> descriptions = dragonEP.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v2.Description> apiDescriptions = apiBinding.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        // TODO apiBinding.setHostingRedirector(value);

        TModelInstanceDetails instanceDetails = new TModelInstanceDetails();
        apiBinding.setTModelInstanceDetails(instanceDetails);
        List<TModelInstanceInfo> instanceInfos = instanceDetails.getTModelInstanceInfo();
        // Add binding as technical info
        TModelInstanceInfo instanceInfo = new TModelInstanceInfo();
        instanceInfo.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(dragonEP.getBinding().getId()));
        Description description = new Description();
        description.setLangCode("en");
        description.setDescription("The wsdl:binding that this wsdl:port (endpoint) "
                + "implements. The instanceParms specifies the port (endpoint) local name.");
        instanceInfo.getDescription().add(toUDDIDescription(description));
        InstanceDetails instanceDetails2 = new InstanceDetails();
        instanceDetails2.setInstanceParms(dragonEP.getName());
        instanceInfo.setInstanceDetails(instanceDetails2);
        instanceInfos.add(instanceInfo);

        /* Add service interface as technical info
        TModelInstanceInfo instanceInfo2 = new TModelInstanceInfo();
        instanceInfo2.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(dragonEP.getBinding().getServiceInterface().getId()));
        Description description2 = new Description();
        description2.setLangCode("en");
        description2
                .setDescription("The wsdl:portType (interface) that this wsdl:port (endpoint) implements.");
        instanceInfo2.getDescription().add(toUDDIDescription(description2));
        instanceInfos.add(instanceInfo2);*/

        /*
         * Add Endpoint related Techservice Service Specs as technical info
         
        Set<ServiceSpecification> serviceSpecs = dragonEP.getTechnicalService()
                .getServiceSpecifications();
        if (serviceSpecs != null) {
            for (ServiceSpecification serviceSpecification : serviceSpecs) {
                TModelInstanceInfo instanceInfo3 = new TModelInstanceInfo();
                instanceInfo3.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(serviceSpecification.getId()));
                Description description3 = new Description();
                description3.setLangCode("en");
                description3
                        .setDescription("The global service wsdl spec tmodel.");
                instanceInfo3.getDescription().add(toUDDIDescription(description3));
                instanceInfos.add(instanceInfo3);
            }
        }*/

        /*
         * Add binding protocol as technical info TModelInstanceInfo
         * instanceInfo = new TModelInstanceInfo();
         * instanceInfo.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(dragonEP
         * .getBinding().getProtocol().getId()));
         * instanceInfos.add(instanceInfo);
         */

        return apiBinding;
    }

    public org.uddi.api_v2.TModel toUDDITModel(TModel dragonTModel) {
        org.uddi.api_v2.TModel apiTModel = new org.uddi.api_v2.TModel();
        apiTModel.setName(toUDDIName(dragonTModel.getName()));
        apiTModel.setCategoryBag(toUDDICategoryBag(dragonTModel.getCategoryBag()));
        // TODO apiTModel.setOperator(value);
        // TODO apiTModel.setAuthorizedName(value);
        apiTModel.setIdentifierBag(toUDDIIdentifierBag(dragonTModel.getIdentifierBag()));
        apiTModel.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(dragonTModel.getId()));
        List<Description> descriptions = dragonTModel.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v2.Description> apiDescriptions = apiTModel.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        List<SimpleFile> simpleFiles = null;
        // For Dragon Binding and ServiceInterface, uses linked ServiceSpec
        // overview doc (only one for the moment)
        if (dragonTModel instanceof ServiceInterface) {
            simpleFiles = ((ServiceInterface) dragonTModel).getServiceSpecs().iterator().next()
                    .getOverviewDocs();
        } else if (dragonTModel instanceof Binding) {
            simpleFiles = ((Binding) dragonTModel).getServiceSpecifications().iterator().next()
                    .getOverviewDocs();
        } else {
            // Else uses Overview docs directly
            simpleFiles = dragonTModel.getOverviewDocs();
        }
        if (simpleFiles != null && !simpleFiles.isEmpty()) {
            // Set the first doc as overview doc
            apiTModel.setOverviewDoc(toUDDIOverviewDoc(simpleFiles.get(0)));
        }
        return apiTModel;
    }

    private OverviewDoc toUDDIOverviewDoc(SimpleFile simpleFile) {
        OverviewDoc overviewDoc = null;
        if (simpleFile != null) {
            overviewDoc = new OverviewDoc();
            String overviewURL = "";
            try {
                overviewURL = DragonRepoFileLocator.getRepoFileURL(simpleFile.getFileType(),
                        simpleFile.getFileName(), simpleFile.getId(), request);
            } catch (IOException e) {
                logger.warn("Can't create document overview URL", e);
            }
            overviewDoc.setOverviewURL(overviewURL);
            // TODO overviewDoc.getDescription().add(o);
        }
        return overviewDoc;
    }

    private org.uddi.api_v2.Description toUDDIDescription(Description description) {
        org.uddi.api_v2.Description apiDescription = null;
        if (description != null) {
            apiDescription = new org.uddi.api_v2.Description();
            apiDescription.setLang(description.getLangCode());
            apiDescription.setValue(description.getDescription());
        }
        return apiDescription;
    }

    private IdentifierBag toUDDIIdentifierBag(List<KeyedReference> identifierBag) {
        IdentifierBag apiBag = null;
        if (identifierBag != null) {
            apiBag = new IdentifierBag();
            for (KeyedReference keyedReference : identifierBag) {
                apiBag.getKeyedReference().add(toUDDIKeyedReference(keyedReference));
            }
        }
        return apiBag;
    }

    private org.uddi.api_v2.KeyedReference toUDDIKeyedReference(KeyedReference keyedReference) {
        org.uddi.api_v2.KeyedReference apiKeyedReference = null;
        if (keyedReference != null) {
            apiKeyedReference = new org.uddi.api_v2.KeyedReference();
            apiKeyedReference.setKeyName(keyedReference.getKeyName());
            apiKeyedReference.setKeyValue(keyedReference.getKeyValue());
            if (keyedReference.getTmodel() != null) {
                apiKeyedReference.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(keyedReference.getTmodel().getId()));
            }
        }
        return apiKeyedReference;
    }

    private CategoryBag toUDDICategoryBag(
            org.ow2.dragon.persistence.bo.common.CategoryBag categoryBag) {
        CategoryBag apiBag = null;
        if (categoryBag != null) {
            apiBag = new CategoryBag();
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            if (keyedReferences != null) {
                for (KeyedReference keyedReference : keyedReferences) {
                    org.uddi.api_v2.KeyedReference apiKeyedReference = toUDDIKeyedReference(keyedReference);
                    apiBag.getKeyedReference().add(apiKeyedReference);
                }
            }
        }
        return apiBag;
    }

    public org.uddi.api_v2.Name toUDDIName(Name dragonName) {
        org.uddi.api_v2.Name apiName = null;
        if (dragonName != null) {
            apiName = new org.uddi.api_v2.Name();
            apiName.setLang(dragonName.getLangCode());
            apiName.setValue(dragonName.getName());
        }
        return apiName;
    }

    public void setRequest(HttpServletRequest request) {
        this.request = request;
    }

    public org.ow2.dragon.persistence.bo.common.CategoryBag toDragonCategoryBag(
            CategoryBag categoryBag) throws InvalidKeyPassedException {
        org.ow2.dragon.persistence.bo.common.CategoryBag dragonBag = null;
        if (categoryBag != null) {
            dragonBag = new org.ow2.dragon.persistence.bo.common.CategoryBag();
            List<org.uddi.api_v2.KeyedReference> content = categoryBag.getKeyedReference();
            if (content != null) {
                for (org.uddi.api_v2.KeyedReference element : content) {
                    dragonBag.addKeyedReference(toDragonKeyedReference(element));
                }
            }

        }
        return dragonBag;
    }

    private KeyedReference toDragonKeyedReference(org.uddi.api_v2.KeyedReference apiKR)
            throws InvalidKeyPassedException {
        KeyedReference keyedReference = null;
        if (apiKR != null) {
            keyedReference = new KeyedReference();
            keyedReference.setKeyName(apiKR.getKeyName());
            keyedReference.setKeyValue(apiKR.getKeyValue());
            if (apiKR.getTModelKey() != null) {
                TModel model = (TModel) universalUnifiedDAO.get(TModel.class, UDDIIdentifierGenerator.toUddiV3Id(apiKR.getTModelKey()));
                if (model != null) {
                    keyedReference.setTmodel(model);
                } else {
                    throw new InvalidKeyPassedException(new ErrorMessage(
                            "errors.invalidkey.TModelNotFound", apiKR.getTModelKey()));
                }
            }
        }
        return keyedReference;
    }

    public List<KeyedReference> toDragonIdentifierBag(IdentifierBag identifierBag)
            throws InvalidKeyPassedException {
        List<KeyedReference> dragonIdentifierBag = new ArrayList<KeyedReference>();
        if (identifierBag != null) {
            List<org.uddi.api_v2.KeyedReference> apiKRs = identifierBag.getKeyedReference();
            if (apiKRs != null) {
                for (org.uddi.api_v2.KeyedReference keyedReference : apiKRs) {
                    dragonIdentifierBag.add(toDragonKeyedReference(keyedReference));
                }
            }
        }
        return dragonIdentifierBag;
    }

    public Name toDragonName(org.uddi.api_v2.Name name) {
        Name dragonName = null;
        if (name != null) {
            dragonName = new Name();
            dragonName.setLangCode(name.getLang());
            dragonName.setName(name.getValue());
        }
        return dragonName;
    }

    public TModelInfos toUDDITModelInfos(List<TModel> tmodels) {
        TModelInfos result = null;
        if (tmodels != null && !tmodels.isEmpty()) {
            result = new TModelInfos();
            for (TModel tmodel : tmodels) {
                result.getTModelInfo().add(toUDDITModelInfo(tmodel));
            }
        }
        return result;
    }

    public TModelInfo toUDDITModelInfo(TModel tmodel) {
        TModelInfo result = null;
        if (tmodel != null) {
            result = new TModelInfo();
            result.setName(toUDDIName(tmodel.getName()));
            result.setTModelKey(UDDIIdentifierGenerator.toUddiV2Id(tmodel.getId()));
        }
        return result;
    }

    public List<Name> toDragonNames(List<org.uddi.api_v2.Name> name) {
        List<Name> result = new ArrayList<Name>();
        if (name != null) {
            for (org.uddi.api_v2.Name uddiName : name) {
                result.add(this.toDragonName(uddiName));
            }
        }
        return result;
    }

    public ServiceInfos toUDDIServiceInfos(List<TechnicalService> techServices) {
        ServiceInfos result = null;
        if (techServices != null && !techServices.isEmpty()) {
            result = new ServiceInfos();
            for (TechnicalService service : techServices) {
                result.getServiceInfo().add(toUDDIServiceInfo(service));
            }
        }
        return result;
    }

    public ServiceInfo toUDDIServiceInfo(TechnicalService service) {
        ServiceInfo result = null;
        if (service != null) {
            result = new ServiceInfo();
            result.setBusinessKey(UDDIIdentifierGenerator.toUddiV2Id(retrieveBusinessKeyFormTechServ(service)));
            result.setServiceKey(UDDIIdentifierGenerator.toUddiV2Id(service.getId()));
            List<Name> names = service.getNames();
            if (names != null) {
                List<org.uddi.api_v2.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
        }
        return result;
    }

    public RelatedBusinessInfos toUDDIRelatedBusinessInfos(List<OrganizationUnit> relatedOrgs) {
        RelatedBusinessInfos result = null;
        if (relatedOrgs != null && !relatedOrgs.isEmpty()) {
            result = new RelatedBusinessInfos();
            for (OrganizationUnit org : relatedOrgs) {
                result.getRelatedBusinessInfo().add(this.toUDDIRelatedBusinessInfo(org));
            }
        }
        return result;
    }

    public RelatedBusinessInfo toUDDIRelatedBusinessInfo(OrganizationUnit org) {
        RelatedBusinessInfo result = null;
        if (org != null) {
            result = new RelatedBusinessInfo();
            result.setBusinessKey(UDDIIdentifierGenerator.toUddiV2Id(org.getId()));
            List<Name> names = org.getNames();
            if (names != null) {
                List<org.uddi.api_v2.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
            List<Description> descriptions = org.getDescriptions();
            if (descriptions != null) {
                List<org.uddi.api_v2.Description> apiDescriptions = result.getDescription();
                for (Description description : descriptions) {
                    apiDescriptions.add(toUDDIDescription(description));
                }
            }

            // TODO add sharedRelationships

        }
        return result;
    }

    public BusinessInfos toUDDIBusinessInfos(List<OrganizationUnit> orgs) {
        BusinessInfos result = null;
        if (orgs != null && !orgs.isEmpty()) {
            result = new BusinessInfos();
            for (OrganizationUnit org : orgs) {
                result.getBusinessInfo().add(this.toUDDIBusinessInfo(org));
            }
        }
        return result;
    }

    private BusinessInfo toUDDIBusinessInfo(OrganizationUnit org) {
        BusinessInfo result = null;
        if (org != null) {
            result = new BusinessInfo();
            result.setBusinessKey(UDDIIdentifierGenerator.toUddiV2Id(org.getId()));
            List<Name> names = org.getNames();
            if (names != null) {
                List<org.uddi.api_v2.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
            List<Description> descriptions = org.getDescriptions();
            if (descriptions != null) {
                List<org.uddi.api_v2.Description> apiDescriptions = result.getDescription();
                for (Description description : descriptions) {
                    apiDescriptions.add(toUDDIDescription(description));
                }
            }
            result.setServiceInfos(retrieveLinkedBusinessServiceInfos(org));
        }
        return result;
    }

    private ServiceInfos retrieveLinkedBusinessServiceInfos(OrganizationUnit dragonBusiness) {
        ServiceInfos serviceInfos = new ServiceInfos();

        Set<Link> roles = dragonBusiness.getFromLinks();
        if (roles != null) {
            List<ServiceInfo> apiServices = serviceInfos.getServiceInfo();
            for (Link role : roles) {
                if (role.getTo() instanceof TechnicalService
                        && role.getType().equalsIgnoreCase(FindServiceQueryHelper.PROVIDER)) {
                    TechnicalService technicalService = (TechnicalService) role.getTo();
                    apiServices.add(toUDDIServiceInfo(technicalService));
                }
            }
        }

        return serviceInfos;
    }

}
