/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDIPublishImpl.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v3.impl;

import java.io.IOException;
import java.net.URI;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.apache.log4j.Logger;
import org.ow2.dragon.api.service.metadata.MetadataService;
import org.ow2.dragon.api.service.metadata.MetadataServiceException;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.CategoryValue;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.service.uddi.v3.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v3.error.FatalErrorException;
import org.ow2.dragon.service.uddi.v3.validator.PublicationValidator;
import org.ow2.dragon.util.FileReaderUtil;
import org.ow2.dragon.util.UDDIUseType;
import org.springframework.transaction.annotation.Transactional;
import org.uddi.api_v3.AddPublisherAssertions;
import org.uddi.api_v3.AssertionStatusReport;
import org.uddi.api_v3.BindingDetail;
import org.uddi.api_v3.BusinessDetail;
import org.uddi.api_v3.DeleteBinding;
import org.uddi.api_v3.DeleteBusiness;
import org.uddi.api_v3.DeletePublisherAssertions;
import org.uddi.api_v3.DeleteService;
import org.uddi.api_v3.DeleteTModel;
import org.uddi.api_v3.GetAssertionStatusReport;
import org.uddi.api_v3.GetPublisherAssertions;
import org.uddi.api_v3.GetRegisteredInfo;
import org.uddi.api_v3.OverviewDoc;
import org.uddi.api_v3.OverviewURL;
import org.uddi.api_v3.PublisherAssertions;
import org.uddi.api_v3.RegisteredInfo;
import org.uddi.api_v3.SaveBinding;
import org.uddi.api_v3.SaveBusiness;
import org.uddi.api_v3.SaveService;
import org.uddi.api_v3.SaveTModel;
import org.uddi.api_v3.ServiceDetail;
import org.uddi.api_v3.SetPublisherAssertions;
import org.uddi.api_v3.TModelDetail;
import org.uddi.api_v3_porttype.DispositionReportFaultMessage;
import org.uddi.api_v3_porttype.UDDIPublicationPortType;

/**
 * @author ofabre
 * 
 */
// @WebService(serviceName = "UDDIPublicationService", endpointInterface =
// "org.uddi.api_v3_porttype.UDDIPublicationPortType", targetNamespace =
// "urn:uddi-org:api_v3")
public class UDDIPublicationImpl implements UDDIPublicationPortType {

    private PublicationValidator validator;

    private UniversalUnifiedDAO universalUnifiedDAO;

    private UDDITransferObjectAssembler transferObjectAssembler;

    private Logger logger = Logger.getLogger(UDDIPublicationImpl.class);

    private MetadataService metadataService;

    public UDDIPublicationImpl(UniversalUnifiedDAO universalUnifiedDAO,
            MetadataService metadataService) {
        super();
        this.universalUnifiedDAO = universalUnifiedDAO;
        this.transferObjectAssembler = new UDDITransferObjectAssembler(universalUnifiedDAO);
        validator = new PublicationValidator();
        this.metadataService = metadataService;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#addPublisherAssertions
     * (org.uddi.api_v3.AddPublisherAssertions)
     */
    public void addPublisherAssertions(AddPublisherAssertions body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#deleteBinding(org.uddi
     * .api_v3.DeleteBinding)
     */
    public void deleteBinding(DeleteBinding body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#deleteBusiness(org.uddi
     * .api_v3.DeleteBusiness)
     */
    public void deleteBusiness(DeleteBusiness body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#deletePublisherAssertions
     * (org.uddi.api_v3.DeletePublisherAssertions)
     */
    public void deletePublisherAssertions(DeletePublisherAssertions body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#deleteService(org.uddi
     * .api_v3.DeleteService)
     */
    public void deleteService(DeleteService body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#deleteTModel(org.uddi
     * .api_v3.DeleteTModel)
     */
    public void deleteTModel(DeleteTModel body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));

    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#getAssertionStatusReport
     * (org.uddi.api_v3.GetAssertionStatusReport)
     */
    public AssertionStatusReport getAssertionStatusReport(GetAssertionStatusReport body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#getPublisherAssertions
     * (org.uddi.api_v3.GetPublisherAssertions)
     */
    public PublisherAssertions getPublisherAssertions(GetPublisherAssertions body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#getRegisteredInfo(org
     * .uddi.api_v3.GetRegisteredInfo)
     */
    public RegisteredInfo getRegisteredInfo(GetRegisteredInfo body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#saveBinding(org.uddi
     * .api_v3.SaveBinding)
     */
    public BindingDetail saveBinding(SaveBinding body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#saveBusiness(org.uddi
     * .api_v3.SaveBusiness)
     */
    public BusinessDetail saveBusiness(SaveBusiness body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#saveService(org.uddi
     * .api_v3.SaveService)
     */
    public ServiceDetail saveService(SaveService body) throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#saveTModel(org.uddi.
     * api_v3.SaveTModel)
     */
    @Transactional(rollbackFor = DispositionReportFaultMessage.class)
    public TModelDetail saveTModel(SaveTModel body) throws DispositionReportFaultMessage {
        // TODO Handle authentication

        // Validate query
        validator.validateSaveTModel(body);

        TModelDetail result = new TModelDetail();

        List<org.uddi.api_v3.TModel> apiTModelList = body.getTModel();

        for (org.uddi.api_v3.TModel apiTModel : apiTModelList) {

            TModel modelTModel = transferObjectAssembler.toDragonTModel(apiTModel);

            // TODO for the moment, we will only save new tmodels and skip
            // already existing ones
            if (modelTModel.getId() == null || !tModelExist(modelTModel.getId())) {
                universalUnifiedDAO.save(modelTModel);
                logger.debug("Register TModel with id: " + modelTModel.getId());

                // Add all categoryBags
                org.ow2.dragon.persistence.bo.common.CategoryBag categoryBag = transferObjectAssembler
                        .toDragonCategoryBag(apiTModel.getCategoryBag());
                if (categoryBag != null) {
                    modelTModel.setCategoryBag(categoryBag);
                }

                // Add all identifiers
                List<KeyedReference> identifierBag = transferObjectAssembler
                        .toDragonIdentifierBag(apiTModel.getIdentifierBag());
                if (identifierBag != null && !identifierBag.isEmpty()) {
                    modelTModel.setIdentifierBag(identifierBag);
                }

                processOverviewDocs(apiTModel, modelTModel);

                universalUnifiedDAO.save(modelTModel);
            }

        }

        return result;
    }

    private void processOverviewDocs(org.uddi.api_v3.TModel apiTModel, TModel modelTModel) {
        List<OverviewDoc> overviewDocs = apiTModel.getOverviewDoc();
        if (overviewDocs != null) {
            for (OverviewDoc overviewDoc : overviewDocs) {
                registerOverviewDoc(overviewDoc, modelTModel);
            }
        }
    }

    private void registerOverviewDoc(OverviewDoc overviewDoc, TModel modelTModel) {
        List<JAXBElement<?>> content = overviewDoc.getContent();
        if (content != null) {
            for (JAXBElement<?> element : content) {
                Object elementContent = element.getValue();
                // Use only overview url (and not description for the moment)
                if (elementContent instanceof OverviewURL) {
                    URI uri = URI.create(((OverviewURL) elementContent).getValue().trim());
                    UDDIUseType useType = UDDIUseType.fromString(((OverviewURL) elementContent)
                            .getUseType());
                    // TODO support only "valueSet" use types for the
                    // moment. Add "text"... use types support
                    if (UDDIUseType.VALUE_SET.equals(useType)) {
                        try {
                            SimpleFile registeredOverviewDoc = metadataService.storeMetadata(null,
                                    uri, null, useType);
                            modelTModel.addOverviewDoc(registeredOverviewDoc);
                            if (UDDIUseType.VALUE_SET.equals(useType)
                                    && modelTModel instanceof Category) {
                                // Extract values from valueSet file
                                extractCategoryTModelValues(modelTModel, registeredOverviewDoc);
                            }
                        } catch (MetadataServiceException e) {
                            logger.warn("Failed to register an overview Doc ('" + uri.toString()
                                    + "') for tmodel: " + modelTModel.getId(), e);
                        } catch (IOException e) {
                            logger.warn("Failed to extract values from a valueSet overview Doc ('"
                                    + uri.toString() + "') for tmodel: " + modelTModel.getId(), e);
                        }

                    }
                }
            }
        }
    }

    private void extractCategoryTModelValues(TModel modelTModel, SimpleFile registeredOverviewDoc)
            throws IOException, MetadataServiceException {
        List<String[]> lines = FileReaderUtil.readLines(metadataService
                .loadMetadataContentAsInputStream(registeredOverviewDoc.getId()), ";");
        if (lines != null) {
            for (String[] line : lines) {
                // skip line without description
                if (!(line.length < 2)) {
                    CategoryValue categoryValue = new CategoryValue(line[0], line[1]);
                    ((Category) modelTModel).addValue(categoryValue);
                }
            }
        }
    }

    private boolean tModelExist(String id) {
        return universalUnifiedDAO.get(TModel.class, id) != null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.uddi.api_v3_porttype.UDDIPublicationPortType#setPublisherAssertions
     * (org.uddi.api_v3.SetPublisherAssertions)
     */
    public PublisherAssertions setPublisherAssertions(SetPublisherAssertions body)
            throws DispositionReportFaultMessage {
        throw new FatalErrorException(new ErrorMessage("errors.Unsupported"));
    }

    public void setMetadataService(MetadataService metadataService) {
        this.metadataService = metadataService;
    }

}
