/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * UDDITransfertObjectAssembler.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v3.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.xml.bind.JAXBElement;

import org.apache.log4j.Logger;
import org.ow2.dragon.persistence.bo.common.Category;
import org.ow2.dragon.persistence.bo.common.Description;
import org.ow2.dragon.persistence.bo.common.Identifier;
import org.ow2.dragon.persistence.bo.common.KeyedReference;
import org.ow2.dragon.persistence.bo.common.KeyedReferenceGroup;
import org.ow2.dragon.persistence.bo.common.Link;
import org.ow2.dragon.persistence.bo.common.Name;
import org.ow2.dragon.persistence.bo.common.TModel;
import org.ow2.dragon.persistence.bo.deployment.Binding;
import org.ow2.dragon.persistence.bo.deployment.Endpoint;
import org.ow2.dragon.persistence.bo.deployment.Protocol;
import org.ow2.dragon.persistence.bo.deployment.TechnicalService;
import org.ow2.dragon.persistence.bo.deployment.Transport;
import org.ow2.dragon.persistence.bo.metadata.SimpleFile;
import org.ow2.dragon.persistence.bo.organization.Address;
import org.ow2.dragon.persistence.bo.organization.AddressLine;
import org.ow2.dragon.persistence.bo.organization.DiscoveryUrl;
import org.ow2.dragon.persistence.bo.organization.Email;
import org.ow2.dragon.persistence.bo.organization.OrganizationUnit;
import org.ow2.dragon.persistence.bo.organization.Party;
import org.ow2.dragon.persistence.bo.organization.Person;
import org.ow2.dragon.persistence.bo.organization.PhoneNumber;
import org.ow2.dragon.persistence.bo.organization.Post;
import org.ow2.dragon.persistence.bo.specification.ServiceInterface;
import org.ow2.dragon.persistence.bo.specification.ServiceSpecification;
import org.ow2.dragon.persistence.dao.UniversalUnifiedDAO;
import org.ow2.dragon.service.uddi.query.FindServiceQueryHelper;
import org.ow2.dragon.service.uddi.v3.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v3.error.InvalidKeyPassedException;
import org.ow2.dragon.util.DragonRepoFileLocator;
import org.ow2.dragon.util.TModelType;
import org.uddi.api_v3.AccessPoint;
import org.uddi.api_v3.BindingTemplate;
import org.uddi.api_v3.BindingTemplates;
import org.uddi.api_v3.BusinessEntity;
import org.uddi.api_v3.BusinessInfo;
import org.uddi.api_v3.BusinessInfos;
import org.uddi.api_v3.BusinessService;
import org.uddi.api_v3.BusinessServices;
import org.uddi.api_v3.CategoryBag;
import org.uddi.api_v3.Contact;
import org.uddi.api_v3.Contacts;
import org.uddi.api_v3.DiscoveryURL;
import org.uddi.api_v3.DiscoveryURLs;
import org.uddi.api_v3.IdentifierBag;
import org.uddi.api_v3.ListDescription;
import org.uddi.api_v3.ObjectFactory;
import org.uddi.api_v3.OverviewDoc;
import org.uddi.api_v3.OverviewURL;
import org.uddi.api_v3.PersonName;
import org.uddi.api_v3.Phone;
import org.uddi.api_v3.RelatedBusinessInfo;
import org.uddi.api_v3.RelatedBusinessInfos;
import org.uddi.api_v3.ServiceInfo;
import org.uddi.api_v3.ServiceInfos;
import org.uddi.api_v3.TModelInfo;
import org.uddi.api_v3.TModelInfos;
import org.uddi.api_v3.TModelInstanceDetails;
import org.uddi.api_v3.TModelInstanceInfo;

/**
 * @author ofabre - ebmwebsourcing
 * 
 */
public class UDDITransferObjectAssembler {

    private static final String ACCESSPOINT_USETYPE_EP = "endpoint";

    private Logger logger = Logger.getLogger(this.getClass());

    private HttpServletRequest request;

    private final static String CATEGORIZATION_KEY = "uddi:uddi.org:categorization:types";

    public UDDITransferObjectAssembler(UniversalUnifiedDAO universalUnifiedDAO) {
        super();
        this.universalUnifiedDAO = universalUnifiedDAO;
    }

    private UniversalUnifiedDAO universalUnifiedDAO;

    public BusinessEntity toUDDIBusiness(OrganizationUnit dragonBusiness) {
        BusinessEntity apiBusiness = new BusinessEntity();
        apiBusiness.setBusinessKey(dragonBusiness.getId());
        apiBusiness.setCategoryBag(toUDDICategoryBag(dragonBusiness.getCategoryBag()));
        apiBusiness.setIdentifierBag(toUDDIIdentifierBag(dragonBusiness.getIdentifierBag()));

        List<Name> names = dragonBusiness.getNames();
        if (names != null) {
            List<org.uddi.api_v3.Name> apiNames = apiBusiness.getName();
            for (Name name : names) {
                apiNames.add(toUDDIName(name));
            }
        }

        List<Description> descriptions = dragonBusiness.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v3.Description> apiDescriptions = apiBusiness.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        List<DiscoveryUrl> dragonDiscoveryUrls = dragonBusiness.getDiscoveryUrls();
        if (dragonDiscoveryUrls != null) {
            DiscoveryURLs apiDiscoveryURLs = new DiscoveryURLs();
            List<DiscoveryURL> apiDiscoveryURLList = apiDiscoveryURLs.getDiscoveryURL();
            for (DiscoveryUrl discoveryUrl : dragonDiscoveryUrls) {
                apiDiscoveryURLList.add(toUDDIDiscoveryUrl(discoveryUrl));
            }
            apiBusiness.setDiscoveryURLs(apiDiscoveryURLs);
        }

        apiBusiness.setContacts(retrieveOrganizationContacts(dragonBusiness));

        apiBusiness.setBusinessServices(retrieveLinkedBusinessServices(dragonBusiness));

        return apiBusiness;
    }

    private BusinessServices retrieveLinkedBusinessServices(OrganizationUnit dragonBusiness) {
        BusinessServices businessServices = new BusinessServices();

        Set<Link> roles = dragonBusiness.getFromLinks();
        if (roles != null) {
            List<BusinessService> apiServices = businessServices.getBusinessService();
            for (Link role : roles) {
                if (role.getTo() instanceof TechnicalService
                        && role.getType().equalsIgnoreCase(FindServiceQueryHelper.PROVIDER)) {
                    TechnicalService techServ = (TechnicalService) role.getTo();
                    apiServices.add(toUDDIService(techServ));
                }
            }
        }

        return businessServices;
    }

    /**
     * Retrieve a list of child parties that are instances of {@link Person} and
     * are marked as contact or have a post marked as contact <br/>
     * TODO if optimization is needed, could use a DAO search instead of a java
     * search
     * 
     * @param dragonBusiness
     * @return a {@link Contacts} element
     */
    private Contacts retrieveOrganizationContacts(OrganizationUnit dragonBusiness) {
        Contacts contacts = new Contacts();
        List<Contact> apiContacts = contacts.getContact();
        // Retrieve contact of this organization
        Set<Party> parties = dragonBusiness.getParties();
        if (parties != null) {
            for (Party party : parties) {
                if (party instanceof Person) {
                    Person person = (Person) party;
                    if (person.isContact()
                            || (person.getPost() != null && person.getPost().isContact())) {
                        apiContacts.add(toUDDIContact(person));
                    }
                }
            }
        }
        return contacts;
    }

    private Contact toUDDIContact(Person person) {
        Contact contact = new Contact();
        if (person.getPost() != null) {
            contact.setUseType(person.getPost().getNature());
        }

        // Add person names as personnames
        List<Name> names = person.getNames();
        List<PersonName> apiNames = contact.getPersonName();
        if (names != null) {
            for (Name name : names) {
                apiNames.add(toUDDIPersonName(name));
            }
        }
        // Role name as personname
        apiNames.add(toUDDIPersonName(person.getPost()));

        List<Description> descriptions = person.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v3.Description> apiDescriptions = contact.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
            // Add role description to contact description
            apiDescriptions.add(toUDDIDescription(person.getPost()));
        }

        List<PhoneNumber> dragonPhones = person.getPhoneNumbers();
        if (dragonPhones != null) {
            List<Phone> apiPhones = contact.getPhone();
            for (PhoneNumber phone : dragonPhones) {
                apiPhones.add(toUDDIPhone(phone));
            }
        }

        List<Email> dragonEmails = person.getEmailAddresses();
        if (dragonEmails != null) {
            List<org.uddi.api_v3.Email> apiEmails = contact.getEmail();
            for (Email email : dragonEmails) {
                apiEmails.add(toUDDIEmail(email));
            }
        }

        // Add person addresses to contact addresses
        List<Address> dragonAddresses = person.getAddresses();
        List<org.uddi.api_v3.Address> apiAddresses = contact.getAddress();
        if (dragonAddresses != null) {
            for (Address address : dragonAddresses) {
                apiAddresses.add(toUDDIAddress(address));
            }
        }
        // Add person parent org addresses to contact addresses
        if (person.getParentOrganization() != null) {
            List<Address> orgAddresses = person.getParentOrganization().getAddresses();
            if (orgAddresses != null) {
                for (Address address : orgAddresses) {
                    apiAddresses.add(toUDDIAddress(address));
                }
            }
        }

        return contact;
    }

    private org.uddi.api_v3.Description toUDDIDescription(Post post) {
        org.uddi.api_v3.Description apiDescription = null;
        if (post != null) {
            apiDescription = new org.uddi.api_v3.Description();
            apiDescription.setLang("EN-us");
            apiDescription.setValue(post.getDescription());
        }
        return apiDescription;
    }

    private org.uddi.api_v3.Address toUDDIAddress(Address address) {
        org.uddi.api_v3.Address apiAddress = null;
        if (address != null) {
            apiAddress = new org.uddi.api_v3.Address();
            apiAddress.setLang(address.getLangCode());
            apiAddress.setUseType(address.getUseType());
            if (address.getTModel() != null) {
                apiAddress.setTModelKey(address.getTModel().getId());
            }
            List<AddressLine> dragonLines = address.getAddressLines();
            if (dragonLines != null) {
                List<org.uddi.api_v3.AddressLine> apiLines = apiAddress.getAddressLine();
                for (AddressLine addressLine : dragonLines) {
                    apiLines.add(toUDDIAddressLine(addressLine));
                }
            }
        }
        return apiAddress;
    }

    private org.uddi.api_v3.AddressLine toUDDIAddressLine(AddressLine addressLine) {
        org.uddi.api_v3.AddressLine apiAddressLine = null;
        if (addressLine != null) {
            apiAddressLine = new org.uddi.api_v3.AddressLine();
            apiAddressLine.setKeyName(addressLine.getKeyName());
            apiAddressLine.setKeyValue(addressLine.getKeyValue());
            apiAddressLine.setValue(addressLine.getAddressLine());
        }
        return apiAddressLine;
    }

    private org.uddi.api_v3.Email toUDDIEmail(Email email) {
        org.uddi.api_v3.Email apiEmail = null;
        if (email != null) {
            apiEmail = new org.uddi.api_v3.Email();
            apiEmail.setUseType(email.getUseType());
            apiEmail.setValue(email.getEmail());
        }
        return apiEmail;
    }

    private Phone toUDDIPhone(PhoneNumber phone) {
        Phone apiPhone = null;
        if (phone != null) {
            apiPhone = new Phone();
            apiPhone.setUseType(phone.getUseType());
            apiPhone.setValue(phone.getPhoneNumber());
        }
        return apiPhone;
    }

    private PersonName toUDDIPersonName(Post post) {
        PersonName apiName = null;
        if (post != null) {
            apiName = new PersonName();
            apiName.setValue(post.getName());
        }
        return apiName;
    }

    private DiscoveryURL toUDDIDiscoveryUrl(DiscoveryUrl discoveryUrl) {
        DiscoveryURL apiUrl = null;
        if (discoveryUrl != null) {
            apiUrl = new DiscoveryURL();
            apiUrl.setUseType(discoveryUrl.getUseType());
            apiUrl.setValue(discoveryUrl.getUrl());
        }
        return apiUrl;
    }

    private DiscoveryUrl toDragonDiscoveryUrl(DiscoveryURL discoveryUrl) {
        DiscoveryUrl dragonUrl = null;
        if (discoveryUrl != null) {
            dragonUrl = new DiscoveryUrl();
            dragonUrl.setUseType(discoveryUrl.getUseType());
            dragonUrl.setUrl(discoveryUrl.getValue());
        }
        return dragonUrl;
    }

    public List<DiscoveryUrl> toDragonDiscoveryUrls(DiscoveryURLs discoveryURLs) {
        List<DiscoveryUrl> dragonDiscoveryUrls = new ArrayList<DiscoveryUrl>();
        if (discoveryURLs != null) {
            List<DiscoveryURL> apiDiscoveryURLs = discoveryURLs.getDiscoveryURL();
            if (apiDiscoveryURLs != null) {
                for (DiscoveryURL discoveryUrl : apiDiscoveryURLs) {
                    dragonDiscoveryUrls.add(toDragonDiscoveryUrl(discoveryUrl));
                }
            }
        }
        return dragonDiscoveryUrls;
    }

    public BusinessService toUDDIService(TechnicalService dragonService) {
        BusinessService apiService = new BusinessService();
        apiService.setServiceKey(dragonService.getId());

        List<Name> names = dragonService.getNames();
        if (names != null) {
            List<org.uddi.api_v3.Name> apiNames = apiService.getName();
            for (Name name : names) {
                apiNames.add(toUDDIName(name));
            }
        }

        List<Description> descriptions = dragonService.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v3.Description> apiDescriptions = apiService.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        apiService.setCategoryBag(toUDDICategoryBag(dragonService.getCategoryBag()));

        apiService.setBusinessKey(retrieveBusinessKeyFormTechServ(dragonService));

        Set<Endpoint> eps = dragonService.getEndpoints();
        if (eps != null) {
            BindingTemplates bindingTemplates = new BindingTemplates();
            List<BindingTemplate> apiBindings = bindingTemplates.getBindingTemplate();
            for (Endpoint ep : eps) {
                apiBindings.add(toUDDIBindingTemplate(ep));
            }
            apiService.setBindingTemplates(bindingTemplates);
        }

        return apiService;
    }

    /**
     * Try to identify an organization unit which is "Provider" of the service
     * and use the first one id as businessKey <br/>
     * TODO replace role strings by Constants or better by TModel describing
     * possible roles
     * 
     * @param dragonService
     * @return the businessKey
     */
    private String retrieveBusinessKeyFormTechServ(TechnicalService dragonService) {
        String bKey = null;
        Set<Link> roles = dragonService.getToLinks();
        if (roles != null) {
            for (Link partyToTechService : roles) {
                if (FindServiceQueryHelper.PROVIDER.equalsIgnoreCase(partyToTechService.getType())) {
                    bKey = partyToTechService.getFrom().getId();
                    break;
                }
            }
        }
        return bKey;
    }

    public BindingTemplate toUDDIBindingTemplate(Endpoint dragonEP) {
        BindingTemplate apiBinding = new BindingTemplate();
        apiBinding.setBindingKey(dragonEP.getId());
        apiBinding.setServiceKey(dragonEP.getTechnicalService().getId());

        AccessPoint accessPoint = new AccessPoint();
        accessPoint.setUseType(ACCESSPOINT_USETYPE_EP);
        accessPoint.setValue(dragonEP.getNetworkAddress());
        apiBinding.setAccessPoint(accessPoint);

        List<Description> descriptions = dragonEP.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v3.Description> apiDescriptions = apiBinding.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }

        apiBinding.setCategoryBag(toUDDICategoryBag(dragonEP.getCategoryBag()));

        TModelInstanceDetails instanceDetails = new TModelInstanceDetails();
        List<TModelInstanceInfo> instanceInfos = instanceDetails.getTModelInstanceInfo();
        // Add Endpoint related Techservice Service Specs as technical info
        Set<ServiceSpecification> serviceSpecs = dragonEP.getTechnicalService()
                .getServiceSpecifications();
        if (serviceSpecs != null) {
            for (ServiceSpecification serviceSpecification : serviceSpecs) {
                TModelInstanceInfo instanceInfo = new TModelInstanceInfo();
                instanceInfo.setTModelKey(serviceSpecification.getId());
                instanceInfos.add(instanceInfo);
            }
        }
        // Add service interface as technical info
        TModelInstanceInfo instanceInfo = new TModelInstanceInfo();
        instanceInfo.setTModelKey(dragonEP.getBinding().getId());
        Description description = new Description();
        description.setLangCode("en");
        description.setDescription("The wsdl:binding that this wsdl:port (endpoint) "
                + "implements.");
        instanceInfo.getDescription().add(toUDDIDescription(description));
        instanceInfos.add(instanceInfo);

        // Add binding as technical info
        TModelInstanceInfo instanceInfo2 = new TModelInstanceInfo();
        instanceInfo2.setTModelKey(dragonEP.getBinding().getServiceInterface().getId());
        Description description2 = new Description();
        description2.setLangCode("en");
        description2
                .setDescription("The wsdl:portType (interface) that this wsdl:port (endpoint) implements.");
        instanceInfo2.getDescription().add(toUDDIDescription(description2));
        instanceInfos.add(instanceInfo2);

        return apiBinding;
    }

    public org.uddi.api_v3.TModel toUDDITModel(TModel dragonTModel) {
        org.uddi.api_v3.TModel apiTModel = new org.uddi.api_v3.TModel();
        apiTModel.setName(toUDDIName(dragonTModel.getName()));
        apiTModel.setCategoryBag(toUDDICategoryBag(dragonTModel.getCategoryBag()));
        apiTModel.setDeleted(dragonTModel.isDeleted());
        apiTModel.setIdentifierBag(toUDDIIdentifierBag(dragonTModel.getIdentifierBag()));
        apiTModel.setTModelKey(dragonTModel.getId());
        List<Description> descriptions = dragonTModel.getDescriptions();
        if (descriptions != null) {
            List<org.uddi.api_v3.Description> apiDescriptions = apiTModel.getDescription();
            for (Description description : descriptions) {
                apiDescriptions.add(toUDDIDescription(description));
            }
        }
        List<SimpleFile> simpleFiles = dragonTModel.getOverviewDocs();
        // For Dragon Binding and ServiceInterface, add linked ServiceSpec (only
        // one for the moment) overview docs
        if (dragonTModel instanceof ServiceInterface) {
            List<ServiceSpecification> serviceSpecs = new ArrayList<ServiceSpecification>(
                    ((ServiceInterface) dragonTModel).getServiceSpecs());
            simpleFiles.addAll(serviceSpecs.get(0).getOverviewDocs());
        } else if (dragonTModel instanceof Binding) {
            List<ServiceSpecification> serviceSpecs = new ArrayList<ServiceSpecification>(
                    ((Binding) dragonTModel).getServiceSpecifications());
            simpleFiles.addAll(serviceSpecs.get(0).getOverviewDocs());
        }
        if (simpleFiles != null) {
            List<OverviewDoc> apiDocs = apiTModel.getOverviewDoc();
            for (SimpleFile simpleFile : simpleFiles) {
                apiDocs.add(toUDDIOverviewDoc(simpleFile));
            }
        }
        return apiTModel;
    }

    public TModel toDragonTModel(org.uddi.api_v3.TModel uddiTModel)
            throws InvalidKeyPassedException {
        TModel dragonTModel = null;
        TModelType tModelType = this.getTModelType(uddiTModel);
        if (tModelType != null) {
            switch (tModelType) {
                case CATEGORY:
                    dragonTModel = new Category();
                    break;
                case IDENTIFIER:
                    dragonTModel = new Identifier();
                    break;
                case PROTOCOL:
                    dragonTModel = new Protocol();
                    break;
                case TRANSPORT:
                    dragonTModel = new Transport();
                    break;
                default:
                    dragonTModel = new TModel();
                    break;
            }
        } else {
            dragonTModel = new TModel();
        }
        dragonTModel.setName(toDragonName(uddiTModel.getName()));
        // dragonTModel.setCategoryBag(toDragonCategoryBag(uddiTModel.getCategoryBag()));
        dragonTModel.setDeleted(uddiTModel.isDeleted());
        // dragonTModel.setIdentifierBag(toDragonIdentifierBag(uddiTModel.getIdentifierBag()));
        dragonTModel.setId(uddiTModel.getTModelKey());
        List<org.uddi.api_v3.Description> descriptions = uddiTModel.getDescription();
        if (descriptions != null) {
            List<Description> dragonDescriptions = dragonTModel.getDescriptions();
            for (org.uddi.api_v3.Description description : descriptions) {
                dragonDescriptions.add(toDragonDescription(description));
            }
        }
        return dragonTModel;
    }

    private TModelType getTModelType(org.uddi.api_v3.TModel uddiTModel) {
        TModelType tModelType = null;
        if (uddiTModel.getCategoryBag() != null) {
            List<JAXBElement<?>> content = uddiTModel.getCategoryBag().getContent();
            if (content != null) {
                for (JAXBElement<?> element : content) {
                    Object elementContent = element.getValue();
                    // Checks only on KeyedReference and not on
                    // KeyedReferenceGroup
                    if (elementContent instanceof org.uddi.api_v3.KeyedReference) {
                        org.uddi.api_v3.KeyedReference current = (org.uddi.api_v3.KeyedReference) elementContent;
                        if (CATEGORIZATION_KEY.equals(current.getTModelKey())) {
                            tModelType = TModelType.fromString(current.getKeyValue());
                            if (tModelType != null) {
                                break;
                            }
                        }
                    }
                }
            }
        }
        return tModelType;
    }

    public Description toDragonDescription(org.uddi.api_v3.Description description) {
        Description dragonDescription = null;
        if (description != null) {
            dragonDescription = new Description();
            dragonDescription.setLangCode(description.getLang());
            dragonDescription.setDescription(description.getValue());
        }
        return dragonDescription;
    }

    private OverviewDoc toUDDIOverviewDoc(SimpleFile simpleFile) {
        OverviewDoc overviewDoc = null;
        if (simpleFile != null) {
            overviewDoc = new OverviewDoc();
            String overviewURL = "";
            try {
                overviewURL = DragonRepoFileLocator.getRepoFileURL(simpleFile.getFileType(),
                        simpleFile.getFileName(), simpleFile.getId(), request);
            } catch (IOException e) {
                logger.warn("Can't create document overview URL", e);
            }
            OverviewURL apiOverviewUrl = new OverviewURL();
            apiOverviewUrl.setValue(overviewURL);
            apiOverviewUrl.setUseType(simpleFile.getFileType().toString());
            overviewDoc.getContent().add(new ObjectFactory().createOverviewURL(apiOverviewUrl));
        }
        return overviewDoc;
    }

    private org.uddi.api_v3.Description toUDDIDescription(Description description) {
        org.uddi.api_v3.Description apiDescription = null;
        if (description != null) {
            apiDescription = new org.uddi.api_v3.Description();
            apiDescription.setLang(description.getLangCode());
            apiDescription.setValue(description.getDescription());
        }
        return apiDescription;
    }

    private IdentifierBag toUDDIIdentifierBag(List<KeyedReference> identifierBag) {
        IdentifierBag apiBag = null;
        if (identifierBag != null) {
            apiBag = new IdentifierBag();
            for (KeyedReference keyedReference : identifierBag) {
                apiBag.getKeyedReference().add(toUDDIKeyedReference(keyedReference));
            }
        }
        return apiBag;
    }

    private org.uddi.api_v3.KeyedReference toUDDIKeyedReference(KeyedReference keyedReference) {
        org.uddi.api_v3.KeyedReference apiKeyedReference = null;
        if (keyedReference != null) {
            apiKeyedReference = new org.uddi.api_v3.KeyedReference();
            apiKeyedReference.setKeyName(keyedReference.getKeyName());
            apiKeyedReference.setKeyValue(keyedReference.getKeyValue());
            if (keyedReference.getTmodel() != null) {
                apiKeyedReference.setTModelKey(keyedReference.getTmodel().getId());
            }
        }
        return apiKeyedReference;
    }

    private CategoryBag toUDDICategoryBag(
            org.ow2.dragon.persistence.bo.common.CategoryBag categoryBag) {
        CategoryBag apiBag = null;
        if (categoryBag != null) {
            apiBag = new CategoryBag();
            List<KeyedReference> keyedReferences = categoryBag.getKeyedReferences();
            List<KeyedReferenceGroup> keyedReferenceGroups = categoryBag.getKeyedReferenceGroups();
            List<JAXBElement<?>> apiCategoryList = apiBag.getContent();
            if (keyedReferences != null) {
                for (KeyedReference keyedReference : keyedReferences) {
                    org.uddi.api_v3.KeyedReference apiKeyedReference = toUDDIKeyedReference(keyedReference);
                    apiCategoryList
                            .add(new ObjectFactory().createKeyedReference(apiKeyedReference));
                }
            }
            if (keyedReferenceGroups != null) {
                for (KeyedReferenceGroup keyedReferenceGroup : keyedReferenceGroups) {
                    org.uddi.api_v3.KeyedReferenceGroup apiKeyedReferenceGroup = toUDDIKeyedReferenceGroup(keyedReferenceGroup);
                    apiCategoryList.add(new ObjectFactory()
                            .createKeyedReferenceGroup(apiKeyedReferenceGroup));
                }
            }
        }
        return apiBag;
    }

    private org.uddi.api_v3.KeyedReferenceGroup toUDDIKeyedReferenceGroup(
            KeyedReferenceGroup keyedReferenceGroup) {
        org.uddi.api_v3.KeyedReferenceGroup apiKeyedReferenceGroup = null;
        if (keyedReferenceGroup != null) {
            apiKeyedReferenceGroup = new org.uddi.api_v3.KeyedReferenceGroup();
            apiKeyedReferenceGroup.setTModelKey(keyedReferenceGroup.getTmodel().getId());
            List<KeyedReference> keyedReferences = keyedReferenceGroup.getKeyedReferences();
            List<org.uddi.api_v3.KeyedReference> apiKeyedReferences = apiKeyedReferenceGroup
                    .getKeyedReference();
            if (keyedReferences != null) {
                for (KeyedReference keyedReference : keyedReferences) {
                    apiKeyedReferences.add(toUDDIKeyedReference(keyedReference));
                }
            }
        }
        return apiKeyedReferenceGroup;
    }

    public org.uddi.api_v3.Name toUDDIName(Name dragonName) {
        org.uddi.api_v3.Name apiName = null;
        if (dragonName != null) {
            apiName = new org.uddi.api_v3.Name();
            apiName.setLang(dragonName.getLangCode());
            apiName.setValue(dragonName.getName());
        }
        return apiName;
    }

    public PersonName toUDDIPersonName(Name dragonName) {
        PersonName apiName = null;
        if (dragonName != null) {
            apiName = new PersonName();
            apiName.setLang(dragonName.getLangCode());
            apiName.setValue(dragonName.getName());
        }
        return apiName;
    }

    public void setRequest(HttpServletRequest request) {
        this.request = request;
    }

    public org.ow2.dragon.persistence.bo.common.CategoryBag toDragonCategoryBag(
            CategoryBag categoryBag) throws InvalidKeyPassedException {
        org.ow2.dragon.persistence.bo.common.CategoryBag dragonBag = null;
        if (categoryBag != null) {
            dragonBag = new org.ow2.dragon.persistence.bo.common.CategoryBag();
            List<JAXBElement<?>> content = categoryBag.getContent();
            if (content != null) {
                for (JAXBElement<?> element : content) {
                    Object elementContent = element.getValue();
                    if (elementContent instanceof org.uddi.api_v3.KeyedReference) {
                        dragonBag
                                .addKeyedReference(toDragonKeyedReference((org.uddi.api_v3.KeyedReference) elementContent));
                    } else if (elementContent instanceof org.uddi.api_v3.KeyedReferenceGroup) {
                        dragonBag
                                .addKeyedReferenceGroup(toDragonKeyedReferenceGroup((org.uddi.api_v3.KeyedReferenceGroup) elementContent));
                    }
                }
            }

        }
        return dragonBag;
    }

    private KeyedReference toDragonKeyedReference(org.uddi.api_v3.KeyedReference apiKR)
            throws InvalidKeyPassedException {
        KeyedReference keyedReference = null;
        if (apiKR != null) {
            keyedReference = new KeyedReference();
            keyedReference.setKeyName(apiKR.getKeyName());
            keyedReference.setKeyValue(apiKR.getKeyValue());
            if (apiKR.getTModelKey() != null) {
                TModel model = (TModel) universalUnifiedDAO.get(TModel.class, apiKR.getTModelKey());
                if (model != null) {
                    keyedReference.setTmodel(model);
                } else {
                    throw new InvalidKeyPassedException(new ErrorMessage(
                            "errors.invalidkey.TModelNotFound", apiKR.getTModelKey()));
                }
            }
        }
        return keyedReference;
    }

    private KeyedReferenceGroup toDragonKeyedReferenceGroup(
            org.uddi.api_v3.KeyedReferenceGroup apiKRG) throws InvalidKeyPassedException {
        KeyedReferenceGroup keyedReferenceGroup = null;
        if (apiKRG != null) {
            keyedReferenceGroup = new KeyedReferenceGroup();
            if (apiKRG.getTModelKey() != null) {
                TModel model = (TModel) universalUnifiedDAO
                        .get(TModel.class, apiKRG.getTModelKey());
                if (model != null) {
                    keyedReferenceGroup.setTmodel(model);
                } else {
                    throw new InvalidKeyPassedException(new ErrorMessage(
                            "errors.invalidkey.TModelNotFound", apiKRG.getTModelKey()));
                }
            }
            List<org.uddi.api_v3.KeyedReference> keyedReferences = apiKRG.getKeyedReference();
            if (keyedReferences != null) {
                for (org.uddi.api_v3.KeyedReference keyedReference : keyedReferences) {
                    keyedReferenceGroup.addKeyedReference(toDragonKeyedReference(keyedReference));
                }
            }
        }
        return keyedReferenceGroup;
    }

    public List<KeyedReference> toDragonIdentifierBag(IdentifierBag identifierBag)
            throws InvalidKeyPassedException {
        List<KeyedReference> dragonIdentifierBag = new ArrayList<KeyedReference>();
        if (identifierBag != null) {
            List<org.uddi.api_v3.KeyedReference> apiKRs = identifierBag.getKeyedReference();
            if (apiKRs != null) {
                for (org.uddi.api_v3.KeyedReference keyedReference : apiKRs) {
                    dragonIdentifierBag.add(toDragonKeyedReference(keyedReference));
                }
            }
        }
        return dragonIdentifierBag;
    }

    public Name toDragonName(org.uddi.api_v3.Name name) {
        Name dragonName = null;
        if (name != null) {
            dragonName = new Name();
            dragonName.setLangCode(name.getLang());
            dragonName.setName(name.getValue());
        }
        return dragonName;
    }

    public TModelInfos toUDDITModelInfos(List<TModel> tmodels) {
        TModelInfos result = null;
        if (tmodels != null && !tmodels.isEmpty()) {
            result = new TModelInfos();
            for (TModel tmodel : tmodels) {
                result.getTModelInfo().add(toUDDITModelInfo(tmodel));
            }
        }
        return result;
    }

    public TModelInfo toUDDITModelInfo(TModel tmodel) {
        TModelInfo result = null;
        if (tmodel != null) {
            result = new TModelInfo();
            result.setName(toUDDIName(tmodel.getName()));
            result.setTModelKey(tmodel.getId());
            List<Description> descriptions = tmodel.getDescriptions();
            if (descriptions != null) {
                List<org.uddi.api_v3.Description> apiDescriptions = result.getDescription();
                for (Description description : descriptions) {
                    apiDescriptions.add(toUDDIDescription(description));
                }
            }
        }
        return result;
    }

    public ListDescription toUDDIListDescription(int actualCount, int includeCount, int head) {
        ListDescription description = new ListDescription();
        description.setActualCount(actualCount);
        description.setIncludeCount(includeCount);
        description.setListHead(head);
        return description;
    }

    public List<Name> toDragonNames(List<org.uddi.api_v3.Name> name) {
        List<Name> result = new ArrayList<Name>();
        if (name != null) {
            for (org.uddi.api_v3.Name uddiName : name) {
                result.add(this.toDragonName(uddiName));
            }
        }
        return result;
    }

    public ServiceInfos toUDDIServiceInfos(List<TechnicalService> techServices) {
        ServiceInfos result = null;
        if (techServices != null && !techServices.isEmpty()) {
            result = new ServiceInfos();
            for (TechnicalService service : techServices) {
                result.getServiceInfo().add(toUDDIServiceInfo(service));
            }
        }
        return result;
    }

    public ServiceInfo toUDDIServiceInfo(TechnicalService service) {
        ServiceInfo result = null;
        if (service != null) {
            result = new ServiceInfo();
            result.setBusinessKey(retrieveBusinessKeyFormTechServ(service));
            result.setServiceKey(service.getId());
            List<Name> names = service.getNames();
            if (names != null) {
                List<org.uddi.api_v3.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
        }
        return result;
    }

    public RelatedBusinessInfos toUDDIRelatedBusinessInfos(List<OrganizationUnit> relatedOrgs) {
        RelatedBusinessInfos result = null;
        if (relatedOrgs != null && !relatedOrgs.isEmpty()) {
            result = new RelatedBusinessInfos();
            for (OrganizationUnit org : relatedOrgs) {
                result.getRelatedBusinessInfo().add(this.toUDDIRelatedBusinessInfo(org));
            }
        }
        return result;
    }

    public RelatedBusinessInfo toUDDIRelatedBusinessInfo(OrganizationUnit org) {
        RelatedBusinessInfo result = null;
        if (org != null) {
            result = new RelatedBusinessInfo();
            result.setBusinessKey(org.getId());
            List<Name> names = org.getNames();
            if (names != null) {
                List<org.uddi.api_v3.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
            List<Description> descriptions = org.getDescriptions();
            if (descriptions != null) {
                List<org.uddi.api_v3.Description> apiDescriptions = result.getDescription();
                for (Description description : descriptions) {
                    apiDescriptions.add(toUDDIDescription(description));
                }
            }

            // TODO add sharedRelationships

        }
        return result;
    }

    public BusinessInfos toUDDIBusinessInfos(List<OrganizationUnit> orgs) {
        BusinessInfos result = null;
        if (orgs != null && !orgs.isEmpty()) {
            result = new BusinessInfos();
            for (OrganizationUnit org : orgs) {
                result.getBusinessInfo().add(this.toUDDIBusinessInfo(org));
            }
        }
        return result;
    }

    private BusinessInfo toUDDIBusinessInfo(OrganizationUnit org) {
        BusinessInfo result = null;
        if (org != null) {
            result = new BusinessInfo();
            result.setBusinessKey(org.getId());
            List<Name> names = org.getNames();
            if (names != null) {
                List<org.uddi.api_v3.Name> apiNames = result.getName();
                for (Name name : names) {
                    apiNames.add(toUDDIName(name));
                }
            }
            List<Description> descriptions = org.getDescriptions();
            if (descriptions != null) {
                List<org.uddi.api_v3.Description> apiDescriptions = result.getDescription();
                for (Description description : descriptions) {
                    apiDescriptions.add(toUDDIDescription(description));
                }
            }
            result.setServiceInfos(retrieveLinkedBusinessServiceInfos(org));
        }
        return result;
    }

    private ServiceInfos retrieveLinkedBusinessServiceInfos(OrganizationUnit dragonBusiness) {
        ServiceInfos serviceInfos = new ServiceInfos();

        Set<Link> roles = dragonBusiness.getFromLinks();
        if (roles != null) {
            List<ServiceInfo> apiServices = serviceInfos.getServiceInfo();
            for (Link role : roles) {
                if (role.getTo() instanceof TechnicalService
                        && role.getType().equalsIgnoreCase(FindServiceQueryHelper.PROVIDER)) {
                    TechnicalService technicalService = (TechnicalService) role.getTo();
                    apiServices.add(toUDDIServiceInfo(technicalService));
                }
            }
        }

        return serviceInfos;
    }

}
