/**
 * Dragon - SOA Governance Platform.
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -------------------------------------------------------------------------
 * InquiryValidator.java
 * -------------------------------------------------------------------------
 */

package org.ow2.dragon.service.uddi.v3.validator;

import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.ow2.dragon.service.uddi.v3.error.ErrorMessage;
import org.ow2.dragon.service.uddi.v3.error.FatalErrorException;
import org.ow2.dragon.service.uddi.v3.error.InvalidCombinationException;
import org.ow2.dragon.service.uddi.v3.error.InvalidKeyPassedException;
import org.ow2.dragon.service.uddi.v3.error.ValueNotAllowedException;
import org.ow2.dragon.service.uddi.v3.impl.FindQualifiersV3;
import org.uddi.api_v3.FindBinding;
import org.uddi.api_v3.FindBusiness;
import org.uddi.api_v3.FindRelatedBusinesses;
import org.uddi.api_v3.FindService;
import org.uddi.api_v3.FindTModel;
import org.uddi.api_v3.GetBindingDetail;
import org.uddi.api_v3.GetBusinessDetail;
import org.uddi.api_v3.GetServiceDetail;
import org.uddi.api_v3.GetTModelDetail;
import org.uddi.api_v3.KeyedReference;
import org.uddi.api_v3.ObjectFactory;
import org.uddi.api_v3.TModelBag;
import org.uddi.api_v3_porttype.DispositionReportFaultMessage;

/**
 * 
 * @author ofabre - ebmwebsourcing
 * 
 */
public class InquiryValidator {

    public void validateGetBusinessDetail(GetBusinessDetail body)
            throws DispositionReportFaultMessage {

        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        // No null or empty list
        List<String> entityKeyList = body.getBusinessKey();
        if (entityKeyList == null || entityKeyList.size() == 0)
            throw new InvalidKeyPassedException(new ErrorMessage("errors.invalidkey.NoKeys"));

        HashSet<String> dupCheck = new HashSet<String>();
        for (String entityKey : entityKeyList) {
            boolean inserted = dupCheck.add(entityKey);
            if (!inserted)
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.DuplicateKey", entityKey));
        }
    }

    public void validateGetServiceDetail(GetServiceDetail body)
            throws DispositionReportFaultMessage {

        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        // No null or empty list
        List<String> entityKeyList = body.getServiceKey();
        if (entityKeyList == null || entityKeyList.size() == 0)
            throw new InvalidKeyPassedException(new ErrorMessage("errors.invalidkey.NoKeys"));

        HashSet<String> dupCheck = new HashSet<String>();
        for (String entityKey : entityKeyList) {
            boolean inserted = dupCheck.add(entityKey);
            if (!inserted)
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.DuplicateKey", entityKey));
        }
    }

    public void validateGetBindingDetail(GetBindingDetail body)
            throws DispositionReportFaultMessage {

        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        // No null or empty list
        List<String> entityKeyList = body.getBindingKey();
        if (entityKeyList == null || entityKeyList.size() == 0)
            throw new InvalidKeyPassedException(new ErrorMessage("errors.invalidkey.NoKeys"));

        HashSet<String> dupCheck = new HashSet<String>();
        for (String entityKey : entityKeyList) {
            boolean inserted = dupCheck.add(entityKey);
            if (!inserted)
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.DuplicateKey", entityKey));
        }
    }

    public void validateGetTModelDetail(GetTModelDetail body) throws DispositionReportFaultMessage {

        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        // No null or empty list
        List<String> entityKeyList = body.getTModelKey();
        if (entityKeyList == null || entityKeyList.size() == 0)
            throw new InvalidKeyPassedException(new ErrorMessage("errors.invalidkey.NoKeys"));

        HashSet<String> dupCheck = new HashSet<String>();
        for (String entityKey : entityKeyList) {
            boolean inserted = dupCheck.add(entityKey);
            if (!inserted)
                throw new InvalidKeyPassedException(new ErrorMessage(
                        "errors.invalidkey.DuplicateKey", entityKey));
        }
    }

    public void validateFindBusiness(FindBusiness body) throws DispositionReportFaultMessage {
        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        if (body.getCategoryBag() == null && body.getFindTModel() == null
                && body.getTModelBag() == null && (body.getName() == null || body.getName().isEmpty())
                && body.getIdentifierBag() == null && body.getDiscoveryURLs() == null
                && body.getFindRelatedBusinesses() == null)
            throw new FatalErrorException(new ErrorMessage("errors.findbusiness.NoInput"));

        validateFindQualifiers(body.getFindQualifiers());
        validateTModelBag(body.getTModelBag());
        validateFindTModel(body.getFindTModel(), true);
        validateFindRelatedBusinesses(body.getFindRelatedBusinesses(), true);
        validateDiscoveryUrls(body.getDiscoveryURLs());
        validateIdentifierBag(body.getIdentifierBag());
        validateCategoryBag(body.getCategoryBag());

    }

    public void validateFindService(FindService body) throws DispositionReportFaultMessage {
        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        if (body.getCategoryBag() == null && body.getFindTModel() == null
                && body.getTModelBag() == null
                && (body.getName() == null || body.getName().isEmpty()))
            throw new FatalErrorException(new ErrorMessage("errors.findservice.NoInput"));

        validateFindQualifiers(body.getFindQualifiers());
        validateTModelBag(body.getTModelBag());
        validateFindTModel(body.getFindTModel(), true);
        validateCategoryBag(body.getCategoryBag());

    }

    public void validateFindBinding(FindBinding body) throws DispositionReportFaultMessage {
        // No null input
        if (body == null)
            throw new FatalErrorException(new ErrorMessage("errors.NullInput"));

        if (body.getCategoryBag() == null && body.getFindTModel() == null
                && body.getTModelBag() == null)
            throw new FatalErrorException(new ErrorMessage("errors.findbinding.NoInput"));

        validateFindQualifiers(body.getFindQualifiers());
        validateTModelBag(body.getTModelBag());
        validateFindTModel(body.getFindTModel(), true);
        validateCategoryBag(body.getCategoryBag());

    }

    public void validateFindTModel(FindTModel body, boolean nullAllowed)
            throws DispositionReportFaultMessage {
        if (body == null) {
            // When FindTModel objects are embedded in other find calls, null is
            // allowed.
            if (nullAllowed)
                return;
            else
                throw new FatalErrorException(new ErrorMessage("errors.NullInput"));
        }

        if (body.getCategoryBag() == null && body.getIdentifierBag() == null
                && body.getName() == null)
            throw new FatalErrorException(new ErrorMessage("errors.findtmodel.NoInput"));

        validateFindQualifiers(body.getFindQualifiers());
        validateIdentifierBag(body.getIdentifierBag());
        validateCategoryBag(body.getCategoryBag());
    }

    public void validateFindRelatedBusinesses(FindRelatedBusinesses body, boolean nullAllowed)
            throws DispositionReportFaultMessage {
        if (body == null) {
            // When FindRelatedBusinesses objects are embedded in other find
            // calls, null is allowed.
            if (nullAllowed)
                return;
            else
                throw new FatalErrorException(new ErrorMessage("errors.NullInput"));
        }

        if ((body.getBusinessKey() == null || body.getBusinessKey().length() == 0)
                && (body.getFromKey() == null || body.getFromKey().length() == 0)
                && (body.getToKey() == null || body.getToKey().length() == 0))
            throw new FatalErrorException(new ErrorMessage("errors.findrelatedbusiness.NoInput"));

        boolean businessKeyExists = false;
        boolean fromKeyExists = false;
        if (body.getBusinessKey() != null && body.getBusinessKey().length() > 0) {
            businessKeyExists = true;

        }
        if (body.getFromKey() != null && body.getFromKey().length() > 0) {
            fromKeyExists = true;
            if (businessKeyExists)
                throw new FatalErrorException(new ErrorMessage(
                        "errors.findrelatedbusiness.MultipleInput"));

        }
        if (body.getToKey() != null && body.getToKey().length() > 0) {
            if (businessKeyExists || fromKeyExists)
                throw new FatalErrorException(new ErrorMessage(
                        "errors.findrelatedbusiness.MultipleInput"));
        }

        KeyedReference keyedRef = body.getKeyedReference();
        if (keyedRef != null) {
            if (keyedRef.getTModelKey() == null || keyedRef.getTModelKey().length() == 0
                    || keyedRef.getKeyName() == null || keyedRef.getKeyName().length() == 0
                    || keyedRef.getKeyValue() == null || keyedRef.getKeyValue().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage(
                        "errors.findrelatedbusiness.BlankKeyedRef"));
        }

    }

    public void validateTModelBag(TModelBag tmodelBag) throws DispositionReportFaultMessage {
        // tmodelBag is optional
        if (tmodelBag == null)
            return;

        if (tmodelBag.getTModelKey() == null || tmodelBag.getTModelKey().size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.tmodelbag.NoInput"));

    }

    public void validateDiscoveryUrls(org.uddi.api_v3.DiscoveryURLs discUrls)
            throws DispositionReportFaultMessage {
        // Discovery Urls is optional
        if (discUrls == null)
            return;

        // If discUrls does exist, it must have at least one element
        List<org.uddi.api_v3.DiscoveryURL> discUrlList = discUrls.getDiscoveryURL();
        if (discUrlList == null || discUrlList.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.discurls.NoInput"));
    }

    public void validateCategoryBag(org.uddi.api_v3.CategoryBag categories)
            throws DispositionReportFaultMessage {

        // Category bag is optional
        if (categories == null)
            return;

        // If category bag does exist, it must have at least one element
        List<JAXBElement<?>> elems = categories.getContent();
        if (elems == null || elems.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.categorybag.NoInput"));

        for (JAXBElement<?> elem : elems) {
            validateKeyedReferenceTypes(elem);
        }
    }

    public void validateIdentifierBag(org.uddi.api_v3.IdentifierBag identifiers)
            throws DispositionReportFaultMessage {

        // Identifier bag is optional
        if (identifiers == null)
            return;

        // If identifier bag does exist, it must have at least one element
        List<org.uddi.api_v3.KeyedReference> keyedRefList = identifiers.getKeyedReference();
        if (keyedRefList == null || keyedRefList.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.identifierbag.NoInput"));

        for (org.uddi.api_v3.KeyedReference keyedRef : keyedRefList) {
            validateKeyedReferenceTypes(new ObjectFactory().createKeyedReference(keyedRef));
        }
    }

    public void validateKeyedReferenceTypes(JAXBElement<?> elem)
            throws DispositionReportFaultMessage {
        if (elem == null || elem.getValue() == null)
            throw new ValueNotAllowedException(new ErrorMessage("errors.keyedreference.NullInput"));

        // Keyed reference groups must contain a tModelKey
        if (elem.getValue() instanceof org.uddi.api_v3.KeyedReferenceGroup) {
            org.uddi.api_v3.KeyedReferenceGroup krg = (org.uddi.api_v3.KeyedReferenceGroup) elem
                    .getValue();
            if (krg.getTModelKey() == null || krg.getTModelKey().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage(
                        "errors.keyedreference.NoTModelKey"));
        }
        // Keyed references must contain a tModelKey and keyValue
        else if (elem.getValue() instanceof org.uddi.api_v3.KeyedReference) {
            org.uddi.api_v3.KeyedReference kr = (org.uddi.api_v3.KeyedReference) elem.getValue();
            if (kr.getTModelKey() == null || kr.getTModelKey().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage(
                        "errors.keyedreference.NoTModelKey"));

            if (kr.getKeyValue() == null || kr.getKeyValue().length() == 0)
                throw new ValueNotAllowedException(new ErrorMessage(
                        "errors.keyedreference.NoKeyValue"));

        }
    }

    private void validateFindQualifiers(org.uddi.api_v3.FindQualifiers findQualifiers)
            throws DispositionReportFaultMessage {
        if (findQualifiers == null)
            return;

        List<String> fqList = findQualifiers.getFindQualifier();
        if (fqList == null || fqList.size() == 0)
            throw new ValueNotAllowedException(new ErrorMessage("errors.findqualifiers.NoInput"));

        Hashtable<String, String> fqTable = new Hashtable<String, String>();
        for (String fq : fqList) {
            String result = fqTable.put(fq.toUpperCase(), fq.toUpperCase());
            if (result != null)
                throw new ValueNotAllowedException(new ErrorMessage(
                        "errors.findqualifiers.DuplicateValue", result));

            // Invalid combo: andAllKeys, orAllKeys, and orLikeKeys
            if (fq.equalsIgnoreCase(FindQualifiersV3.AND_ALL_KEYS)
                    || fq.equalsIgnoreCase(FindQualifiersV3.AND_ALL_KEYS_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.OR_ALL_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.OR_ALL_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.OR_ALL_KEYS));

                if (fqTable.get(FindQualifiersV3.OR_LIKE_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.OR_LIKE_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.OR_LIKE_KEYS));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.OR_ALL_KEYS)
                    || fq.equalsIgnoreCase(FindQualifiersV3.OR_ALL_KEYS_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.AND_ALL_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.AND_ALL_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.AND_ALL_KEYS));

                if (fqTable.get(FindQualifiersV3.OR_LIKE_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.OR_LIKE_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.OR_LIKE_KEYS));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.OR_LIKE_KEYS)
                    || fq.equalsIgnoreCase(FindQualifiersV3.OR_LIKE_KEYS_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.AND_ALL_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.AND_ALL_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.AND_ALL_KEYS));

                if (fqTable.get(FindQualifiersV3.OR_ALL_KEYS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.OR_ALL_KEYS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.OR_ALL_KEYS));
            }

            // Invalid combo: sortByNameAsc and sortByNameDesc
            if (fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_NAME_ASC)
                    || fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_NAME_ASC_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SORT_BY_NAME_DESC.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SORT_BY_NAME_DESC_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SORT_BY_NAME_DESC));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_NAME_DESC)
                    || fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_NAME_DESC_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SORT_BY_NAME_ASC.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SORT_BY_NAME_ASC_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SORT_BY_NAME_ASC));
            }

            // Invalid combo: sortByDateAsc and sortByDateDesc
            if (fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_DATE_ASC)
                    || fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_DATE_ASC_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SORT_BY_DATE_DESC.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SORT_BY_DATE_DESC_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SORT_BY_DATE_DESC));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_DATE_DESC)
                    || fq.equalsIgnoreCase(FindQualifiersV3.SORT_BY_DATE_DESC_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SORT_BY_DATE_ASC.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SORT_BY_DATE_ASC_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SORT_BY_DATE_ASC));
            }

            // Invalid combo: combineCategoryBags, serviceSubset and
            // bindingSubset
            if (fq.equalsIgnoreCase(FindQualifiersV3.COMBINE_CATEGORY_BAGS)
                    || fq.equalsIgnoreCase(FindQualifiersV3.COMBINE_CATEGORY_BAGS_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SERVICE_SUBSET.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SERVICE_SUBSET_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SERVICE_SUBSET));

                if (fqTable.get(FindQualifiersV3.BINDING_SUBSET.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.BINDING_SUBSET_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.BINDING_SUBSET));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.SERVICE_SUBSET)
                    || fq.equalsIgnoreCase(FindQualifiersV3.SERVICE_SUBSET_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.COMBINE_CATEGORY_BAGS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.COMBINE_CATEGORY_BAGS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.COMBINE_CATEGORY_BAGS));

                if (fqTable.get(FindQualifiersV3.BINDING_SUBSET.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.BINDING_SUBSET_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.BINDING_SUBSET));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.BINDING_SUBSET)
                    || fq.equalsIgnoreCase(FindQualifiersV3.BINDING_SUBSET_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.SERVICE_SUBSET.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.SERVICE_SUBSET_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.SERVICE_SUBSET));

                if (fqTable.get(FindQualifiersV3.COMBINE_CATEGORY_BAGS.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.COMBINE_CATEGORY_BAGS_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.COMBINE_CATEGORY_BAGS));
            }

            // Invalid combo: exactMatch and approximateMatch
            if (fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.APPROXIMATE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.APPROXIMATE_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.APPROXIMATE_MATCH));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.APPROXIMATE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.APPROXIMATE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.EXACT_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.EXACT_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.EXACT_MATCH));
            }

            // Invalid combo: exactMatch and caseInsensitiveMatch
            if (fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.CASE_INSENSITIVE_MATCH));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.EXACT_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.EXACT_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.EXACT_MATCH));
            }

            // Invalid combo: binarySort and UTS-10
            if (fq.equalsIgnoreCase(FindQualifiersV3.BINARY_SORT)
                    || fq.equalsIgnoreCase(FindQualifiersV3.BINARY_SORT_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.UTS_10.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.UTS_10_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.UTS_10));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.UTS_10)
                    || fq.equalsIgnoreCase(FindQualifiersV3.UTS_10_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.BINARY_SORT.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.BINARY_SORT_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.BINARY_SORT));
            }

            // Invalid combo: diacriticSensitiveMatch and
            // diacriticInsensitiveMatch
            if (fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_SENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_SENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH_TMODEL
                                .toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.DIACRITIC_SENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.DIACRITIC_SENSITIVE_MATCH_TMODEL
                                .toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.DIACRITIC_SENSITIVE_MATCH));
            }

            // Invalid combo: exactMatch and diacriticInsensitiveMatch
            if (fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.EXACT_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH_TMODEL
                                .toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.DIACRITIC_INSENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.EXACT_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.EXACT_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.EXACT_MATCH));
            }

            // Invalid combo: caseSensitiveSort and caseInsensitiveSort
            if (fq.equalsIgnoreCase(FindQualifiersV3.CASE_SENSITIVE_SORT)
                    || fq.equalsIgnoreCase(FindQualifiersV3.CASE_SENSITIVE_SORT_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_SORT.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_SORT_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.CASE_INSENSITIVE_SORT));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_SORT)
                    || fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_SORT_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.CASE_SENSITIVE_SORT.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.CASE_SENSITIVE_SORT_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.CASE_SENSITIVE_SORT));
            }

            // Invalid combo: caseSensitiveMatch and caseInsensitiveMatch
            if (fq.equalsIgnoreCase(FindQualifiersV3.CASE_SENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.CASE_SENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.CASE_INSENSITIVE_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.CASE_INSENSITIVE_MATCH));
            } else if (fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_MATCH)
                    || fq.equalsIgnoreCase(FindQualifiersV3.CASE_INSENSITIVE_MATCH_TMODEL)) {
                if (fqTable.get(FindQualifiersV3.CASE_SENSITIVE_MATCH.toUpperCase()) != null
                        || fqTable.get(FindQualifiersV3.CASE_SENSITIVE_MATCH_TMODEL.toUpperCase()) != null)
                    throw new InvalidCombinationException(new ErrorMessage(
                            "errors.findqualifiers.InvalidCombo", fq + " & "
                                    + FindQualifiersV3.CASE_SENSITIVE_MATCH));
            }

        }
    }
}
